# coding: utf-8

import copy
import re
import html.entities
import itertools
import json as json
from cgi import escape as text_escape

from django.core.urlresolvers import reverse
from django.core.exceptions import PermissionDenied

from at.api.yaru import activity
from at.api.yaru import atom, atomgen
from at.api.yaru import urlgen
from at.api.yaru.errors import InvalidInputError, EntryNotFound, FixMeError
from at.api.yaru.utils.encoding import force_unicode
from at.api.yaru.utils.py2xml import etree2py
from at.api.yaru.utils.links import Link
from at.api.yaru.templates import AtomFeed, AtomEntry, PostMetaSimple, PostMetaPhoto, KeyValue
from at.api.yaru.instances import Blog, FeedsCache
from at.api.yaru.post import meta_utils
from at.api.yaru import datacore

XHTML_DIV = '{%s}div' % atom.xhtml_ns

entity_re = re.compile(r'&(#?[A-Za-z0-9]+?);')

def html_unescape(data):
    def fixup(m):
        """Взято с http://www.w3.org/QA/2008/04/unescape-html-entities-python.html"""
        text = m.group(0)
        if text[:2] == "&#":
            # character reference
            try:
                if text[:3] == "&#x":
                    return chr(int(text[3:-1], 16))
                else:
                    return chr(int(text[2:-1]))
            except ValueError:
                pass
        else:
            # named entity
            try:
                if text[1:-1] == "amp":
                    text = "&"
                elif text[1:-1] == "gt":
                    text = ">"
                elif text[1:-1] == "lt":
                    text = "<"
                else:
                    text = chr(html.entities.name2codepoint[text[1:-1]])
            except KeyError:
                pass
        return text # leave as is
    return entity_re.sub(fixup, data)

def _element_content_to_string(elem, strip_ns=False):
    # Нужно получить содержимое ElementTree.Element'а в виде текста в unicode.
    # Открывающий-закрывающий теги отрываются варварски, но по-другому сложнее.
    # И мы не пытаемся удалять leading и trailing whitespaces.
    text = atom.etree.tostring(elem, encoding='utf-8').decode('utf-8')
    if strip_ns:
        tag = elem.tag[elem.tag.index() + 1:]
        tag_len = len(tag)
    else:
        tag_len = len(elem.tag)
    text = text[(text.find('>')+1):-(tag_len+3)]
    return text

def parse_content_from_xml(content, strip_ns=True):
    content_type = content.attrib.get('type', 'text/wiki')
    if content_type in ['text', 'text/plain', 'text/wiki']:
        if len(content) == 0:
            content_text = content.text or ''
        else:
            raise InvalidInputError('the content of atom:content type=%s MUST NOT contain child elements' % content_type)
    elif content_type in ['html', 'text/html']:
        content_text = html_unescape(_element_content_to_string(
            content, strip_ns=strip_ns))
    elif content_type == 'xhtml':
        if len(content) == 1 and content[0].tag == XHTML_DIV:
            div = copy.deepcopy(content[0])
            # Now remove the namespaces:
            for el in div.getiterator():
                if el.tag.startswith('{'):
                    el.tag = el.tag.split('}', 1)[1]
            div.tag = 'div'
            content_text = atom.etree.tostring(div, pretty_print=False, encoding='utf-8')
        else:
            raise InvalidInputError('the content of atom:content type=xhtml MUST be a single XHTML div element')
    else:
        raise NotImplementedError('handling of content types other then text, html, xhtml not implemented')
    return (content_type, content_text)

def content_from_yaru_to_atom(content):
    content_type = content.attrib.get('type', 'text/html')
    def text_content(content):
        return text_escape(content.text or '')
    def html_content(content):
        # html в ярушной записи хранится в виде дерева,
        # в atom же ветку надо отдавать в виде текста
        # escaping делается самим lxml
        # print atom.tostring(content)
        return _element_content_to_string(content)
        #return atomgen.E('content', text_escape(content.text or ''), type='html')
    def xhtml_content(content):
        # FIXME адаптируй этот код под elementflow
        div = copy.deepcopy(content[0])
        # Now add the namespaces:
        for el in div.getiterator():
            if el.tag[0] != '{':
                el.tag = '{%s}%s' % (atom.xhtml_ns, el.tag)
        #div.attrib['xmlns'] = atom.xhtml_ns
        return div
    if len(content) == 0:
        f = (content_type == 'text/plain' and text_content or html_content)
    elif len(content) == 1 and content[0].tag == XHTML_DIV:
        f = xhtml_content
    else:
        f = html_content
    content = f(content)
    return content_type, content


class PostAtomStore(object):
    """Atom store записей одного feed'а с точки зрения заданного пользователя."""

    def __init__(self, ai, feed, abs_url=lambda url: url):
        self.abs_url = abs_url
        self.feed = feed
        self.ai = ai
        self._cached_item = None

    def _value_id(self, item):
        #FIXME: может валиться на преобразовании id в строку,
        # если id не правильный -- не число, не ascii
        # так как id может приходит от клиента, нужно проверять и защищаться

        # Experimental URN (http://www.ietf.org/rfc/rfc2141.txt)
        #return 'urn:X-ya-ru:%s,%s' % (self.feed_id, id)
        # tag URI (http://www.faqs.org/rfcs/rfc4151.html)
        return 'urn:ya.ru:post/%s/%s' % (item.posted_in, item.id)

    def _read_and_cache_result(self, id):
        self._cached_item = activity.retrieve_post(self.ai,
                                                   self.feed.get_id(), id)
        return self._cached_item

    def _read_using_cache(self, id):
        if self._cached_item and self._cached_item.id == id:
            result = self._cached_item
        else:
            result = activity.retrieve_post(self.ai, self.feed.get_id(),
                                            id)
            #TODO: проверять, что нам вернули запись действительно ту и того человека
        self._cached_item = None
        return result

    def _sanitize_post_entry(self, item, add_author=True, parse_authors=False, feeds_cache=None):
        feeds_cache = feeds_cache or FeedsCache(abs_url=self.abs_url)
        posted_in_feed = feeds_cache.get(item.posted_in)
        entry = {'id': self._value_id(item),
                 'post_title': item.get('title', ''),
                 'post_updated': item.updated,
                 'post_edited': item.edited,
                 'post_published': item.published,
                 'posted_in': item.posted_in,
                 'access': force_unicode(item.access_type, strings_only=False),
                 'comment_count': item.get('comment_count', 0),
                 'comments_disabled': bool(int(
                     item.get("comments_disabled", 0))),
                 'likes': item.get('likes')
                 }

        author_feed = self.feed
        if add_author:
            if self.feed.get_type() == 'club' or parse_authors:
                author_feed = feeds_cache.get(item.author_uid)
            entry['author'] = author_feed
            # FIXME удалить author_* и использовать только author
            entry['author_name'] = str(author_feed.get_name())
            entry['author_uri'] = author_feed.get_profile_link(author_feed.get_login())
            entry['author_self'] = author_feed.get_self_link()
        entry['links'] = posted_in_feed.get_post_relations(item.id)
        categories = [{'scheme': 'urn:ya.ru:posttypes', 'term': item.filter_type}]
        if item['type_as_digit'] == '21':
            entry['on_premoderation'] = True
        for id, tag in item.tags:
            categories.append({'scheme': urlgen.tag_space_url(posted_in_feed, abs_url=self.abs_url), 'term': tag, 'id': id})
        entry['categories'] = categories
        entry['display'] = item.get('display', None)
        entry['agent'] = item.get('agent', None, )
        if item.get('raw_meta') is not None and item['meta_type']:
            if item.meta_type in ['friend', 'unfriend']:
                friendee_uid = item.raw_meta.xpath('friendee/uid/text()|friendee/text()')
                friendee = feeds_cache.get(friendee_uid[0])
                item.meta = PostMetaSimple([friendee])
            elif item.meta_type in ['join', 'unjoin']:
                friendee_uid = item.raw_meta.xpath('friendee/uid/text()|friendee/text()')
                club = feeds_cache.get(friendee_uid[0])
                item.meta = PostMetaSimple([club])
            elif item.meta_type == 'link':
                # А здесь мета уже в нужном виде, содержит просто URL и всё.
                url = item.raw_meta.xpath('URL/text()')
                if url:
                    text = url[0]
                else:
                    text = ''
                item.meta = PostMetaSimple([Link('url', text=text)])
            elif item.meta_type == 'congratulation':
                # NOTE есть ещё who/uid/text(), но при наличии автора поста не понятно
                # зачем он нужен.
                # whom - кого поздравили, who - кто поздравил
                whom_uid = item.raw_meta.xpath('whom/uid/text()')
                event = item.raw_meta.xpath('event/text()')
                meta_data = []
                if whom_uid:
                    whom = feeds_cache.get(whom_uid[0])
                    meta_data.append(whom)
                meta_data.append(Link('event', text=event[0] if event else ''))
                item.meta = PostMetaSimple(meta_data)
            elif item.meta_type == 'video':
                video_login = item.raw_meta.xpath('video-login/text()')
                video_storage = item.raw_meta.xpath('video-storage/text()')
                if video_login and video_storage:
                    meta_data = [Link('link', attrs={'href': urlgen.video_url(video_login[0], video_storage[0]),
                                                     'rel': 'video',
                                                     })]
                    item.meta = PostMetaSimple(meta_data)
            elif item.meta_type == 'userpic':
                old_upic_id = item.raw_meta.xpath('old/text()')
                new_upic_id = item.raw_meta.xpath('new/text()')
                old_upic_id = old_upic_id[0] if old_upic_id else None
                new_upic_id = new_upic_id[0] if new_upic_id else None
                meta_data = []
                if old_upic_id:
                    meta_data.append(Link('link', attrs={'href': urlgen.verbose_userpic_url(author_feed.get_id(),
                                                                                            old_upic_id),
                                                         'rel': 'old_userpic',
                                                         }))
                if new_upic_id:
                    meta_data.append(Link('link', attrs={'href': urlgen.verbose_userpic_url(author_feed.get_id(),
                                                                                            new_upic_id),
                                                         'rel': 'userpic',
                                                         }))
                item.meta = PostMetaSimple(meta_data)
            elif item.meta_type == 'rename':
                old_name = item.raw_meta.xpath('old/text()')
                new_name = item.raw_meta.xpath('new/text()')
                old_name = old_name[0] if old_name else None
                new_name = new_name[0] if new_name else None
                meta_data = []
                if old_name:
                    meta_data.append(KeyValue('old', old_name))
                if new_name:
                    meta_data.append(KeyValue('new', new_name))
                item.meta = PostMetaSimple(meta_data)
            elif item.meta_type == 'photo':
                images = []
                for image in item.raw_meta.xpath('images/image'):
                    image_entry = {}
                    if image.attrib.get('external') == 'true':
                        image_entry = {'url': image.attrib['page_link'],
                                       'person': author_feed,
                                       }
                        media = {'M': {'width': image.attrib['width'],
                                       'height': image.attrib['height'],
                                       'src': image.attrib['src'],
                                       }
                                 }
                        image_entry['media'] = media
                    else:
                        base_image_url = image.attrib['url']
                        author_login = image.attrib['login']
                        # Иногда uid может и не быть в выдаче
                        # FIXME можно соптимизировать feeds_cache, чтобы работал как по uid так и по login
                        if 'uid' in image.attrib:
                            author_uid = image.attrib['uid']
                        else:
                            # Лишний запрос в корбу get_blog_id
                            author_blog = Blog(author_login)
                            author_uid = author_blog.get_id()
                        image_entry['person'] = feeds_cache.get(author_uid)
                        image_entry['url'] = urlgen.fotki_photo_url(author_login, image.attrib['id'])
                        media = {}
                        sizes = image.xpath('following-sibling::sizes')
                        if sizes:
                            for size in sizes[0]:
                                attrib = size.attrib
                                if 'url' not in attrib:
                                    attrib['url'] = '%s%s' % (base_image_url, attrib['name'])
                                    media[attrib['name']] = {'width': attrib['width'],
                                                             'height': attrib['height'],
                                                             'src': attrib['url'],
                                                             }
                        image_entry['media'] = media
                    images.append(image_entry)
                item.meta = PostMetaPhoto(images)
        entry['meta'] = item.get('meta')
        entry['meta_type'] = item['meta_type']
        entry['original'] = item.get('imported_from', None)
        if 'trackback_item_no' in item and 'trackback_uid' in item:
            post_relations = feeds_cache.get(item.trackback_uid).get_post_relations(item.trackback_item_no)
            href = post_relations['self'][0]['href']
            ref = 'urn:ya.ru:post/%s/%s' % (item.trackback_uid, item.trackback_item_no)
            entry['in-reply-to'] = {'href': href, 'ref': ref}

        #FIXME: работа с content-type content'а должна быть в atomgen.create_entry
        # http://www.atomenabled.org/developers/syndication/atom-format-spec.php#rfc.section.4.1.3.3
        #
        if 'content' in item:
            content_type, content = content_from_yaru_to_atom(item['content'])
            entry['content'] = content
            entry['content_type'] = content_type
        else:
            entry['content'] = None
            entry['content_type'] = None
        if 'corp_meta' in item and item['corp_meta'].text is not None:
            try:
                entry['corp_meta'] = json.loads(item['corp_meta'].text)
            except Exception:
                pass
        return entry

    def __contains__(self, post_id):
        result = self._read_and_cache_result(post_id)
        return result is not None

    def _index(self, page_no=1, page_size=50, escape_html=False, post_type='', cat_id=0, min_time=0, max_time=0):
        offset = (page_no-1)*page_size
        freshest_post = activity.retrieve_posts(
            ai=self.ai,
            feed_id=self.feed.get_id(),
            offset=0,
            count=1,
            escape_html=escape_html,
            post_type=post_type,
            fetch_tags=False,
            cat_id=cat_id,
            min_time=min_time,
            max_time=max_time
        )
        posts = activity.retrieve_posts(
            ai=self.ai,
            feed_id=self.feed.get_id(),
            offset=offset,
            count=page_size,
            escape_html=escape_html,
            post_type=post_type,
            cat_id=cat_id,
            min_time=min_time,
            max_time=max_time,
        )
        info = {'offset': offset,
                'posts': posts,
                'add_author': self.feed.get_type() == 'club',
                'updated': freshest_post[0]['updated'] if freshest_post else None,
                'id': 'urn:ya.ru:feed/%s/%s' % ('person' if self.feed.get_type() == 'person' else 'club', self.feed.get_id()),
                'links': {'related': [{'href': self.abs_url(reverse('blog_person' if self.feed.get_type() == 'person' else 'club',
                                                                    args=[self.feed.get_id()])),
                                       'title': str(self.feed.get_name())},
                                      {'href': self.abs_url(reverse('blog_posts' if self.feed.get_type() == 'person' else 'club_posts', args=[self.feed.get_id()]) + '{type}/')}],
                          }
                }
        if post_type:
            info['id'] = info['id'] + '/' + post_type
        if posts:
            url_name = ''
            reverse_args = [self.feed.get_id()]
            if self.feed.get_type() == 'person':
                if post_type:
                    url_name = 'blog_posts_by_type'
                    reverse_args.append(post_type)
                else:
                    url_name = 'blog_posts'
            else:
                if post_type:
                    url_name = 'club_posts_by_type'
                    reverse_args.append(post_type)
                else:
                    url_name = 'club_posts'
            info['links']['next'] = [{'href': '%s?count=%s&p=%s' % (self.abs_url(reverse(url_name, args=reverse_args)), page_size, page_no+1)}]
        return info

    def index(self, page_no=1, page_size=50, escape_html=False, post_type='', cat_id=0, min_time=0, max_time=0):
        info = self._index(page_no, page_size, escape_html, post_type, cat_id, min_time, max_time)
        posts = info['posts']
        add_author = info['add_author']
        entries = []
        feeds_cache = FeedsCache(abs_url=self.abs_url)
        feed_ids = set(itertools.chain((p.author_uid for p in posts),
                                       (p.posted_in for p in posts)))
        feeds_cache.fill_cache(feed_ids)
        for post in info['posts']:
            entries.append(AtomEntry(self._sanitize_post_entry(post, add_author, feeds_cache=feeds_cache), add_author))
        renderer = AtomFeed(info['id'],
                            'List of posts',
                            self.feed,
                            add_author,
                            info['updated'],
                            entries,
                            info['links'])
        return renderer

    def trends(self, page_no=1, page_size=50):
        posts = activity.retrieve_trends(self.ai, self.feed.get_id(),
                                         page_no, page_size)
        info = {'offset': page_no,
                'posts': posts,
                'add_author': self.feed.get_type() == 'club',
                'updated': posts[0]['updated'] if posts else None,
                'id': 'urn:ya.ru:feed/%s/%s' % ('person' if self.feed.get_type() == 'person' else 'club', self.feed.get_id()),
                'links': {'related': [{'href': self.abs_url(reverse('blog_person' if self.feed.get_type() == 'person' else 'club',
                                                                    args=[self.feed.get_id()])),
                                       'title': str(self.feed.get_name())},
                                      {'href': self.abs_url(reverse('blog_posts' if self.feed.get_type() == 'person' else 'club_posts', args=[self.feed.get_id()]) + '{type}/')}],
                          }
                }
        entries = []
        feeds_cache = FeedsCache(abs_url=self.abs_url)
        for post in posts:
            entries.append(AtomEntry(self._sanitize_post_entry(post,
                                                               info['add_author'],
                                                               feeds_cache=feeds_cache),
                                     info['add_author']))
        return AtomFeed(info['id'],
                        'Trends',
                        self.feed,
                        info['add_author'],
                        info['updated'],
                        entries,
                        info['links'])

    def friends_posts(self, since, count, post_type=''):
        if since:
            posts = activity.retrieve_friends_posts(self.ai, self.feed,
                                                    since, count, post_type)
            freshest_post = activity.retrieve_friends_posts(self.ai,
                                                            self.feed, 0, 1,
                                                            post_type)
        else:
            posts = activity.retrieve_friends_posts(self.ai, self.feed,
                                                    since, count, post_type)
            freshest_post = posts if posts else None
        # FIXME а какой показывать updated? как его взять?
        updated = freshest_post[0]['updated'] if freshest_post else None
        feed_id = 'urn:ya.ru:feed/friends/%s' % (self.feed.get_id(),)
        if post_type:
            feed_id += '/' + post_type

        links = {'related': [{'href': self.abs_url(reverse('blog_person', args=[self.feed.get_id()])),
                              'title': str(self.feed.get_name())}
                             ,
                             {'href': self.abs_url(reverse('blog_friends_posts', args=[self.feed.get_id()]) + '{type}/'),
                             }]}

        if posts:
            max_timestamp = min(posts, key=lambda p: p['published_usec'])
            max_timestamp = max_timestamp['published_usec']
            next_since = int(max_timestamp)
            url_name = 'blog_friends_posts_by_type' if post_type else 'blog_friends_posts'
            reverse_args = [self.feed.get_id(), post_type] if post_type else [self.feed.get_id()]

            links['next'] = [{'href': '%s?count=%s&since=%s' % (self.abs_url(reverse(url_name, args=reverse_args)), count, next_since)}]

        entries = []
        feeds_cache = FeedsCache(abs_url=self.abs_url)
        feed_ids = set(itertools.chain((p.author_uid for p in posts),
                                       (p.posted_in for p in posts)))
        feeds_cache.fill_cache(feed_ids)
        for post in posts:
            entries.append(AtomEntry(self._sanitize_post_entry(post, add_author=True, parse_authors=True, feeds_cache=feeds_cache), add_author=True))
        renderer = AtomFeed(feed_id,
                            'List of friends posts',
                            self.feed,
                            True,
                            updated,
                            entries,
                            links)
        return renderer

    def get(self, post_id):
        result = self._read_using_cache(post_id)
        if not result:
            # если результат не получили - значит прав нету скорее всего
            raise PermissionDenied()
        post = self._sanitize_post_entry(result, add_author=True, parse_authors=True)
        renderer = AtomEntry(post, add_author=True)
        return renderer

    def delete_post(self, ai, feed, post_no):
        response = datacore.delete_post(ai, feed.get_id(), post_no)
        if 'Could not find post' in response:
            raise EntryNotFound(post_no)
        elif 'code="OK"' in response:
            return (True, '')
        else:
            raise FixMeError(response)

    def store_from_json(self, agent, input_entry):
        tag_space = urlgen.tag_space_url(self.feed, abs_url=self.abs_url)
        categories = [i for i in input_entry.get('categories', []) if i['scheme'] in ['urn:wow.ya.ru:posttypes', 'urn:ya.ru:posttypes']]
        tags = [i.term for i in input_entry.get('categories', []) if i['scheme'] == tag_space]

        assert len(categories) <= 1
        assert len(categories) == 1 and categories[0]['term'] != '' or True
        filter_type = len(categories) == 1 and categories[0]['term'] or 'text'
        default_type = filter_type

        meta_type = input_entry['meta'].get('type', default_type) if 'meta' in input_entry else None
        meta = meta_utils.parse_meta(self.feed, input_entry['meta'], meta_type, 'json', False) if 'meta' in input_entry else None
        access = input_entry.get('access', None)
        store_time = input_entry.get('store_time', '')
        if input_entry.get('comments_disabled', False):
            comments_disabled = 'y'
        else:
            comments_disabled = ''

        content = input_entry.get('content', None)
        if content is not None:
            content_type = input_entry.get('content_type', 'text/wiki')  # NOTE нормально ли так предполагать?
            if content_type in ['text', 'text/plain', 'text/wiki']:
                content_text = content or ''
            elif content_type in ['html', 'text/html']:
                content_text = content and html_unescape(content) or ''
            elif content_type == 'xhtml':
                # FIXME implement xhtml
                pass
            else:
                raise NotImplementedError('handling of content types other then text, html, xhtml not implemented')
        else:
            content_type = 'text/wiki'
            content_text = ''

        def yaru_body_content_type(atom_content_type):
            YARU_BODY_CONTENT_TYPES = ('text/plain', 'text/html', 'text/wiki')
            if atom_content_type in YARU_BODY_CONTENT_TYPES:
                return atom_content_type
            try:
                return {
                    'text': 'text/plain',
                    'html': 'text/html',
                    'xhtml': 'text/html',
                }[atom_content_type]
            except KeyError:
                raise NotImplementedError('handling of content types other then text, html, xhtml is not implemented')

        params = {"title": input_entry.get('title', ''),
                  "body": content_text,
                  "body_content_type": yaru_body_content_type(content_type),
                  "filter_type": filter_type,
                  "access_type": access,
                  "freezed": comments_disabled,
                  "store_time": store_time,
                  "tags": ", ".join(tags),
                  }
        result = activity.create_post(
            self.ai,
            self.feed.get_id(),
            agent,
            default_type,
            meta_type = meta_type,  #meta_elem and meta_elem.attrib['type'],
            meta = meta,  #meta_elem and etree2py(meta_elem),
            display =input_entry.get('display'),  #display_elem and display2dict(display_elem),
            params = params,
        )
        if not result:
            return None
        renderer = AtomEntry(self._sanitize_post_entry(result), True)
        return (result.id, renderer)

    def store_from_xml(self, agent, input_entry, preview=False):
        old_ns = input_entry.nsmap.get('y') == atomgen.OLD_YAPI_NS
        #TODO: tag брать из категорий entry
        #TODO: унести отсюда, передать в atom.Entry
        tag_space = urlgen.tag_space_url(self.feed, abs_url=self.abs_url)
        categories = [i for i in input_entry.categories if i.scheme in ['urn:wow.ya.ru:posttypes', 'urn:ya.ru:posttypes']]
        tags = [ i.term for i in input_entry.categories if i.scheme in (tag_space, 'tag')]
        assert len(categories) <= 1
        assert len(categories) == 1 and categories[0].term != '' or True
        filter_type = len(categories) == 1 and categories[0].term or 'text'
        default_type = filter_type

        meta_elem = input_entry.find(atomgen._qname('meta', old_ns=old_ns))
        store_time_elem = input_entry.find(atomgen._qname('store_time', old_ns=old_ns))
        display_elem = input_entry.find(atomgen._qname('display', old_ns=old_ns))
        access_elem = input_entry.find(atomgen._qname('access', old_ns=old_ns))
        comments_disabled_elem = input_entry.find(atomgen._qname('comments_disabled', old_ns=old_ns))
        store_time = store_time_elem.text if store_time_elem is not None else ''
        meta_type = meta_elem.attrib.get('type', default_type) if meta_elem else None
        #TODO: проверять валидность meta
        meta = meta_utils.parse_meta(self.feed, meta_elem, meta_type, 'xml', old_ns=old_ns) if meta_elem else None
        #TODO: проверять валидность display
        display = etree2py(display_elem, single_as_list=False) if display_elem else None
        # TODO: проверять валидность access
        access = access_elem.text if access_elem is not None else None
        comments_disabled = 'y' if comments_disabled_elem is not None else ''

        #TODO: научиться работать с content type="{html,xhtml,xml?}"
        # http://www.atomenabled.org/developers/syndication/atom-format-spec.php#rfc.section.4.1.3.3
        #
        #FIXME: нуждается в причёсывании
        content = input_entry.content
        if input_entry.content is not None:
            content_type, content_text = parse_content_from_xml(input_entry.content)
        else:
            content_type = 'text/wiki'
            content_text = ''

        def yaru_body_content_type(atom_content_type):
            YARU_BODY_CONTENT_TYPES = ('text/plain', 'text/html', 'text/wiki')
            if atom_content_type in YARU_BODY_CONTENT_TYPES:
                return atom_content_type
            try:
                return {
                    'text': 'text/plain',
                    'html': 'text/html',
                    'xhtml': 'text/html',
                }[atom_content_type]
            except KeyError:
                raise NotImplementedError('handling of content types other then text, html, xhtml is not implemented')
        if input_entry.id:
            item_no = input_entry.id.split('/')[-1]
        else:
            item_no = None
        params = {"title": input_entry.title,
                  "body": content_text,
                  "id": item_no,
                  "body_content_type": yaru_body_content_type(content_type),
                  "filter_type": filter_type,
                  "access_type": access,
                  "freezed": comments_disabled,
                  "store_time":  store_time,
                  "tags": ", ".join(tags),
                  }
        if input_entry.corp_meta is not None:
            params['corp_meta'] = input_entry.corp_meta.text
        result = activity.create_post(
            self.ai,
            self.feed.get_id(),
            default_type,
            meta_type = meta_type, #meta_elem and meta_elem.attrib['type'],
            meta = meta, #meta_elem and etree2py(meta_elem),
            display = display, #display_elem and display2dict(display_elem),
            params = params,
            preview=preview
        )
        if not result:
            return None
        renderer = AtomEntry(self._sanitize_post_entry(result), True)
        return (result.id, renderer)
