# -*- coding: utf-8 -*-
from django.http import HttpResponse, HttpResponseBadRequest
from django.utils.encoding import force_str
from at.api.yaru import atom
from at.api.yaru import activity
from at.api.yaru.utils.django_helpers import response_as_atom_feed, \
    response_as_atom_entry, \
    to_canonical, \
    require_permission, \
    get_pagination, \
    check_post_type, \
    get_format, \
    get_ctype, \
    JSON_CONTENT_TYPE, \
    slug_to_id
from at.api.yaru.post.store import PostAtomStore
from at.api.yaru.errors import InvalidInputError, OutOfBoundsError
from at.api.yaru.instances import Blog, Club
from lxml.etree import XMLSyntaxError
import json


def _get_posts(request, feed, post_type='', cat_id=0, min_time=0, max_time=0, escape_html=False):
    if post_type:
        check_post_type(post_type)
    format, forced = get_format(request, ['application/x-yaru-atom+json',
                                          'application/atom+xml'],
                                'application/atom+xml')
    page_no, page_size = get_pagination(request)
    post_store = PostAtomStore(request.ai, feed,
                               abs_url=request.build_absolute_uri)
    renderer = post_store.index(page_no, page_size, escape_html=escape_html,
                                post_type=post_type, cat_id=cat_id,
                                min_time=min_time, max_time=max_time)
    if 'xml' in format:
        feed = renderer.to_xml(forced, render=True)
        return response_as_atom_feed(feed)
    else:
        feed = renderer.to_json(forced, render=True)
        return HttpResponse(feed, content_type='application/x-yaru-atom+json')


def _get_recent_posts(request):
    feed_ids = request.GET.getlist('feed_id', default=[])
    if feed_ids:
        feed_ids = set([slug_to_id(Club, x) if not x.isdigit() else x for x in feed_ids])
    else:
        raise InvalidInputError('No feed_id found')

    if len(feed_ids) > 100:
        raise OutOfBoundsError('Number of feed_id should be less than 100 per one request')

    page_no, page_size = get_pagination(request, max=100)
    posts = []

    for feed_id in feed_ids:
        feed = Club(feed_id)
        post_store = PostAtomStore(request.ai, feed)
        renderer = post_store.index(page_size=page_size)
        posts.extend(renderer.entries)

    recent_posts = sorted(posts, key=lambda x: x.post['post_published'], reverse=True)[:page_size]
    recent_json_posts = [post.to_json(forced=False, render=True) for post in recent_posts]

    return HttpResponse(recent_json_posts, content_type='application/x-yaru-atom+json')


def _get_post(request, feed, post_no):
    format, forced = get_format(request, ['application/x-yaru-atom+json',
                                          'application/atom+xml'],
                                'application/atom+xml')
    post_store = PostAtomStore(request.ai, feed,
                               abs_url=request.build_absolute_uri)
    renderer = post_store.get(post_no)
    if 'xml' in format:
        return response_as_atom_entry(renderer.to_xml(forced, render=True))
    else:
        return HttpResponse(renderer.to_json(forced, render=True),
                            content_type='application/x-yaru-atom+json')


def _preview_post(request, feed):
    ctype, ctype_params = get_ctype(request)
    if not ctype:
        ctype = 'application/atom+xml'
        ctype_params = {}
    format, forced = get_format(request, ['application/x-yaru-atom+json',
                                          'application/atom+xml'],
                                'application/atom+xml')
    post_store = PostAtomStore(request.ai, feed,
                               abs_url=request.build_absolute_uri)
    if ctype == 'application/x-yaru-atom+json':
        return HttpResponse(code=400)
    try:
        input_entry = atom.ATOM(request.body)
    except XMLSyntaxError as e:
        print(e)
        return HttpResponseBadRequest("Malformed XML")
    post_id, renderer = post_store.store_from_xml(None, input_entry,
                                                  preview=True)
    return response_as_atom_entry(renderer.to_xml(forced, render=True),
                                  status=200)


def _create_post(request, feed):
    ctype, ctype_params = get_ctype(request)
    if not ctype:
        ctype = 'application/atom+xml'
        ctype_params = {}
    format, forced = get_format(request, ['application/x-yaru-atom+json',
                                          'application/atom+xml'],
                                'application/atom+xml')
    post_store = PostAtomStore(request.ai, feed,
                               abs_url=request.build_absolute_uri)
    if ctype == 'application/x-yaru-atom+json':
        try:
            input_entry = json.loads(request.body)
        except:  # FIXME Почему не ловишь ничего конкретного?
            return HttpResponseBadRequest("Malformed JSON")
        post_id, renderer = post_store.store_from_json(None,
                                                       input_entry)
    else:
        try:
            input_entry = atom.ATOM(request.body)
        except XMLSyntaxError as e:
            print(e)
            return HttpResponseBadRequest("Malformed XML")
        post_id, renderer = post_store.store_from_xml(None,
                                                      input_entry)
    location = feed.get_post_relations(post_id)['self'][0][
        'href']  # NOTE сюда наверное тоже надо добавлять ?format=...
    if 'xml' in format:
        return response_as_atom_entry(force_str(renderer.to_xml(forced, render=True)),
                                      status=201,
                                      headers={'Location': location})
    else:
        response = HttpResponse(renderer.to_json(forced, render=True),
                                status=201,
                                content_type='application/x-yaru-atom+json')
        response['Location'] = location
        return response


def _delete_post(request, feed, post_no):
    post_store = PostAtomStore(request.ai, feed,
                               abs_url=request.build_absolute_uri)
    (ok, raw_response) = post_store.delete_post(request.ai, feed, post_no)
    if ok:
        return HttpResponse('', status=200)
    else:
        return HttpResponse('', status=500)  # FIXME какой должен быть код?


@to_canonical(Club, 'club_trends')
@require_permission('yaru:read', allow_anonymous=True)
def get_club_trends(request, feed_id):
    return get_feed_trends(request,
                           Club(feed_id, abs_url=request.build_absolute_uri))


@to_canonical(Blog, 'blog_trends')
@require_permission('yaru:read', allow_anonymous=True)
def get_blog_trends(request, feed_id):
    return get_feed_trends(request,
                           Blog(feed_id, abs_url=request.build_absolute_uri))


def get_feed_trends(request, feed):
    format, forced = get_format(request, ['application/x-yaru-atom+json',
                                          'application/atom+xml'],
                                'application/atom+xml')
    page_no, page_size = get_pagination(request)
    post_store = PostAtomStore(request.ai, feed,
                               abs_url=request.build_absolute_uri)
    renderer = post_store.trends(page_no - 1, page_size)
    if 'xml' in format:
        feed = renderer.to_xml(forced, render=True)
        return response_as_atom_feed(feed)
    else:
        feed = renderer.to_json(forced, render=True)
        return HttpResponse(feed, content_type='application/x-yaru-atom+json')


@to_canonical(Blog, 'blog_posts')
@require_permission('yaru:read', allow_anonymous=True)
def get_blog_posts(request, feed_id):
    return _get_posts(request,
                      Blog(feed_id, abs_url=request.build_absolute_uri))


@to_canonical(Blog, 'blog_posts_by_category')
@require_permission('yaru:read', allow_anonymous=True)
def get_blog_posts_by_category(request, feed_id, cat_id):
    return _get_posts(request,
                      Blog(feed_id, abs_url=request.build_absolute_uri),
                      cat_id=int(cat_id))


@to_canonical(Blog, 'blog_posts_by_type')
@require_permission('yaru:read', allow_anonymous=True)
def get_blog_posts_by_type(request, feed_id, post_type):
    return _get_posts(request,
                      Blog(feed_id, abs_url=request.build_absolute_uri),
                      post_type)


@to_canonical(Blog, 'blog_post')
@require_permission('yaru:read', allow_anonymous=True)
def get_blog_post(request, feed_id, post_no):
    return _get_post(request, Blog(feed_id, abs_url=request.build_absolute_uri),
                     post_no)


@require_permission('yaru:write')
def preview_blog_post(request, feed_id):
    return _preview_post(request,
                         Blog(feed_id, abs_url=request.build_absolute_uri))


@require_permission('yaru:write')
def create_blog_post(request, feed_id):
    return _create_post(request,
                        Blog(feed_id, abs_url=request.build_absolute_uri))


@require_permission('yaru:write')
def delete_blog_post(request, feed_id, post_no):
    return _delete_post(request,
                        Blog(feed_id, abs_url=request.build_absolute_uri),
                        post_no)


@to_canonical(Club, 'club_posts')
@require_permission('yaru:read', allow_anonymous=True)
def get_club_posts(request, feed_id):
    return _get_posts(request,
                      Club(feed_id, abs_url=request.build_absolute_uri))


@require_permission('yaru:read', allow_anonymoys=True)
def get_recent_posts(request):
    return _get_recent_posts(request)


@to_canonical(Club, 'club_posts_by_type')
@require_permission('yaru:read', allow_anonymous=True)
def get_club_posts_by_type(request, feed_id, post_type):
    return _get_posts(request,
                      Club(feed_id, abs_url=request.build_absolute_uri),
                      post_type)


@to_canonical(Blog, 'club_posts_by_category')
@require_permission('yaru:read', allow_anonymous=True)
def get_club_posts_by_category(request, feed_id, cat_id):
    return _get_posts(request,
                      Club(feed_id, abs_url=request.build_absolute_uri),
                      cat_id=int(cat_id))


@to_canonical(Blog, 'club_posts_by_interval')
@require_permission('yaru:read', allow_anonymous=True)
def get_club_posts_by_interval(request, feed_id, min_time, max_time):
    return _get_posts(request,
                      Club(feed_id, abs_url=request.build_absolute_uri),
                      min_time=min_time, max_time=max_time)


@to_canonical(Club, 'club_post')
@require_permission('yaru:read', allow_anonymous=True)
def get_club_post(request, feed_id, post_no):
    return _get_post(request, Club(feed_id, abs_url=request.build_absolute_uri),
                     post_no)


@require_permission('yaru:write')
def create_club_post(request, feed_id):
    return _create_post(request,
                        Club(feed_id, abs_url=request.build_absolute_uri))


@require_permission('yaru:write')
def preview_club_post(request, feed_id):
    return _preview_post(request,
                         Club(feed_id, abs_url=request.build_absolute_uri))


@require_permission('yaru:write')
def delete_club_post(request, feed_id, post_no):
    return _delete_post(request,
                        Club(feed_id, abs_url=request.build_absolute_uri),
                        post_no)


def _subscribe_post(ai, feed_id, post_no, subscribe=1):
    return HttpResponse(
        activity.subscribe_post(ai, feed_id, post_no, subscribe),
        content_type=JSON_CONTENT_TYPE)


@to_canonical(Club, 'club_post_subscribe')
@require_permission('yaru:read')
def subscribe_club_post(request, feed_id, post_no):
    return _subscribe_post(request.ai, feed_id, post_no, 1)


@to_canonical(Club, 'club_post_unsubscribe')
@require_permission('yaru:read')
def unsubscribe_club_post(request, feed_id, post_no):
    return _subscribe_post(request.ai, feed_id, post_no, 0)


@to_canonical(Blog, 'blog_post_subscribe')
@require_permission('yaru:read')
def subscribe_blog_post(request, feed_id, post_no):
    return _subscribe_post(request.ai, feed_id, post_no, 1)


@to_canonical(Blog, 'blog_post_unsubscribe')
@require_permission('yaru:read')
def unsubscribe_blog_post(request, feed_id, post_no):
    return _subscribe_post(request.ai, feed_id, post_no, 0)
