# -*- coding: utf-8 -*-



from lxml import etree
import elementflow
from io import BytesIO, StringIO
from datetime import datetime
import json
from at.api.yaru.utils.encoding import force_unicode
from at.api.yaru.utils.links import mk_xml_links, mk_json_links, Link
from at.api.yaru import atom, atomgen, urlgen
from at.api.yaru.instances import Club, Blog


ATOM_NAMESPACES = {'': 'http://www.w3.org/2005/Atom',
                   'y': atomgen.YAPI_NS,
                   'thr': 'http://purl.org/syndication/thread/1.0'}

# C2P = corba -> python
COMMUNITY_TYPES_MAP_C2P = {'NONE': None,
                           'OPENED_COMMUNITY': 'open',
                           'PREMODERATED_COMMUNITY': 'premoderated',
                           'CLOSED_COMMUNITY': 'closed',
                           'PREMODERATED_CLOSED_COMMUNITY': 'premoderated_and_closed',
                           'FORUM': 'forum',
                           }

YARU_PROFILE_FIELDS = ['name',
                       'name_eng',
                       'sex',
                       'email',
                       'birth_year',
                       'birth_month',
                       'birth_day',
                       ]

YARU_PROFILE_STATUS_FIELDS = ['account-status',
                              'qu',
                              'score_to_level',
                              'mood',
                              'qu_up_time',
                              ]


def render_xml(flow):
    return flow.file.getvalue()

def render_json(doc):
    return json.dumps(doc, indent=2)

def mk_flow(root_name, namespaces=None):
    namespaces = namespaces or {'': atomgen.YAPI_NS}
    flow = elementflow.xml(BytesIO(), root_name,
                           namespaces=namespaces,
                           indent=True)
    return flow

def get_flow(flow, root_name, namespaces=None):
    if flow:
        return flow.container(root_name)
    else:
        return mk_flow(root_name, namespaces)

def dict_filter(d, keys):
    return dict((k, v) for (k, v) in d.items() if k in keys)


class KeyValue(object):
    def __init__(self, key, value):
        self.key = key
        self.value = value


class PostMetaSimple(object):
    def __init__(self, meta):
        self.meta = meta

    def entry_to_xml(self, flow, entry, forced):
        if issubclass(entry.__class__, Link):
            flow.element('y:%s' % entry.name, text=str(entry.text), attrs=entry.attrs)
        elif issubclass(entry.__class__, Club) or issubclass(entry.__class__, Blog):
            renderer = YaruProfile(entry, entry.get_profile(), is_short=True)
            renderer.to_xml(forced, flow, namespace='y')
        elif issubclass(entry.__class__, KeyValue):
            flow.element('y:%s' % entry.key, text=str(entry.value or ''))

    def entry_to_json(self, entry, forced):
        if issubclass(entry.__class__, Link):
            new_entry = {}
            if entry.attrs:
                new_entry['attrs'] = entry.attrs
            if entry.text:
                new_entry['text'] = entry.text
            if list(new_entry.keys()) == ['text']:
                return (entry.name, new_entry['text'])
            else:
                return (entry.name, new_entry)
        elif issubclass(entry.__class__, Club) or issubclass(entry.__class__, Blog):
            renderer = YaruProfile(entry, entry.get_profile(), is_short=True)
            return (entry.get_type(), renderer.to_json(forced))
        elif issubclass(entry.__class__, KeyValue):
            return (entry.key, entry.value)

    def to_xml(self, flow, forced):
        for entry in self.meta:
            self.entry_to_xml(flow, entry, forced)
        return flow

    def to_json(self, forced):
        entries = {}
        for entry in self.meta:
            key, value = self.entry_to_json(entry, forced)
            entries[key] = value
        return entries


class PostMetaPhoto(PostMetaSimple):
    def entry_to_xml(self, flow, entry, forced):
        with flow.container('y:image'):
            author = entry['person']
            renderer = YaruProfile(author, author.get_profile(), is_short=True)
            renderer.to_xml(forced, flow, namespace='y')
            flow.element('y:url', text=entry['url'])
            for size, attrs in entry['media'].items():
                flow.element('y:media', attrs={'size': size,
                                               'width': attrs['width'],
                                               'height': attrs['height'],
                                               'src': attrs['src'],
                                               })

    def entry_to_json(self, entry, forced):
        author = entry['person']
        renderer = YaruProfile(author, author.get_profile(), is_short=True)
        entry['person'] = renderer.to_json(forced)
        return entry

    def to_json(self, forced):
        images = []
        for entry in self.meta:
            images.append(self.entry_to_json(entry, forced))
        return {'images': images}


class AtomComments(object):
    def __init__(self, post):
        self.post = post

    def to_json(self, render=False):
        return render_json(self._serialize_json(self.post)) if render else self._serialize_json(self.post)

    def _serialize_json(self, reply):
        reply['replies'] = [_f for _f in reply['replies'] if _f]
        for r in reply['replies']:
            r['links'] = mk_json_links(r['links'], force_format=True)
        list(map(self._serialize_json, reply['replies']))
        return reply

    def _serialize_reply(self, reply):
        xreply = etree.Element('reply')
        etree.SubElement(xreply, 'id').text = reply['id']
        etree.SubElement(xreply, 'published').text = reply['published']
        etree.SubElement(xreply, 'parent').text = reply['parent']
        etree.SubElement(xreply, 'reply_type').text = reply['reply_type']
        etree.SubElement(xreply, 'author_id').text = reply['author_uid']
        etree.SubElement(xreply, 'trackback').text = reply['trackback']
        etree.SubElement(xreply, 'title').text = reply.get('title', '')
        if 'content_type' in reply:
            etree.SubElement(xreply, 'content_type').text = reply['content_type']
        etree.SubElement(xreply, 'content').text = reply.get('content', '')

        replies = etree.SubElement(xreply, 'replies')
        for r in reply['replies']:
            replies.append(self._serialize_reply(r))

        return xreply

    def to_xml(self, render=False):
        comments = etree.Element('comments')
        etree.SubElement(comments, 'id').text = self.post['id']
        etree.SubElement(comments, 'feed_id').text = self.post['feed_id']

        replies = etree.SubElement(comments, 'replies')
        for reply in self.post['replies']:
            replies.append(self._serialize_reply(reply))

        if render:
            return etree.tostring(comments, encoding="utf-8",
                                  xml_declaration=True, pretty_print=True)
        else:
            return comments


class AtomFeed(object):
    def __init__(self, feed_id, title, feed_author, add_author, updated, entries, relations):
        self.feed_id = feed_id
        self.title = title
        self.feed_author = feed_author
        self.add_author = add_author
        self.updated = updated
        self.entries = entries
        self.relations = relations

    def to_xml(self, forced, render=False):
        flow = mk_flow('feed', namespaces=ATOM_NAMESPACES)
        with flow:
            flow.element('id', text=self.feed_id)
            with flow.container('author') as author:
                author.element('name', text=self.feed_author.get_name())
                author.element('uri', text=self.feed_author.get_profile_link(self.feed_author.get_login()))
                author_renderer = YaruProfile(self.feed_author, self.feed_author.get_profile(), is_short=True)
                author_renderer.to_xml(forced, flow, render=False, namespace='y', no_container=True, exclude=['name'])
            flow.element('title', text=self.title)
            flow.element('updated', text=atom.strftime(self.updated) if self.updated else '')
            for entry in self.entries:
                entry.to_xml(forced, flow)
            mk_xml_links(flow, self.relations, forced)
        return render_xml(flow) if render else flow

    def to_json(self, forced, render=False):
        feed = {'id': self.feed_id,
                'title': self.title,
                'updated': atom.strftime(self.updated) if self.updated else '',
                'author': YaruProfile(self.feed_author,
                                      self.feed_author.get_profile(),
                                      is_short=True).to_json(forced),
                }
        es = []
        for entry in self.entries:
            es.append(entry.to_json(forced))
        feed['entries'] = es
        feed['links'] = mk_json_links(self.relations, forced)
        return render_json(feed) if render else feed


class AtomEntry(object):
    def __init__(self, post, add_author):
        self.post = post # At the moment I expect it to be already sanitized
        self.add_author = add_author

    def to_xml(self, forced, flow=None, render=False):
        with get_flow(flow, 'entry', namespaces=ATOM_NAMESPACES) as flow:
            flow.element('id', text=self.post['id'])
            flow.element('title', text=self.post['post_title'])
            flow.element('published', text=atom.strftime(self.post['post_published']))
            flow.element('updated', text=atom.strftime(self.post['post_updated']) if self.post['post_updated'] else '')
            post_edited = self.post.get('post_edited')
            if post_edited:
                flow.element('edited', text=atom.strftime(post_edited))
            flow.element('y:access', text=self.post['access'])
            flow.element('y:agent', text=self.post['agent'] or '')
            flow.element('y:comment_count', text=self.post['comment_count'])
            if self.add_author:
                with flow.container('author'):
                    flow.element('name', text=str(self.post['author_name']))
                    flow.element('uri', text=self.post['author_uri'])
                    author_profile = self.post['author'].get_profile()
                    author_renderer = YaruProfile(self.post['author'], author_profile, is_short=True)
                    author_renderer.to_xml(forced, flow, render=False, namespace='y', no_container=True, exclude=['name'])
            if self.post['comments_disabled']:
                flow.element('y:comments_disabled')
            mk_xml_links(flow, self.post['links'], forced)
            for args in self.post['categories']:
                flow.element('category', attrs=args)
            likes = self.post.get('likes')
            if likes:
                with flow.container('y:likes'):
                    for like in likes:
                        flow.element('y:like', attrs=like)

            display = self.post.get('display')
            if display:
                with flow.container('y:display'):
                    label = display.get('label')
                    style = display.get('style')
                    template = display.get('template')
                    for (elem, value) in [('y:label', label), ('y:style', style), ('y:template', template)]:
                        if value:
                            flow.element(elem, text=force_unicode(value, strings_only=False))
                        else:
                            flow.element(elem)

            meta = self.post.get('meta')
            if meta:
                with flow.container('y:meta'):
                    meta.to_xml(flow, forced)

            if self.post['original']:
                flow.element('y:original', text=self.post['original'])
            if 'in-reply-to' in self.post:
                flow.element('thr:in-reply-to', attrs={'href': self.post['in-reply-to']['href'],
                                                       'ref': self.post['in-reply-to']['ref']})

            if self.post['content'] is not None:
                ctype = self.post['content_type']
                if ctype == 'text/xhtml':
                    # FIXME поддержи xhtml, будь человеком
                    raise NotImplementedError('text/xhtml is not supported at the moment')
                elif ctype == 'text/html':
                    flow.element('content', attrs={'type': 'html'}, text=self.post['content'])
                else:
                    flow.element('content', text=self.post['content'])

        return render_xml(flow) if render else flow

    def to_json(self, forced, render=False):
        entry = {'id': self.post['id'],
                 'title': self.post['post_title'],
                 'published': atom.strftime(self.post['post_published']),
                 'updated': atom.strftime(self.post['post_updated'] if self.post['post_updated'] else ''),
                 'access': self.post['access'],
                 'agent': self.post['agent'],
                 'comment_count': self.post['comment_count'],
                 'comments_disabled': self.post['comments_disabled'],
                 'categories': self.post['categories'],
                 'links': mk_json_links(self.post['links'], forced),
                 'on_premoderation': self.post.get('on_premoderation', False)
                 }
        likes = self.post.get('likes')
        if likes:
            entry["likes"] = likes
        post_edited = self.post.get('post_edited')
        if post_edited:
            entry["edited"] = atom.strftime(post_edited)
        display = self.post.get('display')
        if display:
            entry['display'] = {}
            label = display.get('label')
            style = display.get('style')
            template = display.get('template')
            for (elem, value) in [('label', label), ('style', style), ('template', template)]:
                entry['display'][elem] = force_unicode(value, strings_only=False) if value else None

        meta = self.post.get('meta')
        if meta:
            # XXX looks weird
            meta = meta.to_json(forced)
        corp_meta = self.post.get('corp_meta')
        if corp_meta:
            entry["corp_meta"] = corp_meta

        if self.post['original']:
            entry['original'] = self.post['original']
        if 'in-reply-to' in self.post:
            entry['in-reply-to'] = self.post['in-reply-to']
        if self.add_author:
            author_profile = self.post['author'].get_profile()
            author_rels = self.post['author'].get_relations()
            author_renderer = YaruProfile(self.post['author'], author_profile, is_short=True)
            entry['author'] = author_renderer.to_json(forced)
        entry['meta'] = meta
        entry['content'] = self.post['content'] # TODO handle xhtml
        ctype = self.post['content_type']
        if ctype:
            entry['content_type'] = ctype

        return render_json(entry) if render else entry


class YaruFeed(object):
    def __init__(self, feed_id, entries, relations):
        self.feed_id = feed_id
        self.entries = entries
        self.relations = relations

    def to_xml(self, forced, root_name, render=False):
        flow = mk_flow(root_name)
        with flow:
            if self.feed_id: # FIXME такого воркэраунда быть не должно, но к сожалению не у всех фидов есть id
                flow.element('id', text=self.feed_id)
            for entry in self.entries:
                entry.to_xml(forced, flow)
            mk_xml_links(flow, self.relations, forced)
        return render_xml(flow) if render else flow

    def to_json(self, forced, entries_container, render=False):
        feed = {'id': self.feed_id}
        es = []
        for entry in self.entries:
            es.append(entry.to_json(forced))
        feed[entries_container] = es
        feed['links'] = mk_json_links(self.relations, forced)
        return render_json(feed) if render else feed


class ClubListTemplate(object):

    def __init__(self, feeds):
        self.feeds = feeds

    def to_json(self, forced):
        return render_json([feed.to_json(forced) for feed in self.feeds])

class YaruProfile(object):
    def __init__(self, feed, profile=None, is_short=False):
        self.is_short = is_short
        self.feed = feed
        self.profile = profile
        self.type = self.feed.get_type()

    def to_xml(self, forced, flow=None, render=False, namespace='', no_container=False, exclude=None):
        # NOTE Сейчас поддерживается только exclude=['name']
        exclude = exclude or []
        if no_container:
            self._xml_without_container(forced, flow, namespace=namespace, exclude=exclude)
        else:
            elem = lambda name: name if not namespace else '%s:%s' % (namespace, name)
            with get_flow(flow, elem(self.feed.get_type())) as flow:
                self._xml_without_container(forced, flow, namespace=namespace, exclude=exclude)
        return render_xml(flow) if render else flow

    def _xml_without_container(self, forced, flow=None, namespace='', exclude=None):
        # NOTE Сейчас поддерживается только exclude=['name']
        exclude = exclude or []
        elem = lambda name: name if not namespace else '%s:%s' % (namespace, name)
        flow.element(elem('id'), text='urn:ya.ru:%s/%s' % (self.feed.get_type(), self.feed.get_id()))
        if self.feed.get_type() == 'person': # person
            with flow.container(elem('status')) as status:
                if not self.is_short:
                    for field in YARU_PROFILE_STATUS_FIELDS:
                        status.element(elem(field), text=str(self.profile[field]))
                else:
                    status.element(elem('mood'), text=self.profile['mood'])
            if not self.is_short:
                mk_xml_links(flow, self.feed.get_relations(), forced, namespace=namespace)
                for field in YARU_PROFILE_FIELDS:
                    flow.element(elem(field), text=str(self.profile[field]))
            else:
                mk_xml_links(flow, dict_filter(self.feed.get_relations(), ['userpic', 'self']), forced, namespace=namespace)
                if 'name' not in exclude:
                    flow.element(elem('name'), text=self.profile['name'])
                flow.element(elem('sex'), text=self.profile['sex'])
        else: # club
            flow.element(elem('name'), text=self.profile['name'])
            community_type = COMMUNITY_TYPES_MAP_C2P[str(self.profile['community_type'])]
            flow.element(elem('community_type'), text=str(community_type) if community_type else None)
            flow.element(elem('members_count'), text=str(self.feed.get_members_count()))
            mk_xml_links(flow, self.feed.get_relations(), forced, namespace=namespace)

        if self.profile.tags:
            for tag in self.profile.tags:
                flow.element(elem('category'), attrs=tag)

        return flow

    def to_json(self, forced, render=False):
        entry = {'id': 'urn:ya.ru:%s/%s' % (self.feed.get_type(), self.feed.get_id())}
        if self.feed.get_type() == 'person': # person
            if not self.is_short:
                entry['links'] = mk_json_links(self.feed.get_relations(), forced)

                if self.profile:
                    entry['status'] = dict((field, self.profile[field]) for field in YARU_PROFILE_STATUS_FIELDS)
                    for field in YARU_PROFILE_FIELDS:
                        entry[field] = self.profile.get(field, '')

            else:
                entry['links'] = mk_json_links(dict_filter(
                    self.feed.get_relations(), ['userpic', 'self']), forced)

                if self.profile:
                    entry['status'] = {'mood': self.profile['mood']}
                    entry['name'] = self.profile['name']
                    entry['name_eng'] = self.profile['name_eng']
                    entry['sex'] = self.profile['sex']

        else: # club
            if self.profile:
                entry['name'] = self.profile['name']
                entry['name_eng'] = self.profile['name_eng']
                try:
                    community_type = COMMUNITY_TYPES_MAP_C2P[str(self.profile['community_type'])]
                except KeyError:
                    community_type = None
                entry['community_type'] = str(community_type) if community_type else None
                entry['creation_date'] = atom.strftime(datetime.utcfromtimestamp(float(self.profile['creation_date'])))

            entry['members_count'] = self.feed.get_members_count()
            entry['links'] = mk_json_links(self.feed.get_relations(), forced)

        if self.profile and self.profile.tags:
            entry['categories'] = self.profile.tags

        return render_json(entry) if render else entry
