#
#       Part of Libbokka library
#       (c) humanemagica.com
#       http://humanemagica.com/libbokka
#

import datetime

__all__ = [ "toJson", "fromJson" ]

try:
    import json
except ImportError:
    class json:
        @classmethod
        def dumps(c, obj, cls):
            # str produces single quote symbols around strings
            # while JSON requires double quotes
            return str(obj).replace('\'', '\"')
        @classmethod
        def loads(c, json):
            return json
        class JSONEncoder:
                def default(self,o):
                    return '<not implemented>'

def toJson(obj):
    '''
    Converts object to JSON-compatible string.
    Datetime/date/time aware.

    Example:
    >>> absjson.toJson( dict( test=[1,2,3], str="string", dt=datetime.datetime.now()) )
    '''
    return json.dumps(obj,  cls=Encoder)

def fromJson(json):
    '''
    Converts JSON string into native python object.
    Not aware of datetime or anything, just wrapper over json for some compliance.

    Example:
    >>> absjson.fromJson('{"t":6, "a":5}')
    '''
    return json.loads(json)


class Encoder(json.JSONEncoder):
    """
    JSONEncoder subclass that knows how to encode date/time types
    """

    DATE_FORMAT = "%Y-%m-%d"
    TIME_FORMAT = "%H:%M:%S"

    def default(self, o):
        if isinstance(o, datetime.datetime):
            return o.strftime("%s %s" % (self.DATE_FORMAT, self.TIME_FORMAT) )
        elif isinstance(o, datetime.date):
            return o.strftime(self.DATE_FORMAT)
        elif isinstance(o, datetime.time):
            return o.strftime(self.TIME_FORMAT)
        else:
            return super(Encoder, self).default(o)
