# -*- coding: utf-8 -*-
import cgi

import yenv

from django.http import HttpResponse, HttpResponseForbidden, \
    HttpResponseRedirect
from django.core.urlresolvers import reverse

from at.api.yaru.atomgen import CONTENTTYPE_ATOM_ENTRY, CONTENTTYPE_ATOM_FEED, \
    etree_to_string
from at.api.yaru.errors import InvalidInputError, OutOfBoundsError, \
    MediaTypeError, NotFound

import mimeparse

JSON_CONTENT_TYPE = "application/x-yaru+json"

KNOWN_POST_TYPES = ['activity_fotki',
                    'activity_video',
                    'answer',
                    'complaint',
                    'congratulation',
                    'description',
                    'friend',
                    'join',
                    'link',
                    'model_grade',
                    'model_opinion',
                    'news',
                    'offline',
                    'opinion',
                    'photo',
                    'poll',
                    'premoderated',
                    'question',
                    'rename',
                    'rules',
                    'shop_grade',
                    'shop_opinion',
                    'status',
                    'text',
                    'unfriend',
                    'unjoin',
                    'userpic',
                    'video',
                    'wishlist',
                    ]


def get_format(request, supported_types, default_type):
    format = request.GET.get('format', None)
    forced = False
    if not format:
        format = mimeparse.best_match(
            supported_types,
            request.META.get('HTTP_ACCEPT', default_type)
        )
    else:
        forced = True
        # FIXME что за грязный хак?
        # плюсы нельзя указать в GET-параметрах
        format = [f for f in supported_types if format in f]
        if format:
            format = format[0]
    if not format or format not in supported_types:
        raise MediaTypeError(
            'Supported formats: %s' % ', '.join(supported_types))
    return (format, forced)


def get_ctype(request):
    request_ctype, request_params = cgi.parse_header(
        request.META.get('CONTENT_TYPE', ''))
    return request_ctype, request_params


def check_post_type(post_type):
    if post_type not in KNOWN_POST_TYPES:
        raise NotFound('Unknown post type: %s' % post_type)


def response_as_atom_feed(data, status=200, headers=None):
    if isinstance(data, str):
        atom = data
    else:
        atom = etree_to_string(data)
    response = HttpResponse(atom, content_type=CONTENTTYPE_ATOM_FEED,
                            status=status)
    headers = headers or {}
    for k, v in headers.items():
        response[k] = v
    return response


def response_as_atom_entry(data, status=200, headers=None):
    if isinstance(data, str):
        atom = data
    else:
        atom = etree_to_string(data)
    response = HttpResponse(atom, content_type=CONTENTTYPE_ATOM_ENTRY,
                            status=status)
    headers = headers or {}
    for k, v in headers.items():
        response[k] = v
    return response


def slug_to_id(feed_type, slug):
    feed = feed_type(slug)
    return feed.get_id()


def to_canonical(feed_type, url_name, keys=None):
    def decorator(view):
        def wrapper(request, **kwargs):
            redirect = False
            for key in keys or ['feed_id']:
                value = kwargs[key]
                if not value.isdigit():
                    value = slug_to_id(feed_type, value)
                    redirect = True
                kwargs[key] = value

            feed = feed_type(kwargs['feed_id'])
            feed.exists()

            if redirect:
                target_url = reverse(url_name, kwargs=kwargs)
                url_params = request.GET.urlencode()
                if url_params:
                    target_url += '?' + url_params
                return HttpResponseRedirect(target_url)
            return view(request, **kwargs)

        return wrapper

    return decorator


def require_ctypes(ctypes):
    def decorator(view):
        def wrapper(request, *args, **kwargs):
            if request.method in ['POST', 'PUT']:
                allowed_ctypes = [cgi.parse_header(ct) for ct in ctypes]
                request_ctype, request_params = cgi.parse_header(
                    request.META.get('CONTENT_TYPE', ''))
                print('CTYPE:', request_ctype)
                print('AC:', allowed_ctypes)
                for ctype, params in allowed_ctypes:
                    if ctype == request_ctype and \
                            set(params.items()).issubset(
                                set(request_params.items())):
                        break
                else:
                    return HttpResponse(
                        'Unsupported media type. Expected: %s' % ', '.join(
                            ctypes), status=415, content_type='text/plain')
            return view(request, *args, **kwargs)

        return wrapper

    return decorator


def require_permission(*args, **kwargs):
    return lambda view: view


def require_read_permission(view):
    return view


def require_write_permission(view):
    return view


def get_pagination(request, max=200):
    try:
        page_no = int(request.GET.get('p', '1'))
        page_size = int(request.GET.get('count', 20))
    except ValueError:
        raise InvalidInputError('')
    if page_size > max:
        page_size = max
    if page_no < 1:
        raise OutOfBoundsError("")
    return (page_no, page_size)
