# -*- coding: utf-8 -*-

from urllib.parse import urlparse, urlunparse
from cgi import parse_qsl
from django.utils.http import urlencode


EXTERNAL_RELS = ['userpic', 'www', 'alternate', 'service']


class Link(object):
    def __init__(self, name, text='', attrs=None):
        self.name = name
        self.text = text
        self.attrs = attrs or {}


def update_hrefs(relations, format):
    new_relations = {}
    for rel, links in relations.items():
        if rel not in EXTERNAL_RELS:
            for link in links:
                if 'href' in link:
                    link['href'] = url_with_format(link['href'], format)
        new_relations[rel] = links
    return new_relations

def url_with_format(url, format):
    scheme, netloc, path, params, query, fragments = urlparse(url)
    parsed_query = dict(parse_qsl(query))
    parsed_query.update({'format': format})
    return urlunparse((scheme, netloc, path, params, urlencode(parsed_query), fragments))

def mk_json_links(relations, force_format):
    """
    >>> mk_json_links({'self': [{'href': 'hello'}, {'href': 'hello again'}], 'edit': [{'foo': 'bar'}], 'alternate': [{'href': 'hi', 'type': 'text/html'}, {'href': 'bye', 'type': 'application/octet-stream'}]})
    {'edit': {'foo': 'bar'}, 'self': ['hello', 'hello again'], 'alternate': [{'href': 'hi', 'type': 'text/html'}, {'href': 'bye', 'type': 'application/octet-stream'}]}
    """

    if force_format:
        relations = update_hrefs(relations, 'json')

    new_relations = {}
    for rel, links in relations.items():
        if len(links) == 1:
            link = links[0]
            if list(link.keys()) == ['href']:
                new_relations[rel] = link['href']
            else:
                new_relations[rel] = links[0]
        else:
            ls = []
            for link in links:
                if list(link.keys()) == ['href']:
                    ls.append(link['href'])
                else:
                    ls.append(link)
            new_relations[rel] = ls
    return new_relations

def mk_xml_links(xml, relations, force_format, namespace=''):
    if force_format:
        relations = update_hrefs(relations, 'xml')

    elem_name = 'link' if not namespace else '%s:link' % namespace
    for rel, links in relations.items():
        for link in links:
            new_link = {'rel': rel}
            new_link.update(link)
            xml.element(elem_name, attrs=new_link)


if __name__ == "__main__":
    import doctest
    doctest.testmod()
