# -*- coding: utf-8 -*-

import lxml.etree as et
import types
import re

def py2etree(entity, name, namespace=None):
    def _buildname(localname):
        return namespace and ('{%s}%s' % (namespace, localname)) or localname

    def _buildtree(parent, entity, object_attrs = False):

        if isinstance(entity, (bool, int, float, str)):
            parent.text = str(entity)
        elif isinstance(entity, (list, tuple)):
            if len(parent.tag) > 1 and parent.tag.endswith('s') and '{' not in parent.tag:
                item_name = parent.tag[:-1]
            else:
                item_name = 'item'
            list(map(lambda x: _buildtree(et.SubElement(parent, _buildname(item_name)), x), entity))
        elif isinstance(entity, dict):
            isidentifier = lambda x: isinstance(x, str) and x and not re.match(r'\d', x[0])
            if object_attrs:
                goodkey = lambda x: isidentifier(x) and not x.startswith('_')
            else:
                goodkey = isidentifier
            list(map(lambda x: _buildtree(et.SubElement(parent, _buildname(x)), entity[x]), list(filter(goodkey, iter(entity.keys())))))
        elif type(entity) is types.InstanceType:
            _buildtree(parent, entity.__dict__, True)

    e = et.Element(_buildname(name))
    _buildtree(e, entity)
    return e

def py2xml(entity, name='data', namespace=None, encoding='utf-8'):
    '''
    Переводит данные произвольного python объекта в xml представление.

    Объекты воспринимаются только как структуры данных.
    Действительное устройство объекта теряется, методы игнорируются.

    Обрабатываются численые типы, строки, последовательности.

    Поддержка dictionary весьма ограничена:
     * поддерживаются только строчные ключи
     * и только те, что начинаются не с цифры
    Записи с неподходящими ключами пропускаются.

    Также:
     * в  __dict__ python-объектов игнорируются записи с ключами,
       начинающимися с '_'
     * элементы последовательностей переводятся в последовательности
       XML-элементов 'item'

    XML генерируется без переводов строк и отступов (pretty print нет).

    Пример:
        class B:
            def __init__(self):
                self.tom = 5
        class A:
            def __init__(self):
                self.a = 1
                self.b = "tale"
                self.items = (1,2,[3,4])
                self.d = B()
        py2xml(A())

        получается (pretty print для наглядности):

        <data>
          <a>1</a>
          <items>
            <item>1</item>
            <item>2</item>
            <item>
              <item>3</item>
              <item>4</item>
            </item>
          </items>
          <b>tale</b>
          <d>
            <tom>5</tom>
          </d>
        </data>
    '''
    e = py2etree(entity, name, namespace)
    return et.tostring(e, encoding)

#
# number convertion are taken from django/template

# matches if the string is valid number
_number_re = re.compile(r'[-+]?(\d+|\d*\.\d+)$')

def _resolve_value(value):
    if value and _number_re.match(value):
        number_type = '.' in value and float or int
        try:
            return number_type(value)
        except ValueError:
            pass
    return value

def etree2py(root, single_as_list=True, value_getter=None):
    '''
    Конструирует python-структуру данных по XML.
    На выходе получается простой тип или tuple или dictionary.

    Если все child-элементы некоторого XML-элемента имеют одинаковые
    имена, то весь элемент считается списком и на выходе выдаётся tuple.
    Иначе элемент считается структурой и выдаётся dictionary.

    Если single_as_list == True, то элемент с единственным подъэлементом
    считается списком, иначе -- dictionary.
    '''

    def _strip_namespace(name):
        return name[name.find('}') + 1:]

    def _same_tag_children(node):
        children = node.getchildren()
        if len(children) == 1:
            return False
        for i in children:
            if node[0].tag != i.tag:
                return False
        return True

    def _walktree(node):
        if len(node) == 0:
            if value_getter:
                return value_getter(node)
            else:
                return _resolve_value(node.text)
        # if _same_tag_children(node) and (len(node) == 1 and single_as_list):
        if _same_tag_children(node):
            return tuple([_walktree(i) for i in node.getchildren()])
        else:
            return dict([(_strip_namespace(i.tag), _walktree(i)) for i in node.getchildren()])

    return _walktree(root)

def xml2py(xmltext, single_as_list=True):
    return etree2py(et.XML(xmltext), single_as_list)
