# -*- coding: utf-8 -*-
#
#       XML -> python dict-like struct conversion
#
#       NB: requires lxml
#
import datetime

__all__ = [ "NAMESPACES", "TYPES", "unxml", "unxml_list", "listify" ]

NAMESPACES = {
  "rdf"  : "http://www.w3.org/1999/02/22-rdf-syntax-ns#",
  "dc"   : "http://purl.org/dc/elements/1.1/",
  "atom" : "http://www.w3.org/2005/Atom",
  "foaf" : "http://xmlns.com/foaf/0.1/",
  "xhtml": "http://www.w3.org/1999/xhtml",
  "widget" : "http://www.netvibes.com/ns/",
}

from lxml import etree as ET

class TYPES:
    @classmethod
    def int(cls,s, item):
        return int(s)
    @classmethod
    def xml2str(cls, s, item):
        return cls.str(ET.tostring(item), item)
    @classmethod
    def str(cls,s, item):
        # do not force unicode string into byte string
        # convert nonascii byte string into unicode string
        if isinstance(s, str):
            return s
        try:
            return str(s)
        except UnicodeEncodeError:
            return str(s)
    @classmethod
    def bool(cls,s, item):
        return bool(s) # @TODO корректно сделать
    @classmethod
    def dt(cls,x, item):
        return datetime.datetime.strptime(x[:20], "%Y-%m-%dT%H:%M:%SZ")

def normalize_name( name ):
    parts = name.split("-")
    return parts[0] + "".join([x.capitalize() for x in parts[1:]])
    
def strip_tag_prefix(tagName):
    parts = tagName.split("}")
    return parts[ len(parts) -1 ]
    
def strip_attribute_prefix(attr):
    parts = attr.split("#")
    return parts[ len(parts) -1 ]

_sample_config = {
        "@create"   : "Social.Person",
        "id"        : TYPES.int,
        "updated"   : TYPES.dt,
        "links"     : ("atom:link", "rel", "href"),
        "content"   : {
                        "@root" : "./atom:content/y:data",
                        "qu" : TYPES.int
                        },
        "link"      : False,
        "generator" : False,
        }

def listify( suspect ):
    '''
    Converts suspect to list.
    '''
    if type(suspect) == list:
        return suspect
    elif type(suspect) == dict:
        return [suspect]
    else:
        return []

def unxml_list( Elist, config=None, namespaces=None ):
    '''
    For intrinsic use, converts list of E into None, one-element-dict, list-of-items
    '''
    if len(Elist) == 0:
        return None
    if len(Elist) == 1:
        return unxml( Elist[0], config=config, namespaces=namespaces )
    else:
        return [unxml( x, config=config, namespaces=namespaces ) for x in Elist]

def unxml(E, config=None, namespaces=None, normalize_names=False ):
    '''
    Convert all inlaid tags from E into recursive dict.
    Config is used to skip some tags, to custom conversion and to cut short.
    Supported specials in config:
      - "@root" in inner dict: XPath shortcut
      - other keys starting from "@" -- passed "as is" to result
      - string value -- pack XPath contents:
        - xpath returns string -- there would be string
        - xpath returns 1 element -- will be unxml-ed as dict
        - xpath returns N elemets -- there will be list of unxml-ed dicts
      - False value: skip this key
      - TYPES.* -- convert value of this key to given type
      - tuple (xpath,keyAttr,valueAttr) -- convert all found by xpath to inner dict where key=attrubute[keyAttr], value=attribute[valueAttr]
    '''
    if type(E) is str: 
        return E
    if namespaces==None:
        namespaces=NAMESPACES
    if config==None:
        config = dict()
    result = dict()
    #1. specials
    for (k,v) in list(config.items()):
        if type(v) == tuple:
            (tag,key,value) = v
            result[k] = dict()
            items = E.xpath( tag, namespaces=namespaces )
            for i in items:
                result[k][ strip_attribute_prefix(i.get(key))] = i.get(value)
        elif type(v) == str:
            if k[0] == "@":
                if k != "@root":
                    result[k] = v
            else:
                result[k] = unxml_list(E.xpath( v, namespaces=namespaces ), namespaces=namespaces)
        elif type(v) == dict:
            result[k] = unxml_list(E.xpath( v["@root"], namespaces=namespaces ), config=v, namespaces=namespaces)
        else:
            pass
    #2. all other stuff
    items = E.xpath( "./*", namespaces=namespaces )
    for item in items:
        v = item.text
        k = strip_tag_prefix(item.tag)
        if k in config:
            if type(config[k]) in (type(TYPES.int), type(TYPES.dt), type(TYPES.str)):
                v = config[k](v, item)
            else:
                v = None
        else:
            if v == None:
                v = unxml(item, namespaces=namespaces)
        if v != None:
            result[ k ] = v
    if normalize_names:
        _result = dict()
        for k in result:
            _result[ normalize_name(k) ] = result[k]
        return _result
    else:
        return result
