# -*- coding: utf-8 -*-
import datetime
from functools import wraps, partial

from at.common import exceptions
from at.common.groups import GroupType
from at.aux_ import Relations
from at.aux_.models import Person


def _has_additional(func, field):
    @wraps(func)
    def _(self, *args):
        if not getattr(self, field): # не hasattr, а именно непустое значение
            raise RuntimeError('%s called without %s specified' % (func.__name__, field))
        return func(self, *args)
    return _
post = partial(_has_additional, field='entry')
target = partial(_has_additional, field='target')

def _normal(func, who):
    @wraps(func)
    def _(self, *args):
        return getattr(self, who).status == 'normal' and func(self, *args)
    return _
normal_user = partial(_normal, who='user')
normal_feed = partial(_normal, who='feed')


def normal_community_or_any_diary(func):
    @wraps(func)
    def _(self, *args):
        if self.feed.is_community() and self.feed.status != 'normal':
            return False
        return func(self, *args)
    return _


def feed_is_user(func):
    @wraps(func)
    def _(self, *args):
        if self.feed.is_community():
            raise ValueError('Club can\'t follow users!') # не return False, потому что это скорее ошибка в коде
        return func(self, *args)
    return _

def target_is_club(func):
    @wraps(func)
    def _(self, *args):
        if not self.target or not self.target.is_community():
            raise ValueError('Target for %s should be a club' % func.__name__) # не return False, потому что это скорее ошибка в коде
        return func(self, *args)
    return _


class BaseAccess(object):
    def __str__(self):
        return ('Access<user=%d, feed=%d' % (self.user.person_id, self.feed.person_id)) + \
            (', entry=%s' % self.entry if self.entry else '') + \
            (', target=%s' % self.target.person_id if self.target else '') + '>'

    def __init__(self, user, feed, entry=None, target=None):
        if entry and not hasattr(entry, 'item_no'):
            # вместо not isinstance(entry, entries.entry), чтоб не тащить зависимость
            raise ValueError('%s given to Access instead of entry' % type(entry))
            # Была идея разрешать передавать item_no в поле entry, но тогда
            # не будут детектиться случаи, когда я забываю target передавать через
            # kwargs и пишу его третьим позиционным параметром. Так что лучше
            # всё-таки формировать модельку перед вызовом и передавать её
            # в конструктор явно.
        if target and not isinstance(target, (int, Person)):
            raise ValueError('%s given to Access instead of person' % type(target))

        resolve_ids = [value for value in (user, feed, target) if not isinstance(value, (Person, type(None)))]
        models = Person.get_persons(resolve_ids) if resolve_ids else {}
        # пытался сделать изящно через locals(), не прокатило
        try:
            self.user = user if isinstance(user, Person) else models[str(user)]
            self.feed = feed if isinstance(feed, Person) else models[str(feed)]
            self.target = target if isinstance(target, (Person, type(None))) else models[str(target)]
        except KeyError as e:
            error = 'unknown feed'
            if e.args:
                error = e.args[0]
            raise exceptions.NotFound('Not found feed %s' % error)
        if self.user.user_type == 'community':
            raise ValueError('club provided as user to Access')
        self.entry = entry

    @classmethod
    def _list_operations(cls):
        return [m.split('_', 2)[2] for m in dir(cls) if m.startswith('_can_')]

    def _assert(self, operation):
        def _(*args):
            if not getattr(self, '_can_'+operation)(*args):
                raise exceptions.AccessDenied(
                    "%s denied for %s" % (operation, self))
        return _

    def _can(self, operation):
        return getattr(self, '_can_'+operation)

    def __getattr__(self, key):
        if key.startswith('assert_can_'):
            return self._assert(key[11:])
        elif key.startswith('can_'):
            return self._can(key[4:])
        elif key.startswith('is_'):
            return lambda: self._is(key[3:])
        raise AttributeError(key)

    @property
    def roles(self):
        if not hasattr(self, '_roles'):
            self._roles = GroupType.expand_roles(self._load_roles())
        return self._roles

    def _is(self, role):
        if isinstance(role, str):
            role = GroupType.get_num(role)
        return role in self.roles

    def _load_roles(self):
        raise NotImplementedError

class RealAccess(BaseAccess):

    @property
    def roles(self):
        if not hasattr(self, '_roles'):
            self._roles = GroupType.expand_roles(self._load_roles())
        return self._roles

    def _load_roles(self):
        return [r.role for r in Relations.get_relations(self.user.person_id, self.feed.person_id)]

    @property
    def feed_target_access(self):
        if self.target is None:
            return None
        else:
            if not hasattr(self, '_feed_target_access'):
                self._feed_target_access = Access(self.feed, self.target)
            return self._feed_target_access

    @property
    def user_target_access(self):
        if self.target is None:
            return None
        else:
            if not hasattr(self, '_user_target_access'):
                self._user_target_access = Access(self.user, self.target)
            return self._user_target_access

    @property
    def tb_access(self):
        if not self.entry or not self.entry.tb_feed_id:
            return None
        if not hasattr(self, '_tb_access'):
            self._tb_access = Access(self.user.person_id, self.entry.tb_feed_id)
        return self._tb_access

    @normal_user
    @feed_is_user
    @target
    def _can_add_followship(self):
        """
        Проверяем, может ли user добавить target_id в ленту к feed.
        @param target_id: фид, который хотим добавить во френдленту
        @return: bool
        """
        # вообще-то не факт, что это ограничение нужно:
        # человек не может сам себя сделать членом приватного клуба и,
        # следовательно, получить доступ, но подписаться на него
        # (не имея доступа) – почему нет, строго говоря?..
        if self.target.is_community():
            if self.target.is_private_community():
                return self.feed_target_access.is_invited()
            else:
                return True
        else:
            return self.user == self.target

    @normal_user
    @feed_is_user
    @target
    def _can_remove_followship(self):
        return self._can_add_followship()

    @normal_user
    @feed_is_user
    @target
    def _can_add_friend(self):
        if self.feed_target_access.is_friend(): # ох уж эти чёртовы инвертированные отношения
            raise exceptions.OperationNotDone('AlreadyHave')
        return self.user == self.target

    @normal_user
    @feed_is_user
    @target
    def _can_remove_friend(self):
        return self.user == self.target

    @normal_user
    @normal_feed
    @feed_is_user
    @target
    def _can_ban(self):
        if self.target.is_community():
            # разрешаем модераторам банить других модераторов и запасаемся попкорном
            return self.user_target_access.is_moderator() # and not self.feed_target_access.is_moderator()
        else:
            return self.user == self.target

    @normal_user
    @normal_feed
    @feed_is_user
    @target_is_club
    def _can_invite(self, role):
        if self.feed_target_access._is(role) \
                or self.feed_target_access._is(GroupType.invite[role]):
            raise exceptions.OperationNotDone('AlreadyHave')
        return self.user_target_access.is_moderator()

    @normal_user
    @normal_feed
    @feed_is_user
    @target_is_club
    def __can_remove_from_group_impl(self, role):
        return self.user == self.feed \
            or self.user_target_access.is_moderator()


    def _can_remove_from_group(self, role=GroupType.FRIEND):
        if role == GroupType.BANNED:
            return self._can_ban()
        else:
            return self.__can_remove_from_group_impl(role)

    @normal_user
    @normal_feed
    @feed_is_user
    @target_is_club
    def _can_add_to_group(self, role):
        if self.feed_target_access._is(role):
            raise exceptions.OperationNotDone('AlreadyHave')
        if self.user_target_access.is_moderator():
            return True
        # важно, что проверка на feed.is_banned() идёт после проверки на user.is_moderator()
        # чтобы модератор мог своим инвайтом выводить человека из бана
        if self.feed_target_access.is_banned():
            return False
        if role == GroupType.MEMBER:
            return self.feed_target_access.is_invited() or not self.target.is_private_community()
        if role == GroupType.MODERATOR:
            return self.feed_target_access.is_moderator_invited()
        return False

    @normal_user
    @post
    def _can_read_post(self):
        if self.feed.status == 'deleted':
            return False

        # Костыль. Приглашённый должен иметь доступ к постам для членов,
        # но добавлять MEMBER в expand_role(INVITED) нельзя, потому что сломается
        # проверка групп при addGroup(), счётчики и т.д.
        # В будущем надо вообще избавиться от роли INVITED: чтоб приглашение в клуб фактически
        # сразу давало MEMBER, а человеку потом надо было только подписаться.
        roles = self.roles.copy()
        if GroupType.INVITED in roles:
            roles.add(GroupType.MEMBER)

        return self.user.person_id == self.entry.author_id or \
                self.entry.access_group in roles

    @normal_feed
    @normal_user
    def _can_tag_posts(self):
        return self.tb_access.__can_tag_post_impl() if self.tb_access \
                else self.__can_tag_post_impl()

    def __can_tag_post_impl(self):
        return self.is_moderator() or not self.no_user_tags

    @property
    def no_user_tags(self):
        from . import ProfileStorage
        if not hasattr(self, '__no_user_tags'):
            values = ProfileStorage.get_entry(
                    self.feed.person_id, 'OptionsOthers', 'no_user_tags')
            self.__no_user_tags = values and (values[0] == '1')
        return self.__no_user_tags

    def __has_access_to_feed(self):
        if self.feed.is_private_community():
            return self.is_member()
        else:
            return not self.is_banned()

    @normal_community_or_any_diary
    @normal_user
    def _can_comment(self):
        return self.__has_access_to_feed() and not self.entry.block_comments \
                and (not self.tb_access or self.tb_access.can_post())


    @normal_feed
    @normal_user
    # Может использоваться без entry, в смысле просто "может постить в этот фид"
    # поэтому не ставлю декоратор @post и добавляю условие not self.entry or ...
    def _can_post(self):
        if not self.feed.is_community():
            return self.feed == self.user

        if not self.__has_access_to_feed():
            return False
        if self.entry:
            if self.entry.type in ['rules', 'description'] and not self.is_moderator():
                return False
            if self.feed.is_premoderated():
                if not self.entry.on_moderation and not self.is_moderator():
                    return False
        return True

    def _can_save(self):
        if not self.entry.is_new:
            return self.can_edit()
        elif self.entry.is_comment:
            return self.can_comment()
        else:
            return self.can_post()

    @normal_feed
    @normal_user
    @post
    def _can_edit(self):
        if self.entry is None:
            raise RuntimeError('Called can_edit() without an entry')
        try:
            if self.entry.is_comment:
                assert self.entry.author_id == self.user.person_id
            else:
                assert self.is_moderator() or \
                        self.entry.author_id == self.user.person_id
        except AssertionError:
            return False
        if self.tb_access and not self.tb_access.can_edit():
            return False
        return True


    def _can_delete(self):
        # Кажется, в общем случае удаление это редактирование, а редактирование это can_edit
        return self._can_edit()

    @normal_user
    @post
    def _can_vote(self):
        if not self.can_read_post():
            return False
        if self.entry.expires and self.entry.expires < datetime.date.today():
            return False
        return True

    # NB: @normal_feed не требуем – восстановление клубов работает через эту проверку
    @normal_user
    def _can_update_profile(self):
        if self.feed.user_type == 'community':
            return self.feed.status in ['normal', 'deleted'] and self.is_moderator()
        else:
            return self.user == self.feed

    @normal_user
    def _can_get_friendlist(self):
        # Да, так и есть, доступ без ограничений
        return True

    @normal_user
    def _can_read_blog(self):
        return not self.feed.is_private_community() or self.is_invited()


Access = RealAccess # for monkeypatching with FakeAccess


class FakeAccess(BaseAccess):
    def __init__(self, user, feed, entry=None, _fake_roles=[GroupType.MEMBER]):
        super(FakeAccess, self).__init__(user, feed, entry)
        self._fake_roles = _fake_roles

    def _load_roles(self):
        return self._fake_roles

    def _can_read_blog(self):
        return True

    @post
    def _can_read_post(self):
        return True

    def _can_post(self):
        return True

    def _can_save(self):
        return True

    def _can_comment(self):
        return True

    @post
    def _can_edit(self):
        return True

    def _can_tag_posts(self):
        return True

    def _can_update_profile(self):
        return True

    def _can_get_friendlist(self):
        return True

    def _can_delete(self):
        return True

    def _can_add_followship(self):
        return True

    def _can_remove_followship(self):
        return True

    def _can_add_friend(self):
        return True

    def _can_remove_friend(self):
        return True

    def _can_ban(self):
        return True

    def _can_vote(self):
        return True

    def _can_invite(self):
        return True

    def _can_remove_from_group(self, group=GroupType.FRIEND):
        return True

    def _can_add_to_group(self, group=GroupType.FRIEND):
        return True
