# -*- coding: utf-8 -*-

import logging

import lxml.etree as ET

from at.aux_ import Entries
from at.aux_.models import Person
from at.aux_.BlogsAuxiliary import BlogsAuxiliaryMix
from at.aux_ import XDecorators
from at.aux_ import Tags
from at.aux_ import Relations
from at.common import exceptions
from django.conf import settings
from at.common import groups
from at.common import utils

_log = logging.getLogger(__name__)

TagsFacade = Tags.Tags()


def _createApiPostImpl(
        ai,
        feed_id=None,
        item_no=None,
        parent_id=None,
        type_default=None,
        meta=None,
        do_trackback=False,
        params=None,
        comment_id=None):
    params = utils.params2dict(params or [])

    def error_if_invalid_type(name, value):
        if value not in Entries.META_TYPES + Entries.NOMETA_TYPES:
            raise exceptions.InvalidParams(
                'Unsupported or not implemented %s value: "%s"' % (name, value))

    # * type должен быть указан
    # * type не валидный -- ошибка
    # * filter_type и meta_type, если не указаны, выводятся из type
    # * filter_type, meta_type не валидные -- ошибка

    if type_default is None:
        raise exceptions.InternalError('Default post type must be specified')
    error_if_invalid_type('type', type_default)

    filter_type = params.pop('filter_type', type_default)
    error_if_invalid_type('filter_type', filter_type)

    explicit_meta_type = params.get('meta_type', None)
    meta_type = params.pop('meta_type', type_default)
    error_if_invalid_type('meta_type', meta_type)

    # * тип структуры meta должен соответствовать meta_type (указанному или
    # выведенному из type_default)
    #
    # * meta is None -- meta_type должен быть "text" или "status" или что там ещё без меты
    # * meta is not None, meta_type не указан, тип meta не соответствует type --
    #  -- ошибка (или meta_type могла бы выводиться из типа meta)

    if meta_type in Entries.NOMETA_TYPES and meta is not None:
        raise exceptions.InvalidParams(
            'meta_type "%s" requires no meta structure at all' % meta_type)
    if meta is None and meta_type not in Entries.NOMETA_TYPES:
        raise exceptions.InvalidParams(
            'meta_type "%s" requires meta structure' % meta_type)

    if meta is not None and meta_type not in Entries.META_TYPES:
        if explicit_meta_type is None:
            raise exceptions.InvalidParams(
                'Meta type conflict (meta=%s, meta_type=%s): meta_type was implicitly inferred from type, if type of meta is correct, meta_type must be specified explicitly' % (
                meta.__class__, meta_type))
        else:
            raise exceptions.InvalidParams(
                'Meta type conflict (meta=%s, meta_type=%s)' % (
                meta.__class__, meta_type))
    feed_info = Person.get_person(feed_id)
    if feed_info.community_type in ('CLOSED_COMMUNITY',
                                    'PREMODERATED_COMMUNITY'):
        params['premoderated'] = 'to_moderate'
    post = Entries.Post(meta_type)(
        auth_info=ai,
        feed_id=feed_id,
        item_no=item_no,
        parent_id=parent_id,
        comment_id=comment_id,
        store_time=int(params.pop('store_time', None) or 0),
        access_type=params.pop('access_type', None),
        freezed=bool(params.pop('freezed', 0)),
        tags=params.pop('tags', ''),
        do_trackback=do_trackback,
        post_type=type_default,
        filter_type=filter_type,
        meta_type=meta_type,
        meta=meta,
    )
    post.createItemContent(params)
    return post.do_post()


class Api(XDecorators.XDecorators):
    @utils.log_exception
    def CreateComment(self,
                      ai,
                      feed_id,
                      item_no,
                      parent_id,
                      post_type,
                      meta=None,
                      trackback=None,
                      params=None,
                      escape_html=False):
        params = utils.params2dict(params)
        if "comment_id" in params:
            # Если редактируем существующий комментарий,
            # проверить авторство
            comment_id = int(params.pop("comment_id"))
            aux = BlogsAuxiliaryMix()
            old_comment = ET.XML(
                aux.GetSingleComment(ai, feed_id, item_no, comment_id, escape_html=escape_html))
            try:
                current_author_id = int(
                    old_comment.find('item/item/author/uid').text)
                assert current_author_id == ai.uid
            except AssertionError:
                raise exceptions.AccessDenied("%s can't edit comment %s.%s.%s" %
                                              (ai.uid, feed_id, item_no,
                                               comment_id))
        else:
            comment_id = settings.UNDEFINED_COMMENT_ID
        new_comment_id = _createApiPostImpl(
            ai=ai,
            feed_id=feed_id,
            item_no=item_no,
            parent_id=parent_id,
            type_default=post_type,
            meta=meta,
            do_trackback=trackback,
            params=list(params.items()),
            comment_id=comment_id
        )
        aux = BlogsAuxiliaryMix()
        return aux.GetSingleComment(ai, feed_id, item_no, new_comment_id, escape_html=escape_html)

    @utils.log_exception
    def CreatePost(
            self,
            ai,
            feed_id,
            post_type,
            meta=None,
            params=None
        ):
        item_no = _createApiPostImpl(
            ai=ai,
            feed_id=feed_id,
            item_no=None,
            parent_id=None,
            type_default=post_type,
            meta=meta,
            do_trackback=False,
            params=params
        )
        aux = BlogsAuxiliaryMix()
        return aux.GetPostSafe(ai, feed_id, item_no)

    def _get_likes(self, ai, likes_key):
        try:
            likes_string = self.deco_GetLikeCounts(ai, likes_key)
            if b"error" in likes_string:
                raise ValueError("Invalid likes response")
            else:
                return ET.fromstring(likes_string)
        except Exception:
            _log.exception("Error while parsing likes {}".format(likes_key))
            return None

    @utils.log_exception
    def GetPost(self, ai, feed_id, item_no):
        likes_key = "%s.%s" % (feed_id, item_no)
        try:
            aux = BlogsAuxiliaryMix()
            post = aux.GetPostSafe(ai, feed_id, item_no)
        except exceptions.NotFound:
            logging.warning('Post %s not found', (ai, feed_id, item_no))
            return ''
        likes = self._get_likes(ai, likes_key)
        if likes and len(likes):
            xml_post = ET.fromstring(post)
            item = xml_post.find("./item")
            if item:
                item.append(likes)
            post = ET.tostring(xml_post)
        return post

    @utils.log_exception
    def GetPosts(self, ai, source_id, escape_html, post_types_str,
                 cat_id, count, tb, min_store_time, max_store_time):
        aux = BlogsAuxiliaryMix()
        feeds = aux.GetFeed2Light(ai,
                                  source_id,
                                  escape_html,
                                  post_types_str,
                                  cat_id,
                                  count,
                                  tb,
                                  min_store_time,
                                  max_store_time)
        feeds = ET.fromstring(feeds)
        feeds_ids = feeds.xpath("feed/item/id")
        as_string_ids = feeds.xpath("feed/item/id/asString/node()")
        if len(feeds_ids) != len(as_string_ids):
            _log.warning(
                "for source_id '%s' feeds_ids len != as_string_ids len",
                source_id)
        likes_key = ','.join(as_string_ids)
        if likes_key:
            likes = self._get_likes(ai, likes_key)
        else:
            likes = None
        if likes is not None and len(likes):
            for item in feeds.xpath("feed/item"):
                item.append(ET.Element("likes"))
                item.find("likes").append(
                    likes.find(
                        "like[@id='%s']" % item.xpath("id/asString/node()")[0])
                )
        return ET.tostring(feeds)

    @utils.log_exception
    def GetTagsByFeed(self, feed_id):
        tags = TagsFacade.getTags(feed_id)[0]
        answer = ET.Element("club-tags", {"feed-id": str(feed_id)})
        for (_, tag_id, tag_name) in tags:
            answer.append(ET.Element("tag", {"id": str(tag_id),
                                             "name": tag_name.decode('utf-8')}))
        return ET.tostring(answer)

    @utils.log_exception
    def GetUsersByGroup(self, feed_id, role='FRIEND', page=1, count=20,
                        typ='blogs'):
        # XXX в нынешнем виде работает не как friends/friend-of/mutual,
        # а как friends/friend-of невзаимоисключительно
        role = role.upper()
        if page < 1:
            page = 1
        if role == 'FRIEND-OF':
            role, inverted = groups.GroupType.FRIEND, True
        else:
            role, inverted = groups.GroupType.get_num(role), typ == 'clubs'
        relations = Relations.list_feed_relations(feed_id, role, None, count,
                                                  page - 1, inverted)
        total = Relations.get_total_relations_counters(feed_id, role, inverted)[
            role]
        root = ET.Element('feeds', {'total': str(total), 'page': str(page),
                                    'page-size': str(count)})
        for uid in [(r.feed_id if inverted else r.uid) for r in relations]:
            ET.SubElement(root, 'feed', {'feed-id': str(uid)})
        return ET.tostring(root, encoding='utf-8')

    @utils.log_exception
    def GetMembers(self, feed_id, page=1, count=20):
        return self.GetUsersByGroup(feed_id, 'member', page, count)

    @utils.log_exception
    def GetModerators(self, feed_id, page=1, count=20):
        return self.GetUsersByGroup(feed_id, 'moderator', page, count)

    @utils.log_exception
    def GetFriends(self, feed_id, page=1, count=20):
        return self.GetUsersByGroup(feed_id, 'friend', page, count)

    @utils.log_exception
    def GetFrienders(self, feed_id, page=1, count=20):
        return self.GetUsersByGroup(feed_id, 'friend-of', page, count)

    @utils.log_exception
    def GetClubs(self, feed_id, page=1, count=20):
        return self.GetUsersByGroup(feed_id, 'member', page, count, 'clubs')

    @utils.log_exception
    def GetModeratedClubs(self, feed_id, page=1, count=20):
        return self.GetUsersByGroup(feed_id, 'moderator', page, count, 'clubs')

    @utils.log_exception
    def GetOwnedClubs(self, feed_id, page=1, count=20):
        return self.GetUsersByGroup(feed_id, 'owner', page, count, 'clubs')


test_label = """
<conversion id="пошёл куда все">
    <one gender="m">пошёл куда все</one>
    <one gender="w">пошла куда все</one>
    <one gender="u">пошло куда все</one>
    <some>пошли куда все</some>
    <many>пошли куда все</many>
</conversion>
"""


def test_api(ai, text, label=test_label, template=''):
    post = Entries.Post('text')(ai,
                                post_type='offline',
                                display={'label': label,
                                         'template': template,
                                         })
    post.createItemContent({'body': text})
    return post.do_post()
