# -*- coding: utf-8 -*-

"""Yandex.Blogs python servant"""

# Common and third-party libraries
import time
import logging

import lxml.etree as ET

from at.common import utils
from at.common import dbswitch
from at.common import ServiceAccess
from at.common.groups import GroupType
from at.common import isearch
from at.aux_ import Accesses
from at.aux_ import Items
from at.aux_ import ClubsSuggest
from at.aux_ import Community
from at.aux_ import Entries
from at.aux_ import feeds
from at.aux_ import FeedInfo
from at.aux_ import Friends
from at.aux_ import Moderation
from at.aux_ import Mailing
from at.aux_ import Profile
from at.aux_ import models
from at.aux_ import RecentActions
from at.aux_ import Relations
from at.aux_ import Sources
from at.aux_ import Statistics
from at.aux_ import Tags
from at.aux_ import Trackers
from at.aux_ import UserEvents
from at.aux_ import XDecorators

# all handlers should be imported so that they are registered
from at.pump import Handlers

_log = logging.getLogger(__name__)


def node(name, *content):
    el = ET.Element(name)
    for i in content:
        el.append(i)
    return el


def text(name, text):
    el = ET.Element(name)
    el.text = text
    return el


def iso_time(t):
    return time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime(t))


class BlogsAuxiliaryMix(
    Profile.Profile,
    Friends.Friends,
    Items.Items,
    RecentActions.RecentActions,
    Tags.Tags,
    Community.Community,
    Statistics.Statistics,
    Mailing.Mailing,
    Moderation.Moderation,
    XDecorators.XDecorators,
    UserEvents.UserEvents,
    Trackers.Trackers,
    FeedInfo.FeedInfo,
    feeds.Feeds,
    ClubsSuggest.ClubsSuggest
):

    def Ping(self):
        return 1

    def PingAsString(self):
        return "Ok"

    def PingAsXML(self):
        return '<OK/>'

    @utils.log_exception
    def GetFeedID(self, slug):
        try:
            return "<feed_id>%s</feed_id>" % models.GetClubID(slug)
        except Exception as e:
            error = 'Unknown error'
            if e.args:
                error = e.args[0]
            return "<error>%s</error>" % error

    @utils.log_exception
    def Authenticate(self, uid):
        has_access = ServiceAccess.check_accessibility(uid)
        return '<intranet-auth>%s</intranet-auth>' % \
               ('OK' if has_access else 'Forbidden')

    @utils.et2xml
    @utils.log_exception
    def GetMlSuggest(self, ai, part):
        results = isearch.get_maillist_suggest(auth=ai, text=part)
        root = ET.Element('suggests')
        for result in results:
            ET.SubElement(root, 'suggest').text = result['id']
        return ET.ElementTree(root)

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def GetCurrentBirthdays(self, ai):
        return Sources._getCurrentBirthdays(ai)

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def GetPostPreview(self, ai, request):
        req = utils.req2dict(request)
        try:
            feed_id = int(req.get('feed_id', ai.uid))
            item_no = req.get("item_no") and int(req.get("item_no"))
            trackback = int(req.get('trackback') or 0)
            post_type = req.get('type') or 'text'
            escape_html = True if req.get('escape_html') else False
            post = Entries.Post(post_type)(
                auth_info=ai,
                feed_id=feed_id,
                item_no=item_no,
                access_type=req.get("access_type", None),
                freezed=not (req.get("replies") or True),
                tags=req.get('tags', '') + ',' + req.get('tag', ''),
                do_trackback=trackback,
                tb_feed_id=req.get('tb_feed_id')
                )
            post.createItemContent(req, preview=True)
            result = post.getPostPreview(escape_html=escape_html)
            if result[-6:] != b'</aux>':
                return (b'<aux hostname="%s">' % utils._hostname.encode('utf-8')) + utils.Status('Success') + result + b"</aux>"
            return result[:-6] + utils.Status('Success') + result[-6:]
        except Exception:
            _log.exception('GetPostPreview failed with request %s' % req)
            raise


    @utils.et2xml
    @utils.log_exception
    def FriendsDoAddByUID(self, ai, request):
        uid = request['uid']
        with dbswitch.root_rw_session():
            Friends.friendAndFollow(ai, uid, ai.uid)
        return utils.Status('OK')

    @utils.et2xml
    @utils.log_exception
    def FriendsDoAddBulcByUID(self, ai, request, uid_arg_name='user'):
        with dbswitch.root_rw_session():
            new_friends = request['uids'].split(',')
            for uid in new_friends:
                try:
                    Friends.friendAndFollow(ai, uid, ai.uid)
                except:
                    pass
        return utils.Status("Success")


    @utils.et2xml
    @utils.log_exception
    def FriendsDoDeleteByUID(self, ai, request):
        person_id = request['person_id']
        with dbswitch.root_rw_session():
            Friends.unfriendAndUnfollow(ai, person_id, ai.uid)
        return utils.Status("Success")

    @utils.et2xml
    @utils.log_exception
    def FriendsDoDeleteBulcByUID(self, ai, request, uid_arg_name='user'):
        old_friends = request['uids'].split(',')
        with dbswitch.root_rw_session():
            for person_id in old_friends:
                try:
                    Friends.unfriendAndUnfollow(ai, person_id, ai.uid)
                except: pass
        return utils.Status("Success")

    @utils.log_exception
    def LogRotate(self):
        utils.reopen_log(self.name)


    @utils.et2xml
    @utils.log_exception
    def AllCommunityBlock(self, ai, feed_id, members_count, moderators_count, page_num):
        access = Accesses.Access(ai.uid, feed_id)
        access.assert_can_get_friendlist()
        counters = Relations.get_total_relations_counters(feed_id,
                [GroupType.MODERATOR, GroupType.MEMBER])
        R = Relations.RelationsList
        data = {
            'members': (counters[GroupType.MEMBER], R()),
            'moderators': (counters[GroupType.MODERATOR], R())
            }
        method = Relations.list_feed_relations
        if members_count:
            data['members'][1].extend(
                method(feed_id, GroupType.MEMBER, None, members_count, page_num)
            )
        if moderators_count:
            data['moderators'][1].extend(
                method(feed_id, GroupType.MODERATOR, None, moderators_count, page_num)
            )
        return ET.ElementTree(Relations.serialize_grouplist(data))

    @utils.et2xml
    @utils.log_exception
    def AllFriendsBlock(self, ai, feed_id, friends_count, want_friends_count, friendees_count, page_num):
        access = Accesses.Access(ai.uid, feed_id)
        access.assert_can_get_friendlist()
        friends_counters = Relations.get_relations_counters(feed_id)
        R = Relations.RelationsList
        data = {
            'friends-mutual': (friends_counters[(GroupType.FRIEND, 1)], R()),
            'friends': (friends_counters[(GroupType.FRIEND, 0)], R())
            }
        friendees_counters = Relations.get_relations_counters(feed_id, inverted=True)
        data['friend-of'] = (friendees_counters[(GroupType.FRIEND, 0)], R())
        # TODO
        method = Relations.list_feed_relations
        if friends_count:
            data['friends-mutual'][1].extend(
                method(feed_id, GroupType.FRIEND, True, friends_count, page_num)
            )
        if want_friends_count:
            data['friends'][1].extend(
                method(feed_id, GroupType.FRIEND, False, want_friends_count, page_num)
            )
        if friendees_count:
            data['friend-of'][1].extend(
                method(feed_id, GroupType.FRIEND, False, friendees_count, page_num, True)
            )
        return ET.ElementTree(Relations.serialize_grouplist(data))

    def SendDigest(self):
        from at.pump import Digest
        Digest.create_digest_events()
        return 'ok'
