# -*- coding: utf-8 -*-


import re
import logging

import lxml.etree as ET

from django.conf import settings

import at.aux_.models
from at.common import exceptions
from at.common import assertions
from at.common import dbswitch
from at.common.groups import GroupType
from at.common.Rubrics import Rubrics
from at.common import utils
from at.aux_ import Accesses
from at.aux_ import Friends
from at.aux_ import Profile
from at.aux_ import ProfileStorage
from at.aux_ import Relations
from at.aux_.models import Person
from at.aux_.CommunityOptions import *
from at.pump import HandlerRegistry

_log = logging.getLogger(__name__)

clubs_forbidden = {
    'names': [],
    'regexp': []
}


def _community_check_login_impl(login):
    if not re.match("^[a-z][a-z0-9-]*[a-z0-9]$", login):
        return "BadChars"

    if len(login) < 2 or len(login) > 30:
        return "BadLength"

    if login in clubs_forbidden['names']:
        return "ForbiddenName"

    for regexp in clubs_forbidden['regexp']:
        if regexp.search(login):
            return "ForbiddenRegexp"

    # FIXME вообще-то это проверка должна делаться ключом в persons:
    if Person.get_by(login=login, user_type='community'):
        return "Busy"
    else:
        return None


def update_community_options(feed_id, rubric_id, adult):
    sql = """
      update Communities
      set
      rubric_id = :rubric_id,
      adult = :adult
      where feed_id = :feed_id
    """
    conn = get_connection()
    conn.execute(sql, {
        'feed_id': feed_id,
        'rubric_id': rubric_id,
        'adult': adult,
    })
    conn.execute(
        "insert ignore CommunitiesStat (feed_id) values (:feed_id)", {
            'feed_id': feed_id,
        }
    )


def update_interests(feed_id, interests):
    interest_set = set(interest.strip() for interest in interests.split(','))
    interest_items = [
        ('interest', interest, 'text')
        for interest in interest_set if interest
    ]
    try:
        ProfileStorage.put(feed_id, "ClubsInterests", interest_items)
    except Exception:
        _log.exception('Failed to save club interests')


def update_community(model, req):
    RBAC_MAP = {
        # is_premoderated, is_private
        (False, False): 'OPENED_COMMUNITY',
        (False, True): 'CLOSED_COMMUNITY',
        (True, False): 'PREMODERATED_COMMUNITY',
        (True, True): 'PREMODERATED_CLOSED_COMMUNITY'
    }
    is_premoderated = req['premoderated']
    is_private = req['private']
    community_type = RBAC_MAP[(is_premoderated, is_private)]
    model.community_type = community_type
    Profile.update_model(model, req)
    model.save()

    update_community_options(feed_id=model.person_id, rubric_id=req['rubric_id'],
                             adult='private' if req['is_adult'] else 'public'
                             )

    # XXX хорошо бы их тоже в транзакцию втащить
    update_interests(model.person_id, req.get('interest', ''))


def get_alive_moderators(feed_id):
    return [
        rel.uid for rel in
        Relations.list_feed_relations(feed_id, GroupType.MODERATOR)
    ]


class Community:
    def updateCommunity(self, ai, feed_id, req):
        access = Accesses.Access(ai.uid, feed_id)
        access.assert_can_update_profile()
        assert access.feed.user_type == 'community'
        with dbswitch.root_rw_session():
            model = Person.get_person(feed_id)
            was_private = model.is_private_community()
            update_community(model, req)
            is_private = model.is_private_community()

        if was_private != is_private:
            old_access_type, new_access_type = ('public', 'members') if is_private else ('members', 'public')
            HandlerRegistry.put_event(
                'ChangePostsAccessesHandler',
                feed_id,
                old_access_type,
                new_access_type
            )

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def UpdateCommunityReq(self, ai, request):
        feed_id = request.pop('feed_id')
        self.updateCommunity(ai, feed_id, request)
        return ET.ElementTree(ET.XML('<new-club><uid>%s</uid></new-club>' % feed_id))

    def createCommunity(self, ai, slug, req):
        slug = slug.strip()
        user = Person.get_person(ai.uid)
        # не вынес эту проверку в Accesses, т.к. там предполагается пара (user, feed):
        if user.status != 'normal':
            raise exceptions.AccessDenied(
                '%s user %s tries to create a club'
                % (user.status, user.login)
            )
        with dbswitch.root_rw_session():
            err = _community_check_login_impl(slug)
            assertions.assertion(err is None, 'name', 'NOT_VALID', err)
            cursor = get_connection()
            cursor.execute('insert into CommunitiesSequence values ()')
            new_feed_id = cursor.execute('select LAST_INSERT_ID()').fetchone()[0]
            # FIXME надо бы добавить уникальный ключ на (login, user_type)
            # и при duplicate key error кидать corba_assert(0, 'name', NOT_VALID, 'Busy')
            model = Person()
            model.person_id = new_feed_id
            model.login = slug
            model.user_type = 'community'
            cursor.execute('insert into Communities (feed_id, name)' \
                           ' values (:new_feed_id, :login)',
                           {'new_feed_id': model.person_id, 'login': model.login}
                           )

            update_community(model, req)

            at.aux_.models.Friend(uid=user, person_id=model, fgroup_id=GroupType.MODERATOR).save()
            at.aux_.models.Follower(uid=user, person_id=model).save()

            HandlerRegistry.put_event('CommunityStatHandler', new_feed_id)
            return model.person_id

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def CreateCommunityReq(self, ai, request):
        slug = request.pop('slug')
        feed_id = self.createCommunity(ai, slug.strip(), request)
        return ET.ElementTree(ET.XML('<new-club><uid>%s</uid></new-club>' % feed_id))

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def GetCommunityOptions(self, host_person_id):
        c = CommunityStat.select([host_person_id])
        root = ET.Element("community-options")
        if c:
            c[0].append_to(root)
        return ET.ElementTree(root)

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def GetRubricsList2(self, ai, host_person_id):
        member_of = None
        if utils.is_community_id(host_person_id):
            c = CommunityStat.select([host_person_id])
            if c: member_of = c[0].rubric_id

        lang = ProfileStorage.get_language(ai.uid)
        root = ET.Element("community_catalog")
        for rub in Rubrics().filter_hidden(0):
            node = rub.to_elem(lang)
            if member_of == rub.rubric_id:
                node.attrib['containt_community'] = str(host_person_id)
            root.append(node)
        return ET.ElementTree(root)

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def GetCommunitiesRubrics2(self, ai):
        sql = """select c.rubric_id, count(c.feed_id)
        from Communities as c, persons as p
        where
            p.community_type not in ('PREMODERATED_CLOSED_COMMUNITY', 'CLOSED_COMMUNITY') and
            p.person_id = c.feed_id and
            p.status='normal'
        group by c.rubric_id"""
        with utils.get_connection() as con:
            counts = dict((r[0], r[1]) for r in con.execute(sql))
        lang = ProfileStorage.get_language(ai.uid)
        root = ET.Element("community_catalog")
        for rub in Rubrics().filter_hidden(0):
            try:
                rub.c_count = counts.get(rub.rubric_id, 0)
            except KeyError:
                continue
            else:
                root.append(rub.to_elem(lang))
        return ET.ElementTree(root)

    @utils.stopwatch
    @utils.et2xml
    @utils.log_exception
    def GetRubricWithPager(self, rubric_id, page, count):
        try:
            rubric = Rubrics().by_rubric_id(rubric_id)
            root = rubric.to_elem('ru')  # XXX fix language
            return get_public_communties_with_pager(root, page, count, 'rating', rubric_id)
        except IndexError:
            return ET.ElementTree(ET.XML(utils.Status("NotFound")))

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def GetUserCommunitiesPagedLight(self, ai, feed_id, page_num=0, page_size=-1):
        role = GroupType.MEMBER
        # обратная совместимость:
        count = 0 if page_size == -1 else page_size
        access = Accesses.Access(ai.uid, feed_id)
        access.assert_can_get_friendlist()
        root = ET.Element('communities', {
            'page': str(page_num),
            'page-size': str(page_size),
            'total': str(Relations.get_total_relations_counters(
                feed_id, role, inverted=True, mutually=False)[role])
        })
        order = 'fgroup_id DESC'
        for relation in Relations.list_feed_relations(feed_id, role, mutually=False,
                                                      count=count, page=page_num, inverted=True, order_by=order):
            comm = ET.SubElement(root, 'community')
            ET.SubElement(comm, 'feed_id').text = str(relation.feed_id)
        return ET.ElementTree(root)

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def InviteToGroup(self, ai, request):
        community_feed_id = request.pop('community_feed_id')
        person_id = request.pop('person_id')
        group_name = request.pop('group_name')
        with dbswitch.root_rw_session():
            try:
                Friends.inviteToGroup(ai=ai, uid=person_id, feed_id=community_feed_id,
                                      group_id=GroupType.get_num(group_name))
                return utils.Status("OK")
            except exceptions.AccessDenied:
                return utils.Status("AccessDenied")
            except exceptions.NotFound:
                return utils.Status("NotFound")
            except exceptions.OperationNotDone:
                logging.exception('InviteToGroup failed')
                return utils.Status("NotDone")

    @utils.et2xml
    @utils.log_exception
    def AcceptModerationInvitation(self, ai, request):
        community_feed = request.pop('community_feed')
        group = GroupType.MODERATOR

        try:
            with dbswitch.root_rw_session():
                Friends.joinCommunity(ai, ai.uid, community_feed, group)
        except exceptions.AccessDenied:
            return utils.Status("AccessDenied")
        except exceptions.NotFound:
            return utils.Status("NotFound")
        except exceptions.OperationNotDone:
            return utils.Status("AlreadyDone")

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def RemoveFromCommunityGroup(self, ai, request):
        community_feed = request.pop('community_feed')
        person_ids = request.pop('person_ids')
        group = request.pop('group')
        uids = [int(i) for i in person_ids.split(',') if i.isdigit()]
        done, failed = 0, 0
        old_role = GroupType.get_num(group)
        with dbswitch.root_rw_session():
            for uid in uids:
                try:
                    # с т.з. производительности нехорошо, что мы столько всего тащим из базы,
                    # когда можно было бы накостылить проверок типа ai.is_moderator or ai.uid == uid,
                    # но зато так стройнее, а вызовов этой ручки с большим списком uid-ов должно
                    # быть исчезающе мало.
                    access = Accesses.Access(ai.uid, uid, target=community_feed)
                    access.assert_can_remove_from_group(old_role)
                    Relations.removeRelation(uid, community_feed, old_role)
                    done += 1
                except Exception as e:
                    _log.warn('removeFromGroup(%s) not done: %s' % ((ai, uid, community_feed, old_role), e))
                    failed += 1
        # XXX
        return utils.Status('OK' if done > 0 else 'Failed')

    @staticmethod
    def _count_community_users(feed_id):
        return Relations.get_total_relations_counters(
            feed_id, [GroupType.MEMBER]
        )[GroupType.MEMBER]

    def _change_community_status(self, ai, feed_id, require_status, new_status):
        access = Accesses.Access(ai.uid, feed_id)
        if not access.can_update_profile():
            assertions.assertion(False, 'user', 'ACCESS_DENIED', 'post')
        assert access.feed.status == require_status
        assert self._count_community_users(feed_id) <= settings.MAX_MEMBERS_CLUB_DELETE
        return Profile._change_blog_status_impl(feed_id, status=new_status)

    @utils.et2xml
    @utils.log_exception
    @utils.check_auth(1)
    def DeleteCommunity(self, ai, request):
        feed_id = request.pop('feed_id')
        return self._change_community_status(ai, feed_id, 'normal', 'deleted')

    @utils.et2xml
    @utils.log_exception
    @utils.check_auth(1)
    def RestoreCommunity(self, ai, request):
        feed_id = request.pop('feed_id')
        return self._change_community_status(ai, feed_id, 'deleted', 'normal')

    @utils.et2xml
    @utils.log_exception
    def GetCommunityMemberCount(self, feed_id):
        root = ET.Element('club_member_count')
        root.set('delete_limit', str(settings.MAX_MEMBERS_CLUB_DELETE))
        root.text = str(self._count_community_users(feed_id))
        return ET.ElementTree(root)

    @utils.et2xml
    @utils.log_exception
    def GetMyDeletedCommunities(self, ai):
        access = Accesses.Access(ai.uid, ai.uid)
        access.assert_can_get_friendlist()
        relations = Relations.list_feed_relations(ai.uid, [GroupType.MODERATOR],
                                                  inverted=True, status='deleted')
        root = relations.to_xml('deleted_clubs', 'community')
        return ET.ElementTree(root)
