# -*- coding: utf-8 -*-



import logging

import lxml.etree as ET

from at.common.Rubrics import *
from at.common import dbswitch
from at.common.utils import is_community_id, get_connection
from at.common.groups import GroupType


_log = logging.getLogger(__name__)


FGROUPS_NAMES = {
    GroupType.OWNER:'OWNER',
    GroupType.MODERATOR:'MODERATOR',
    GroupType.MEMBER:'MEMBER',
}


class CommunityStat:
    def __init__(self, feed_id, members_count, posts_count,
                  last_comment, last_post,
                  adult=None, rubric_id=None,
                  rating=None
                  ):
        self.feed_id = feed_id
        self.members_count = members_count
        self.posts_count = posts_count
        self.last_comment = last_comment
        self.last_post = last_post
        self.rubric_id = rubric_id
        self.metarubric = Rubrics().by_rubric_id(self.rubric_id).metarubric
        self.adult = adult
        self.rating = rating
        self.role = None

    def append_to(self, parent):
        d = dict((field, getattr(self, field)) for field in \
                ["feed_id", "members_count", "posts_count", "last_comment",
                "rubric_id", "metarubric", "adult"]
        )
        if self.role is not None:
            d['role_type'] = FGROUPS_NAMES[self.role]
            d['role_type_num'] = self.role

        c_el = ET.SubElement(parent,"community")
        for tag, val in d.items():
            if val is not None:
                e = ET.SubElement(c_el, tag)
                e.text = str(val or 0)

    @staticmethod
    def select(id_list, as_dict=False):
        id_seq = {id for id in id_list if is_community_id(id)}
        if not id_seq:
            return {} if as_dict else []

        sql_t = """
            select
                cs.feed_id,cs.members_count,cs.posts_count,
                unix_timestamp(cs.last_comment_time),
                unix_timestamp(cs.last_post_time),
                c.adult, c.rubric_id, c.rating
            from CommunitiesStat as cs join Communities as c
              on (cs.feed_id=c.feed_id)
            where cs.feed_id in (%s)
        """
        sql = sql_t % ",".join([str(i) for i in id_seq])
        _log.debug(sql)

        cur = get_connection().execute(sql)
        stats = {row[0]: CommunityStat(*row) for row in cur}
        return stats if as_dict else list(stats.values())


def get_public_communties_with_pager(root, page, count, order_type, in_rubric=None ):
    orders = {
        'rating': 'c.rating desc',
        'recent': 'c.feed_id desc',
    }
    assert order_type in orders, 'unknown order type'
    with get_connection() as conn:
        # Клубы по приглашениям не входят в рейтинг
        cur = conn.execute("""select sql_calc_found_rows feed_id
        from Communities as c, persons as p
        where
            %s
            p.person_id = c.feed_id
            and p.status = 'normal'
            and p.community_type not in ('PREMODERATED_CLOSED_COMMUNITY', 'CLOSED_COMMUNITY')
        order by %s
        limit %d,%d"""%("c.rubric_id=%d and"%in_rubric if in_rubric is not None else "",orders[order_type],page*count, count))

        feeds = [f[0] for f in cur]
        cur = conn.execute('select found_rows()')
        overall_count =  [x[0] for x in cur] [0]

        comms = CommunityStat.select(feeds, as_dict=True)
        for feed_id in feeds:
            if feed_id in comms:
                comms[feed_id].append_to(root)
            else:
                _log.warn('No community stats for %s!' % feed_id)

        root.set("overall_count",str(overall_count)) # will be deprecated
        root.set("total",str(overall_count))
        root.set("page-size", str(count))
        root.set("page",str(page)) # will be deprecated
        root.set("count",str(count))

        return ET.ElementTree(root)

