# -*- coding: utf-8 -*-


import datetime
import copy
import hashlib
import itertools
import logging
import time
from xml.sax.saxutils import escape
from functools import cmp_to_key
from lxml import etree as ET

from at.common import exceptions
from at.common import assertions
from at.common.groups import AccessLevel
from at.common import postlink
from at.common import utils
from at.common import dbswitch
from at.common import Maillists
from at.common.Rubrics import Rubrics, get_rubric, NoRubric
from at.common.utils import (
    stopwatch, is_community_id, force_unicode, force_str, get_connection
)
from at.aux_ import entries
from at.aux_ import Accesses
from at.aux_ import Friends
from at.aux_ import Profile
from at.aux_ import ProfileStorage
from at.aux_ import Tags
from at.aux_ import Mailing
from at.aux_.Trackers import STARTREK
from at.aux_.CommunityOptions import CommunityStat
from at.aux_.models import Person
from at.pump import HandlerRegistry

_log = logging.getLogger(__name__)

META_TYPES = [
    'link',
    'friend',
    'unfriend',
    'join',
    'unjoin',
    'congratulation',
]

NOMETA_TYPES = [
    'text',
    'status',
    'question',
    'answer',
]

LOGGER_REG_CALLBACK_MESSAGE = 'Register new callback | Post_id: {}.{}.{} | Callback {}'

# === content utils ===

def SubElementWithText(parent, tag, text, attribs=None):
    el = ET.SubElement(parent, tag, attribs or {})
    el.text = force_unicode(text)
    return el


def prepareRequest(orig_req, kw):
    req = copy.copy(orig_req)
    req.update(kw)
    for key, val in req.items():
        req[key] = force_unicode(val)
    return req


# ===============================

def return_post_xml(item_no, blog_id):
    root = ET.Element("aux")
    root.append(ET.XML(utils.Status("Success")))
    ET.SubElement(root, 'new-post-id').text = str(item_no)
    ET.SubElement(root, 'feed-id').text = str(blog_id)
    retpath = ET.SubElement(root, "retpath")
    retpath.text = postlink.make_url(blog_id, item_no)
    _log.debug('return to: %s' % retpath.text)
    return ET.ElementTree(root)


PRIVATE_ACCESS_TYPES = [
    AccessLevel.PRIVATE,
    AccessLevel.MODERATORS,
]

CC_FIELDS = ['carbon-copy-address', 'carbon-copy-address-input']


class ItemContent(object):
    def __init__(self, typecode, item_time, xml):
        self.typecode = typecode
        self.item_time = item_time
        self.xml = xml


class Entry(object):
    meta_type_default = None
    filter_type_default = None
    _v_fields = []
    fields = [
        'item_time', 'item_time_year', 'item_time_month', 'item_time_day',
        'item_time_hour', 'item_time_minute',
        'rss_guid', 'type', 'userpic'
    ]

    def __init__(self,
                 auth_info,
                 feed_id=None,
                 item_no=None,
                 parent_id=None,
                 comment_id=None,
                 store_time=0,
                 access_type=None,  # string or None
                 freezed=False,  # disable comments
                 tags='',  # comma separated string of tag names
                 link='',
                 do_trackback=False,
                 tb_feed_id=None,
                 base_url='',

                 # these fields are for API:
                 post_type=None,
                 filter_type=None,
                 meta_type=None,
                 meta=None,
                 ):
        self.callbacks = []
        self.auth_info = auth_info
        self.feed_id = feed_id if (feed_id is not None) else auth_info.uid
        self.item_no = item_no
        self.parent_id = parent_id
        self.comment_id = comment_id
        self.store_time = store_time

        # DIARY-360
        # If link is available, try to fetch post_no, to see if post being updated.
        # Correct is_editing flag is required to avoid duplicate CCNewPost events.
        if self.item_no is not None and link:
            link_key = link
            if not isinstance(link_key, bytes):
                link_key = link_key.encode('utf-8')
            with get_connection() as conn:
                curs = conn.execute(
                    'select item_no from PostStoreData where feed_id = %s and link_md5 = %s',
                    (self.feed_id, hashlib.md5(link_key).digest()))
                if curs.rowcount:
                    self.item_no, = curs.fetchone()

        if self.comment_id is None and self.parent_id is None:
            self.is_a_comment = False
        else:
            self.is_a_comment = True

        # в этом стремном коде местами придется ориентироваться на is_editing,
        # потому что is_new превращается из True в False после model.save()
        if self.is_a_comment:
            self.is_editing = self.comment_id is not None
        else:
            self.is_editing = self.item_no is not None

        if self.is_editing:
            _log.debug("Will load %s",
                       (self.feed_id, self.item_no, self.comment_id or 0))
            self.model = entries.load_entry(self.feed_id, self.item_no,
                                            self.comment_id or 0)
        else:
            kls = entries.models.get_post_class(self.post_type)
            self.model = kls(self.feed_id, self.item_no, self.comment_id)
        self.access_manager = Accesses.Access(self.auth_info.uid, self.feed_id,
                                              self.model)
        # requires access_manager:
        self.set_access_type(access_type)

        _log.debug("is_new = %s" % self.model.is_new)
        _log.debug("is_comment = %s" % self.model.is_comment)

        self.tags = Tags.split(force_unicode(tags))
        # Костыльно, но пока сойдет
        # Вообще надо переделать работу с link, выкинуть imported_from, rss_guid и тд
        self.link = '' if self.is_editing else link
        self.do_trackback = do_trackback
        if do_trackback:
            self.model.do_trackback = do_trackback
            self.model.tb_feed_id = self.tb_feed_id = int(
                tb_feed_id or self.auth_info.uid)
        self.item_content = None
        self.base_url = base_url or None
        if self.base_url is None:
            model = Person.get_person(self.feed_id)
            if is_community_id(self.feed_id):
                self.base_url = 'http://clubs.at.yandex-team.ru/%s/' % model.login
            else:
                self.base_url = 'http://%s.at.yandex-team.ru' % model.login
        self.preview_mode = False
        # these fields are for API:
        if post_type:
            self.post_type = post_type
        self.meta_type = meta_type or self.meta_type_default or self.post_type
        self.filter_type = filter_type or self.filter_type_default or self.post_type
        self.meta = meta
        self.model.block_comments = freezed if \
            self.access_manager.is_moderator() else False
        self.tb_item_content = ItemContent(0, 0, '')

    def set_access_type(self, access_type):
        self.access_type = access_type
        # бессмысленная санитизация доступов
        # правильнее написать corba_assert, что доступы поставлены корректные,
        # а не корректировать их на ходу
        if self.access_manager.feed.is_community():
            if self.access_manager.feed.is_private_community():
                if self.access_type != AccessLevel.MODERATORS:
                    self.access_type = AccessLevel.MEMBERS
            else:
                if self.access_type != AccessLevel.MODERATORS:
                    self.access_type = AccessLevel.PUBLIC
        else:
            if self.access_type in (
            AccessLevel.MODERATORS, AccessLevel.MEMBERS):
                self.access_type = AccessLevel.PRIVATE

    def save_meta(self, req):
        pass

    def load_meta(self):
        """Загружает meta из item_content.

        Если do_post в производном классе обращается к мете, необходимо
        переопределить этот метод и загрузить ее из item_content.

        Требуется для сохранения записей, загруженных через _load_post.
        """

    def validate(self, req):
        self.validate_fields(req)

    def validate_fields(self, req):
        for f in self._v_fields:
            assertions.assertion(req.get(f, '').strip(), f, 'NOT_EMPTY', self.post_type)

    def create_content_impl(self, req):
        """ Here you can also make additional changes to attributes etc. """
        # Default: valid for text, rules (with changes)
        self.model.content_type = "text/wiki"
        self.model.title = req.get("title", "").strip()
        self.model.body = req.get("body", "").strip()

    def set_cc_addresses(self, req):
        if self.access_type == AccessLevel.ALL_FRIENDS:
            return
        if all(key not in req for key in CC_FIELDS):
            if self.model.is_new:
                self.model.cc_address = None
            else:
                # в model.cc_address уже лежит что надо
                pass
            return
        else:
            addresses = itertools.chain(*(
                    req.get(field, '').split(',') for field in CC_FIELDS
                    ))
            # не делаю таких проверок для данных из профиля,
            # т.к. предполагаю, что они уже валидны (а если вдруг нет,
            # то тут-то пользователь всё равно не может их поправить).
            addrlist = [_f for _f in (a.strip() for a in addresses) if _f]
            assertions.assertion(
                all(map(utils.validate_email_address, addrlist)),
                'carbon-copy-address',
                'NOT_VALID',
                'post',
            )
        self.model.cc_address = ','.join(addrlist)

    def set_post_rubric(self, req):
        self.rubric = None

        request_rubric = req.get('rubric')
        if request_rubric:
            self.rubric = Rubrics().by_metarubric(req['rubric'])
        elif self.model.is_comment:
            if self.do_trackback:
                try:
                    self.rubric = get_rubric(self.feed_id, self.item_no)
                except NoRubric:
                    if is_community_id(self.feed_id):
                        c = CommunityStat.select([self.feed_id])
                        if c.adult:
                            self.rubric = Rubrics().by_metarubric('hidden')
                        else:
                            self.rubric = Rubrics().by_rubric_id(c.rubric_id)
                    else:
                        self.rubric = Rubrics().by_metarubric('work')
        if self.rubric is None:
            self.rubric = Rubrics().by_metarubric('work')
        return self.rubric

    def set_pinned_after(self, req):
        self.model.pinned_after_old = self.model.pinned_after
        if req.get('pinned'):
            self.model.pinned_after = int(req.get('pinned_after'))
        else:
            self.model.pinned_after = None

    def calculate_on_moderation(self):
        if self.model.is_comment:
            # коменты не бывают премодерируемыми
            return False

        feed_is_permoderated = self.access_manager.feed.is_premoderated()
        author_is_moderator = self.access_manager.is_moderator()
        # единственный случай, когда пост становится на модерацию —
        # немодератор создает или реадктирует его в премодерируемом клубе
        # (однако интерфейсно редактирование после апрува запрещено почему-то)
        if feed_is_permoderated and not author_is_moderator:
            return True

        return False

    @stopwatch
    def createItemContent(self, req, preview=False):
        self.preview_mode = preview
        req = prepareRequest(req, {})  # делает всё юникодом
        _log.debug('REQUEST: ' + str(req))

        on_moderation_before = bool(self.model.on_moderation)
        on_moderation_after = self.calculate_on_moderation()
        self.model.on_moderation = on_moderation_after
        passed_moderation = on_moderation_before and not on_moderation_after
        marked_for_moderation = not on_moderation_before and on_moderation_after

        if passed_moderation:
            self.model.moderated_by = self.auth_info.login
            self.callbacks.append(self.callback_moderate)

        if self.model.is_new:
            self.model.form_id = req.get('form_id', '')

        self.save_meta(req)
        self.set_cc_addresses(req)
        self.set_post_rubric(req)
        self.set_pinned_after(req)

        if self.model.is_new:
            self.model.store_time = datetime.datetime.now()
            self.model.author_id = self.auth_info.uid
        self.model.item_time = datetime.datetime.now()

        self.validate(req)
        self.create_content_impl(req)

        access_type_before = self.model.access_type
        if marked_for_moderation:
            self.model.original_access_type = self.access_type
            self.model.access_type = AccessLevel.MODERATORS
            if self.model.is_new:
                self.callbacks.append(self.callback_premoderate)
        else:
            self.model.access_type = self.access_type
        became_visible = (
            (
                access_type_before in PRIVATE_ACCESS_TYPES
                or
                self.model.is_new
            )
            and
            self.model.access_type not in PRIVATE_ACCESS_TYPES
        )

        if became_visible:
            # если один раз уже были разосланы письма и прочее, то не
            # ничего делаем
            if self.model.published:
                return
            self.model.published = True
            self.model.store_time = datetime.datetime.now()
            self.callbacks.append(self.callback_publish)
            self.callbacks.append(self.callback_cc)

        if self.access_manager.can_tag_posts():
            self.model.tags = self.tags

        self.model.entry_type = self.model.type

        if self.do_trackback:
            if self.model.is_new:
                tb = self.access_manager.tb_access
                if tb.feed.is_premoderated() and not tb.is_moderator():
                    self.model.tb_access_type = AccessLevel.MODERATORS
                    self.callbacks.append(self.callback_tb_premoderate)
                else:
                    self.model.tb_access_type = AccessLevel.PUBLIC

    def callback_edit(self):
        _log.info(LOGGER_REG_CALLBACK_MESSAGE.format(self.feed_id, self.item_no, self.comment_id, 'EditHandler'))
        HandlerRegistry.put_event('EditHandler',
                self.feed_id, self.model.item_no, self.model.comment_id or 0)

    def callback_premoderate(self):
        _log.info(LOGGER_REG_CALLBACK_MESSAGE.format(self.feed_id, self.item_no, self.comment_id, 'PremoderateHandler'))
        HandlerRegistry.put_event('PremoderateHandler',
                self.model.author_id, self.model.feed_id, self.model.item_no)

    def callback_tb_premoderate(self): # XXX copy-pasted
        _log.info(LOGGER_REG_CALLBACK_MESSAGE.format(self.feed_id, self.item_no, self.comment_id, 'tb_PremoderateHandler'))
        HandlerRegistry.put_event('PremoderateHandler',
                self.model.author_id, self.model.tb_feed_id, self.model.tb_item_no)

    def callback_moderate(self):
        _log.info(LOGGER_REG_CALLBACK_MESSAGE.format(self.feed_id, self.item_no, self.comment_id, 'ModerateHandler'))
        HandlerRegistry.put_event('ModerateHandler', self.feed_id, self.item_no)

    def callback_publish(self):
        _log.info(LOGGER_REG_CALLBACK_MESSAGE.format(self.feed_id, self.item_no, self.comment_id, 'PublishHandler'))
        HandlerRegistry.put_event('PublishHandler',
                self.model.author_id, self.model.feed_id, self.model.item_no,
                self.model.comment_id or 0, self.model.store_time.isoformat())

    def getPostPreview(self, escape_html):
        serialized = self.model.serializer.serialize_for_preview(escape_html=escape_html)
        return serialized

    @stopwatch
    def do_post(self):
        self.access_manager.assert_can_save()
        if not self.access_manager.is_moderator():
            # рядовой пользователь может постить пост с доступом "для модераторов"
            # только тогда, когда это либо пост на модерацию, либо он редактирует свой пост,
            # который модераторы перед тем перевели в "для модераторов".
            if self.access_type == AccessLevel.MODERATORS:
                assert self.model.on_moderation or not self.model.is_new

        if self.rubric:
            if self.do_trackback:
                self.model.tb_rubric_id = self.rubric.rubric_id
            else:
                self.model.rubric_id = self.rubric.rubric_id

        if self.model.is_new:
            self.model.parent_comment_id = self.parent_id or 0
        try:
            _log.debug('Before save into mysql: %s', self.model.__dict__)
            self.model.save()
        except exceptions.AccessDenied:
            assertions.assertion(False, 'user', 'ACCESS_DENIED', 'post')
        self.callbacks.append(self.callback_edit)

        self.finalize()

        if not self.is_editing and self.do_trackback:
            # Subscribe to this trackback those guys that subscribed to original post.
            Mailing.Subscriptions.PropogateSubscriptions(self.feed_id,
                                                         self.item_no,
                                                         self.model.tb_feed_id,
                                                         self.model.tb_item_no)

        return self.model.comment_id if self.model.is_comment else self.model.item_no

    do_comment = do_post

    def finalize(self):
        for cb in self.callbacks:
            cb()
        return True

    def callback_cc(self):
        """Send carbon-copies of the post directly to emails from cc_address.
        cc_address may be empty.
        If cc_address is None, load list of addresses from Profile.
        """
        if self.model.is_comment and not self.do_trackback:
            return

        if self.do_trackback:
            tb = self.access_manager.tb_access
            if tb.feed.is_premoderated() and not tb.is_moderator():
                return
            feed_id, item_no = self.model.tb_feed_id, self.model.tb_item_no
        else:
            feed_id, item_no = self.model.feed_id, self.model.item_no

        if self.model.cc_list is None:
            # Если трекбек, то адрес cc берем из профиля целевого фида
            # Уведомление отсылаем о посте-трекбеке, а не об оригинальном посте
            cc_list = ProfileStorage.get_entry(feed_id, 'CarbonCopy',
                                               'carbon-copy-address')
        else:
            cc_list = self.model.cc_list
        for cc in set(cc_list):
            _log.info(LOGGER_REG_CALLBACK_MESSAGE + '| CC: {}'.format(
                self.feed_id,
                self.item_no,
                self.comment_id,
                'CCFinalizeHandler',
                cc
            ))
            HandlerRegistry.put_event('CCFinalizeHandler', feed_id,
                                                            item_no,
                                                            self.model.author_id,
                                                            cc)


# ====================  Post types ==========================


class Text(Entry):
    _v_fields = ["body"]
    post_type = 'text'


class Description(Text):
    post_type = 'description'


class Link(Entry):
    post_type = 'link'

    def save_meta(self, req):
        url = req.pop('URL', '')
        try:
            sane_url = utils.sanitize_url(url).strip()
        except AssertionError:
            assertions.assertion(False, 'URL', 'NOT_EMPTY', self.post_type)

        self.model.url = sane_url
        # XXX did not put it into meta BECAUSE FUCK YOU
        # in fact, if we decide that shared_post_id should always refer to shared URL,
        # then it's fine; but then we shouldn't take it from req but parse the URL itself.
        # But if we want to pass down the shared_post_id from multi-level shared ancestor,
        # then we should also pass it through API.
        parts = None
        if 'shared_post_id' in req:
            try:
                parts = [int(_) for _ in req['shared_post_id'].split('.')]
                assert len(parts) in [2, 3], 'Invalid shared_post_id="%s"' % \
                                             req['shared_post_id']
                if len(parts) == 2:
                    parts.append(0)
            except Exception as e:
                _log.error('Failed to parse shared_post_id="%s": %s' % (
                req['shared_post_id'], e))
        else:
            try:
                parts = postlink.MyUrlParser().match_item(sane_url)
            except Exception:
                _log.exception('Failed to parse %s with MyUrlParser', sane_url)

        if parts:
            self.model.shared_feed_id = parts[0]
            self.model.shared_item_no = parts[1]
            self.model.shared_comment_id = parts[2]


class Poll(Entry):
    _v_fields = ['title']
    post_type = 'poll'

    def save_meta(self, req):
        def _cmp(a, b):
            a = int(a.replace('option', ''))
            b = int(b.replace('option', ''))
            return (a > b) - (a < b)
        opt_keys = [opt for opt in list(req.keys()) if opt.startswith('option')]
        opt_keys.sort(key=cmp_to_key(_cmp))
        try:
            expires = utils.parse_date(req.get('expires'))
        except ValueError:
            expires = None
        hidden = bool(req.get('hidden')) if expires else False # не имеет смысла без expires
        self.model.expires = expires
        self.model.hidden = hidden
        self.model.poll_type = req.get('poll_type')
        self.model.options = [req[k] for k in opt_keys if req[k].strip()]
        try:
            _poll_, poll_feed_id, poll_key = req.get('poll_id').split('_')
            assert _poll_ == 'poll'
            assert int(poll_feed_id) == self.feed_id
            int(poll_key)
            poll_id = '_'.join([_poll_, poll_feed_id, poll_key])
        except:
            poll_id = 'poll_%s_%s' % (self.feed_id, int(time.time()))
        self.model.poll_id = poll_id

        _log.debug('After save_meta %s', self.model.__dict__)


class Rules(Text):
    post_type='rules'
    def create_content_impl(self, req):
        if not is_community_id(self.feed_id):
            raise exceptions.AccessDenied("can't publish rules to blog")
        self.model.block_comments = True # Правила не обсуждать!
        Text.create_content_impl(self, req)


class Status(Entry):
    post_type = 'status'

    def create_content_impl(self, req):
        super(Status, self).create_content_impl(req)
        new_mood = self.model.snippet
        Profile.UpdateSource(self.auth_info, {'mood': force_str(new_mood)})


class AbstractFriending(Entry):
    def save_meta(self, req):
        super(AbstractFriending, self).save_meta(req)
        assertions.assertion('user' in req, 'user', 'NOT_EMPTY', self.post_type)
        assertions.assertion(req['user'].isdigit(), 'NOT_VALID', self.post_type)

    def wrap(self, fnc, *args):
        "Непонятно, как из Blogger кинуть осмысленный RoleAccessDenied"
        try:
            fnc(*args)
        except exceptions.AccessDenied:
            assertions.assertion(False, 'access', 'NOT_NULL', self.post_type)
        except:
            raise

    def side_effects_impl(self):
        pass

    def do_post(self):
        if self.model.is_new:
            self.side_effects_impl()
        try:
            return Entry.do_post(self)
        finally:
            if hasattr(self, 'callback'):
                self.callback()

    do_comment = do_post  # переопределили do_post – надо переопределить и do_comment


class Friendship(AbstractFriending):
    def save_meta(self, req):
        super(Friendship, self).save_meta(req)
        self.model.friender = self.auth_info.uid
        self.model.friendee = int(req['user'])

    def load_meta(self):
        entry = ET.XML(self.item_content.xml)
        self.model.friender = int(entry.findtext('meta/friender'))
        self.model.friendee = int(entry.findtext('meta/friendee'))

    def side_effects_impl(self):
        with dbswitch.root_rw_session():
            method = Friends.friendAndFollow if self.post_type == 'friend' else Friends.unfriendAndUnfollow
            self.wrap(method, self.auth_info, self.model.friendee,
                      self.model.friender)


class Friend(Friendship):
    post_type = 'friend'


class Unfriend(Friendship):
    post_type = 'unfriend'


class Membership(AbstractFriending):
    def save_meta(self, req):
        super(Membership, self).save_meta(req)
        self.model.person = self.auth_info.uid
        self.model.club = int(req['user'])

    def load_meta(self):
        entry = ET.XML(self.item_content.xml)
        self.model.person = int(entry.findtext('meta/friender'))
        self.model.club = int(entry.findtext('meta/friendee'))

    def side_effects_impl(self):
        access = Accesses.Access(self.model.person, self.model.club)
        club = access.feed
        club_id, user_id = access.feed.person_id, access.user.person_id
        assertions.assertion(club.is_community(), 'user', 'NOT_VALID', self.post_type)
        is_member = access.is_member()
        if self.post_type == 'join':
            assertions.assertion(not is_member, 'user-already', 'NOT_VALID', self.post_type)
            with dbswitch.root_rw_session():
                self.wrap(Friends.joinCommunity, self.auth_info, user_id, club_id)
        elif self.post_type == 'unjoin':
            assertions.assertion(is_member, 'user-already', 'NOT_VALID', self.post_type)
            # Отложим выход из клубов по приглашениям на потом, иначе
            # не получится опубликовать запись (в таких клубах могут
            # писать только их члены).
            if access.feed.is_private_community():
                def delay_callback():
                    with dbswitch.root_rw_session():
                        self.wrap(Friends.unjoinFromCommunity,
                                  self.auth_info, user_id, club_id)

                self.callback = delay_callback
            else:
                with dbswitch.root_rw_session():
                    self.wrap(Friends.unjoinFromCommunity,
                              self.auth_info, user_id, club_id)
                # Unsubscribe the user from everything he could've subscribed
                Mailing.Mailing.SetBlogSubscription(self.auth_info, club_id,
                                                    False, False)


class Join(Membership):
    post_type = 'join'


class Unjoin(Membership):
    post_type = 'unjoin'


class Summon(Entry):
    post_type = 'summon'

    def save_meta(self, req):
        if self.meta is None:
            ml_name = user = None
            # "summon__name" is in case of autocompleted maillist,
            # contains only name in case of y-t lists, full email otherwise:
            if 'summon__name' in req:
                ml_name = req['summon__name']
                if '@' not in ml_name:
                    ml_name += '@yandex-team.ru'
            # "summon_login" is in case of autocompleted user:
            elif 'summon__login' in req:
                user = req['summon__login']
            # only "summon" and none of the above in case of non-autocompleted value:
            else:
                value = req.get('summon')
                if value:
                    if '@' in value:
                        ml_name = value
                    else:
                        user = value
            assertions.assertion(ml_name or user, 'user', 'NOT_EMPTY',
                                 self.post_type)

            if ml_name:
                assertions.assertion(Maillists.is_email_ml(ml_name), 'user',
                                     'NOT_NULL', 'friend')
                self.model.summon_address = ml_name
            else:
                try:
                    uid = utils.getAuthInfo(any=user, request=req).uid
                    self.model.summon_uid = uid
                except:
                    assertions.assertion(False, 'user', 'NOT_NULL', 'friend')

    def finalize(self):
        super(Summon, self).finalize()
        if self.is_editing:
            # XXX NB: Вообще-то тут правильнее использовать что-то про published
            # (отсылать уведомление надо в момент публикации), но для комментов это одно и то же
            return True
        try:
            target = self.model.summon_uid or self.model.summon_address
            assert target
        except AssertionError:
            _log.warn('No valid summon meta in post: %s' % self.meta)
            return True
        HandlerRegistry.put_event('SummonFinalizeHandler',
            self.auth_info.uid, self.model.feed_id, self.model.item_no,
            self.model.comment_id, target
        )


class Congratulation(Entry):
    post_type = 'congratulation'

    def create_content_impl(self, req):
        event = req.get('event', 'newyear')
        try:
            whom = utils.getAuthInfo(any=req.get('whom', ''), request=req).uid
        except (exceptions.NotFound, utils.blackbox.BlackboxError):
            whom = None
        self.model.title = ''
        self.model.body = req.get("body", "").strip()
        self.model.event = escape(event)
        self.model.whom = whom


class TrackerEntry(Entry):
    # Не буду переименовывать тип, чтобы не сломать старые тикеты
    post_type = 'jira'

    def make_body(self, body):
        url = postlink.make_url(self.feed_id, self.item_no, self.parent_id)
        return 'Тикет создан через этушку в ответ на %s\n\n %s' % (url, body)

    def save_meta(self, req=None):
        project = req['project']
        if not project.strip():
            project = req['queue_suggest']

        assertions.assertion(req['title'], 'title', 'NOT_EMPTY', 'jira')
        if not self.preview_mode:
            # интерфейс сейчас позволяет заводить тикеты только в ответ, но мало ли:
            if self.model.is_comment:
                parent_entry = entries.load_entry(
                    self.feed_id, self.item_no, self.parent_id)
                copy_to = Person.get_person(parent_entry.author_id).login
            else:
                copy_to = None
            issueKey = STARTREK.create_issue(
                project,
                self.auth_info,
                req['assignee'],
                req['title'],
                self.make_body(req['body']),
                copy_to)
        else:
            issueKey = project + '-????'
        self.model.issue = issueKey


# =======================================

post_classes = {}

for postclass in [cls for cls in locals().values() if \
                  isinstance(cls, type) and \
                          issubclass(cls, Entry) and hasattr(cls, 'post_type')]:
    post_classes[postclass.post_type] = postclass


def Post(key):
    """ posttype-to-postclass register. Is filled from Entries.py ."""
    return post_classes[key]
