# encoding: utf-8


import logging
import time
import traceback
from functools import wraps

from at.common import exceptions
from at.common import decorators
from at.aux_.models import Person
from at.common import dbswitch
from at.common import utils

_log = logging.getLogger(__name__.lower())


def raise_internal(raise_as_is=(exceptions.NotFound,)):
    def inner_wrapper(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except Exception as e:
                if type(e) in raise_as_is:
                    raise
                raise exceptions.InternalError(traceback.format_exc())

        return wrapper

    return inner_wrapper


class FeedInfo:
    @utils.log_exception
    @raise_internal([])
    def GetFeedInfoBulcXMLStr(self, feed_id_str):
        @decorators.or_none
        def str_to_long(el):
            return int(el.strip())

        feeds = [_f for _f in map(str_to_long, feed_id_str.split(',')) if _f]
        persons = Person.get_persons(set(feeds))
        return b'<sources>%s</sources>' % b''.join(
            to_xml(p) for p in list(persons.values()))

    @utils.log_exception
    @raise_internal()
    def GetFeedInfoXML(self, feed_id):
        from datetime import datetime
        try:
            before = datetime.now()
            person = Person.get_person(feed_id)
            passed = datetime.now() - before
            msg = 'GetFeedInfoXML trace get_person in %d.%06d'
            _log.debug(msg, passed.seconds, passed.microseconds)
            before = datetime.now()
            xml = to_xml(person)
            passed = datetime.now() - before
            msg = 'GetFeedInfoXML trace to_xml in %d.%06d'
            _log.debug(msg, passed.seconds, passed.microseconds)
            return xml
        except exceptions.NotFound:
            return utils.build_xml(('error', 'NOT_FOUND'))
        except Exception:
            return utils.build_xml(('error', traceback.format_exc()))


def to_xml(person):
    def _is_read_only():
        return 'no'

    def _is_master_down():
        return 'no'

    def _get_regular_person_fields():
        if person.bd_year:
            bd_year = person.bd_year + 1900
        else:
            bd_year = None
        return [('qu', person.qu),
                ('mood', person.mood),
                ('bd_year', person.bd_year),
                ('bd_month', person.bd_month),
                ('bd_day', person.bd_day),
                ('email', person.email)]

    def _get_community_fields():
        types = {
            'PREMODERATED_CLOSED_COMMUNITY': ['private', 'premoderated'],
            'PREMODERATED_COMMUNITY': ['premoderated'],
            'CLOSED_COMMUNITY': ['private']
        }
        subtree = [('community_type', person.community_type)]
        subtree.extend(
            [(el, '') for el in types.get(person.community_type, [])])
        return subtree

    def _get_xml_inflections():
        if not utils.is_community_id(person.person_id):
            subtree = [('inflection', {'case': inf}, value) for (inf, value)
                       in list(person.inflections.items())]
        else:
            subtree = []
        return [('title-inflections', subtree)]

    def _date_convert(value):
        if value is None:
            return 0
        return int(time.mktime(value.timetuple()))

    children = [('id', person.person_id),
                ('login', person.login),
                ('login-lower', person.login.lower()),
                ('title', person.title),
                ('title_eng', person.title_eng),
                ('sex', person.sex),
                ('picture', person.picture_html),
                ('picture_url', person.picture_url),
                ('default_picture', person.default_picture),
                ('modification_date', _date_convert(person.modification_date)),
                ('creation_date', _date_convert(person.creation_date)),
                ('status', person.status),
                ('type', person.user_type),
                ('metatype', person.metatype),
                ('master_url', person.login)]
    children.extend(_get_xml_inflections())
    if utils.is_community_id(person.person_id):
        children.extend(_get_community_fields())
    else:
        children.extend(_get_regular_person_fields())
    tree = \
        ('source', {'read-only': _is_read_only(),
                    'all-read-only': _is_master_down()},
         children)
    return utils.build_xml(tree)
