# -*- coding: utf-8 -*-



import logging

import lxml.etree as ET

from django.conf import settings

import at.aux_.models
from at.common import utils
from at.common.groups import GroupType
from at.common.utils import Status, log_exception, get_connection, stopwatch
from at.common import intrafriend
from at.common import dbswitch
from at.aux_ import Accesses
from at.aux_ import models
from at.aux_ import Relations
from at.aux_ import UserEvents
from at.pump import HandlerRegistry

_log = logging.getLogger('aux/friends')


def friendship_sort_uid_list(person_id, uids):
    sql = '''
            select uid, fgroup_id
            from FriendGroupMember
            where person_id = %%s and fgroup_id in (-1, 0) and uid in (%s)
            '''

    with get_connection() as connection:
        cursor = connection.execute(
            sql % (', '.join(str(u) for u in uids) or '0'), (person_id,))
        friends = dict((uid, group) for uid, group in cursor)
    return sorted(((uid, friends.get(uid)) for uid in uids),
                  key=lambda uid_group: (
                  uid_group[1] if uid_group[1] is not None else 1, uid_group[0]))


def update_mutually(uid, person_id):
    # взаимность имеет смысл только для дружбы
    update_mutually_sql = "CALL FriendGroupMember_update_mutually_2(:person_id, :uid, :fgroup_id)"  # мне плевать, что там внутри
    get_connection().execute(update_mutually_sql, {'uid': uid, 'person_id': person_id,
                                          'fgroup_id': GroupType.FRIEND})


def addFriendshipNotification(whom, where):
    # Если Люба подружилась с Фёклой, когда у Любы висела нотификация,
    # что Фёкла хочет с ней дружить, скрыть эту нотификацию:
    UserEvents.mark_seen(where, ['Friender'], friender_uid=whom)

    # На всякий случай:
    if whom == where:
        return False

    access = Accesses.Access(where, whom)
    # Если Вася подружился с Петей, а Петя до того забанил Васю, не уведомлять Петю:
    if access.is_banned():
        return False
    # Если Циля подружилась с Соней, а Соня на тот момент уже дружила с Цилей,
    # не уведомлять Соню:
    if access.is_friend():
        return False
    # Если Егор подружился с Кузьмой повторно, т.е. Кузьма уже раньше получал
    # нотификацию о том, что Егор хочет с ним дружить, не уведомлять Кузьму.
    # Явным образом проверку не делаю, т.к. EventList просто вернёт False в случае
    # DuplicateKey error
    return UserEvents.EventList(whom).push(
        UserEvents.FrienderNotification(where))


def markFriendshipNotification(whom, where):
    UserEvents.mark_seen(whom, ['Friender'], friender_uid=where)
    UserEvents.mark_seen(where, ['Friender'], friender_uid=whom)


def addClubNotification(whom, where, group_id, initiator_id):
    NOTIFYABLE_ROLES = {
        GroupType.INVITED: 'invitation',
        GroupType.MODERATOR_INVITED: 'moder_invitation',
    }
    UserEvents.EventList(whom).push(UserEvents.ClubNotification(
        NOTIFYABLE_ROLES[group_id], initiator_id, where, 0
    ))


def markClubNotification(uid, feed_id):
    UserEvents.mark_seen(uid, ['ClubNotification'], feed_id=feed_id, item_no=0)


def friendAndFollow(ai, whom, where):
    access = Accesses.Access(ai.uid, whom, target=where)
    access.assert_can_add_friend()
    Relations.addRelation(uid=whom, person_id=where,
                          group=GroupType.FRIEND)  # дружба – чёртово инвертированное отношение
    update_mutually(whom, where)
    access.assert_can_add_followship()
    at.aux_.models.Follower(uid_id=where, person_id_id=whom).save()
    addFriendshipNotification(whom, where)
    # никаких почтовых нотификаций в случае дружбы мы не рассылаем


def unfriendAndUnfollow(ai, whom, where):
    access = Accesses.Access(ai.uid, whom, target=where)
    access.assert_can_remove_friend()
    Relations.removeRelation(uid=whom, person_id=where,
                             group=GroupType.FRIEND)  # дружба – чёртово инвертированное отношение
    update_mutually(whom, where)
    access.assert_can_remove_followship()
    at.aux_.models.Follower.objects.filter(uid_id=where, person_id_id=whom).delete()
    markFriendshipNotification(whom, where)
    # никаких почтовых нотификаций в случае дружбы мы не рассылаем


def unbanPerson(ai, uid, feed_id):
    access = Accesses.Access(ai.uid, uid, target=feed_id)
    access.assert_can_ban()
    Relations.removeRelation(uid, feed_id, GroupType.BANNED)


def banPerson(ai, uid, feed_id):
    access = Accesses.Access(ai.uid, uid, target=feed_id)
    access.assert_can_ban()
    if access.target.is_community():
        unjoinFromCommunity(ai, uid, feed_id)
    else:
        unfriendAndUnfollow(ai, uid, feed_id)
    Relations.addRelation(uid, feed_id, GroupType.BANNED,
                          clear_all_groups=True)


def inviteToGroup(ai, uid, feed_id, group_id):
    access = Accesses.Access(ai.uid, uid, target=feed_id)
    access.assert_can_invite(group_id)
    real_group_id = GroupType.invite[group_id]
    Relations.addRelation(uid, feed_id, real_group_id)
    HandlerRegistry.put_event('MailNewRelationHandler',
                              uid, ai.uid, feed_id, real_group_id)
    addClubNotification(whom=uid, where=feed_id, group_id=real_group_id,
                        initiator_id=ai.uid)


def joinCommunity(ai, uid, feed_id, group=GroupType.MEMBER):
    access = Accesses.Access(ai.uid, uid, target=feed_id)
    access.assert_can_add_to_group(group)
    access.assert_can_add_followship()
    Relations.addRelation(uid, feed_id, group)
    if not at.aux_.models.Follower.objects.filter(uid_id=uid, person_id_id=feed_id).count():
        at.aux_.models.Follower(uid_id=uid, person_id_id=feed_id).save()
    markClubNotification(uid, feed_id)


def unjoinFromCommunity(ai, uid, feed_id):
    group = GroupType.MEMBER
    access = Accesses.Access(ai.uid, uid, target=feed_id)
    access.assert_can_remove_from_group(group)
    access.assert_can_remove_followship()
    Relations.removeRelation(uid, feed_id, group)
    at.aux_.models.Follower.objects.filter(uid=uid, person_id=feed_id).delete()
    markClubNotification(uid, feed_id)


class Friends:
    @utils.et2xml
    @log_exception
    def UnjoinFromCommunity(self, ai, request):
        uid = request['uid']
        community_feed = request['community_feed']
        with dbswitch.root_rw_session():
            unjoinFromCommunity(ai, uid, community_feed)
        return Status('OK')

    @utils.et2xml
    @log_exception
    def JoinToCommunity(self, ai, request):
        uid = request['uid']
        community_feed = request['community_feed']
        with dbswitch.root_rw_session():
            joinCommunity(ai, uid, community_feed)
        return Status('OK')

    @utils.et2xml
    @log_exception
    def FindRelationsStr(self, user_ids, feed_ids):
        fnc = lambda s: [_f for _f in [x.strip() for x in s.split(',')] if _f]
        roles = Relations.get_relations_multi(fnc(user_ids), fnc(feed_ids),
                                              flat=True)
        # обратные отношения нужны, как минимум, для in-friends-of-current-user
        roles.extend(Relations.get_relations_multi(fnc(feed_ids), fnc(user_ids),
                                                   flat=True))
        roles.expand()
        return ET.ElementTree(roles.to_xml('relations', 'relation'))

    @utils.et2xml
    @log_exception
    def FindRelations(self, person_id, feed_id):
        root = ET.Element('person-relations')
        if utils.is_community_id(feed_id):
            uids = [person_id]
            feed_ids = [feed_id]
        else:
            # если хост – это пользователь, то нам ещё понадобятся инвертированные отношения для friend-of
            uids = [person_id, feed_id]
            feed_ids = [person_id, feed_id]
        roles = Relations.get_relations_multi(uids, feed_ids)
        direct_roles = roles.get((person_id, feed_id), Relations.RelationsList())
        direct_roles.expand()
        for r in direct_roles:
            ET.SubElement(root, 'is-' + GroupType.get_str(r.role).lower(),
                          {'mutual': str(r.mutual)}
                          )
        if not utils.is_community_id(feed_id):
            inverse_roles = roles.get((feed_id, person_id), Relations.RelationsList())
            inverse_roles.expand()
            if GroupType.FRIEND in [r.role for r in inverse_roles]:
                ET.SubElement(root, 'is-friend-of')
        return ET.ElementTree(root)

    @utils.et2xml
    @log_exception
    @stopwatch
    def UpdateIntraFriends(self, ai):
        intrafriend.update_person(ai)
        return Status('Success')

    @utils.et2xml
    @log_exception
    @stopwatch
    def MassFriendingAction(self, ai, request):
        action = request['action']
        if action == 'breakup':
            # кажется, "кого я считаю другом" – это так:
            friends = [
                rel.uid for rel in Relations.get_relations_multi(None, [ai.uid], flat=True)
                if rel.role == GroupType.FRIEND
            ]
            qu_limit = request.get('qu_limit', 90)

            for offset in range(0, len(friends), 100):
                chunk = friends[offset:offset + 100]
                feed_info = list(models.Person.get_persons(chunk).values())
                uids = [fi.person_id for fi in feed_info if fi.qu < qu_limit]

                _log.debug('breaking up with %r' % (uids,))

                for uid in uids:
                    HandlerRegistry.put_event('UnfriendHandler', uid, ai.uid)

        if action == 'clubs-join':
            with get_connection() as conn:
                club_ids = conn.execute(
                    """
                        select person_id from persons
                        where person_id >= %s and status = 'normal' and
                            community_type IN (
                            'OPENED_COMMUNITY', 'PREMODERATED_COMMUNITY')
                    """,
                    (settings.COMMUNITY_START_ID,)
                )

            _log.debug('joining clubs %r' % (club_ids,))

            for club_id, in club_ids:
                HandlerRegistry.put_event('AutofriendingHandler',
                                          club_id, ai.uid)

        if action == 'clubs-leave':
            roles = set(
                getattr(GroupType, role, None)
                for field, role in list(request.items())
                if field.startswith('role-')
            )

            for rel in Relations.get_relations_multi([ai.uid], None, flat=True):
                if rel.role in roles:
                    HandlerRegistry.put_event('UnfriendHandler',
                                              rel.feed_id, ai.uid)

        return Status('Success')

    @utils.et2xml
    @utils.dict2etree(root_name='groups')
    @log_exception
    @stopwatch
    def GetStaffTree(self):
        # сломана из-за кривых походов в центр
        pass
        # TODO: заменить на походы в стафф
        # groups = dict(
        #     (g['id'], g) for g in center.api.v1.groups(fields='id|parent|name'))
        # for group in groups.itervalues():
        #     if group['parent'] in groups:
        #         parent = groups[group['parent']]
        #         try:
        #             siblings = parent['groups']
        #         except KeyError:
        #             siblings = parent['groups'] = []
        #         siblings.append(group)
        #
        # return [group for group in groups.itervalues() if
        #         group['parent'] not in groups]
