# coding=utf-8



import datetime
import logging

from at.common import exceptions
from at.common import dbswitch
from at.common import utils
from at.common import groups
from at.common import postlink
from at.common.assertions import *
from at.common.groups import GroupType
from at.common.utils import get_connection
from at.common.utils import (
    Status, log_exception, check_auth, stopwatch, is_community_id
)
from at.aux_ import Accesses
from at.aux_ import Entries
from at.aux_ import entries
from at.aux_ import Friends
from at.aux_ import Mailing
from at.aux_ import Polls
from at.aux_ import ProfileStorage
from at.pump import HandlerRegistry

from django.core.exceptions import PermissionDenied

_log = logging.getLogger('aux/items')


def parse_item_id(id):
    id_list = [int(e) for e in id.split('.')]
    if len(id_list) < 3:
        return id_list[0], id_list[1], 0
    else:
        return tuple(id_list[:3])


def approvePost(auth_info, feed_id, post_no):
    # XXX всё это ужасно и должно реюзать код из контроллера
    post = entries.load_entry(feed_id, post_no)
    post.moderated_by = auth_info.login
    post.on_moderation = False
    post.published = True
    post.store_time = datetime.datetime.now()
    # в случае трекбека в премодерируемый клуб нет возможности проставить
    # тип доступа — ставим публичный
    post.access_type = post.original_access_type or 'public'
    post.save()

    author_uid = post.author_id
    cc_list = post.cc_list
    if cc_list is None:
        cc_list = ProfileStorage.get_entry(feed_id, 'CarbonCopy',
                                           'carbon-copy-address')
    for cc in set(cc_list):
        HandlerRegistry.put_event('CCFinalizeHandler',
                                  feed_id, post_no, author_uid, cc)
    # Распознаём упоминания в посте после одобрения модераторами.
    HandlerRegistry.put_event('EditHandler', feed_id, post_no, 0)
    HandlerRegistry.put_event('PublishHandler',
        author_uid, feed_id, post_no, 0, post.store_time.isoformat())
    HandlerRegistry.put_event('ModerateHandler', post.feed_id, post.item_no)
    return True


class Items:
    @utils.et2xml
    @stopwatch
    @log_exception
    def GetLikeDetails(self, ai, id, limit=None, offset=None):
        if not limit:
            limit = 5000
        limit = 5000 if limit >= 1000 else limit

        offset = offset or 0
        _log.info("Getting like details for id: %s, ai: %s" % (id, ai.uid))

        feed_id, item_no, comment_id = parse_item_id(id)

        # >= -1 в запросах означает, что первыми мы хотим иметь знакомые морды:
        # -1 - я сам
        #  0 - друзья

        # A little optimization since we're not caching anything here
        # If the user and the feed are on the same shard, do it in one query
        sql = '''
        select l.uid, fgm.fgroup_id, l.value
        from Likes l left join FriendGroupMember fgm
        on (l.uid = fgm.uid and fgm.person_id = %s and fgroup_id in (-1, 0))
        where l.feed_id = %s and l.item_no = %s and l.comment_id = %s
        order by fgroup_id is NULL, fgroup_id, l.uid
        limit %s offset %s
        '''
        with get_connection() as connection:
            cursor = connection.execute(sql, (
            ai.uid, feed_id, item_no, comment_id, limit, offset))
            result = [(str(uid), group, value) for uid, group, value in cursor]
        # ...convert to xml
        # XXX: атрибуты count / total используются только в ajax/jsonp_do_like с бесконечным размером страницы,
        # и только благодаря ему оно работает хоть сколько-то разумно. Вообще адово, конечно.
        root = ET.Element('like', count=str(len(result)), id=id,
                          total=str(sum((r[2] for r in result), 0)))
        list(map(root.append, [root.makeelement('user',
                                           {'uid': uid,
                                            'group': groups.GroupType.get_str(
                                                group).lower(), 'value': str(
                                               value)} if group is not None else {
                                               'uid': uid, 'value': str(value)})
                          for uid, group, value in result]))
        details = ET.tostring(root)

        return details

    @utils.deprecated
    def LikePost(self, ai, id):
        return self.LikePost2(self, ai, id, 1)

    @stopwatch
    @log_exception
    def LikePost2(self, ai, request):
        id = request['id']
        value = request['value']

        feed_id, item_no, comment_id = parse_item_id(id)
        url = postlink.make_url(feed_id, item_no, comment_id)
        value = (value > 0) - (value < 0)  # sign(value)
        if value == 0:
            return self.UnlikePost(ai, id)
        # нам потребуется сниппет корневого поста, а не коммента:
        post = entries.models.load_entry(feed_id, item_no)
        query = '''
        replace into Likes
        (feed_id, item_no, comment_id, uid, title, url, timestamp, value)
        values (%s, %s, %s, %s, %s, %s, now(), %s)
        '''
        args = (feed_id, item_no, comment_id, ai.uid, post.snippet, url, value)
        try:
            with get_connection() as connection:
                connection.execute(query, args)
        except Exception:
            _log.exception('Failed to store like data for uid %s, id %s',
                           ai.uid, id)
            return Status('Failed')

        # а вот тут нам понадобится автор коммента
        if comment_id:
            comment = entries.models.load_entry(feed_id, item_no, comment_id)
        else:
            comment = post
        if ai.uid != comment.author_id:
            HandlerRegistry.put_event('LikeNotificationHandler',
                comment.author_id, feed_id, item_no, comment_id, ai.uid)

        HandlerRegistry.put_event('ScorePostHandler', feed_id, item_no)

        return Status('Success')

    @stopwatch
    @log_exception
    def UnlikePost(self, ai, request):
        id = request['id']

        feed_id, item_no, comment_id = parse_item_id(id)

        args = (ai.uid, feed_id, item_no, comment_id)
        query = 'delete from Likes where uid = %s and feed_id = %s and item_no = %s and comment_id = %s'

        try:
            with get_connection() as connection:
                connection.execute(query, args)
        except Exception:
            _log.error(
                'Failed to remove like data for uid %s, id: %s' % (ai.uid, id),
                exc_info=1)
            return Status('Failed')

        # If the item was previously liked by the same user, mark the corresponding event as read
        try:
            from at.aux_.UserEvents import EventList  # FIXME циклическая ссылка
            author_id = entries.load_entry(feed_id, item_no,
                                           comment_id).author_id
            uel = EventList(author_id, types=['Like'], exclude=['seen', 'sent'])
            for event in uel.filter(uid=ai.uid, feed_id=feed_id,
                                    item_no=item_no):
                event.state.add('seen')
            uel.save()
        except:
            _log.error('Failed to mark like notification as seen', exc_info=1)

        HandlerRegistry.put_event('ScorePostHandler', feed_id, item_no)
        return Status('Success')

    @utils.et2xml
    @log_exception
    @check_auth(1)
    @stopwatch
    def RepliesDoAddForID(self, ai, request):
        def mk_result_xml(item_no, comment_id, trackback_item_no=None):
            root = ET.Element("aux")
            ET.SubElement(root, 'new-reply-id').text = str(comment_id)
            ET.SubElement(root, 'item-no').text = str(item_no)
            if trackback_item_no is not None:
                ET.SubElement(root, 'trackback-item-no').text = str(
                    trackback_item_no)
            ET.SubElement(root, 'retpath').text = \
                "/replies.xml?item_no=%s%s" % (
                item_no, (comment_id and ('&#reply-%s' % comment_id) or ''))
            _log.debug("retpath set /replies.xml?item_no=%s%s" % (
            item_no, (comment_id and ('&#reply-%s' % comment_id) or '')))
            root.append(ET.XML(Status('Success')))
            return ET.ElementTree(root)

        feed_id = request['feed_id']
        reply_type = request.get('type', 'text')
        item_no = int(request['item_no'])

        if request.get('welcome_to_the_club') == 'yes' and request.get('type') != 'join':
            if not is_community_id(feed_id):
                _log.warn('Trying "Post and Join" not to a club')
            else:
                try:
                    with dbswitch.root_rw_session():
                        Friends.joinCommunity(ai, ai.uid, feed_id)
                except:
                    _log.exception('Failed "Post and Join" for %s in %s',
                                   ai, feed_id)

        if reply_type == 'approve':
            approvePost(ai, feed_id, item_no)
            return Status('Success')
        elif reply_type == 'subscribe':
            directions = {
                    'subscribe': Mailing.Subscriptions.SubscribeToPost,
                    'unsubscribe': Mailing.Subscriptions.UnSubscribeFromPost
                    }
            return directions[request.get('direction', 'subscribe')](ai, feed_id, item_no)
        elif reply_type == 'ban_and_delete':
            uid = int(request.get('author_uid', 0))
            comment_id = int(request.get('parent_id', 0))
            if not comment_id:
                self.DeletePostForID(ai, feed_id, item_no)
            else:
                self.DeleteCommentByFeedID( \
                    ai, feed_id, item_no, comment_id)
            self.BanPersonRaw(ai, feed_id, uid)
            if 'delete_all' in request and request['delete_all'].strip():
                # не работает, надо удалить
                self.RemoveUserPosts(ai, feed_id, uid)
            return Status('Success')

        parent_id = request.get('parent_id') and int(request.get('parent_id'))
        comment_id = request.get('comment_id') and int(request.get('comment_id'))
        post = Entries.Post(reply_type == 'post' and 'text' or reply_type)(
            auth_info=ai,
            feed_id=feed_id,
            item_no=item_no,
            parent_id=parent_id,
            comment_id=comment_id,
            tags=request.get('tags', '') + ',' + request.get('tag', ''),
            link=request.get('form_id'),
            do_trackback=reply_type == 'link' or request['trackback'] or 0,
            tb_feed_id=request['tb_feed_id'] or 0,
        )
        _log.debug(post)
        post.createItemContent(request)
        new_comment_id = post.do_comment()

        return mk_result_xml(post.item_no,
                             new_comment_id)  # post.trackback_item_no)

    @utils.et2xml
    @log_exception
    @stopwatch
    @check_auth()
    def PostsDoAdd(self, ai, request):
        request = request or {}
        req = utils.req2dict(request)
        feed_id = int(req.get('feed_id') or ai.uid)
        item_no = req.get("item_no") and int(req.get("item_no"))

        # hack; see AT-1978
        if all(key not in req for key in Entries.CC_FIELDS):
            req[Entries.CC_FIELDS[0]] = ''

        try:
            post = Entries.Post(req.get('type', 'text'))(
                auth_info=ai,
                feed_id=feed_id,
                item_no=item_no,
                base_url=req.get("base_url", ''),
                access_type=req.get("access_type", None),
                freezed=not req.get("replies", False),
                tags=req.get('tags', '') + ',' + req.get('tag', ''),
                link=req.get('form_id'),
            )
            post.createItemContent(req)
            item_no = post.do_post()
        except:
            _log.exception('PostsDoAdd failed with request', req)
            raise
        return Entries.return_post_xml(item_no, feed_id)

    @log_exception
    @stopwatch
    def Vote(self, ai, request):
        feed_id = request['feed_id']
        item_no = request['item_no']
        options = request['options']

        votes = [int(i) for i in options.split(',')]
        try:
            Polls.PollManager.vote_for(ai, feed_id, item_no, votes)
            HandlerRegistry.put_event('ScorePostHandler', feed_id, item_no)

            return Status('Success')
        except (exceptions.AccessDenied, exceptions.AccessDenied) as ex:
            return Status(ex.__class__.__name__, ex.dsc)

    @utils.et2xml
    @log_exception
    @stopwatch
    @check_auth()
    def DeleteCommentByFeedID(self, ai, request):
        feed_id = request['feed_id']
        item_no = request['item_no']
        comment_id = request['comment_id']

        model = entries.load_entry(feed_id, item_no, comment_id)
        model.delete()

        HandlerRegistry.put_event('DeleteHandler',
                                  ai.uid,
                                   feed_id,
                                   item_no,
                                   comment_id,
                                   False)
        return utils.build_xml(("status",
                                {"code": "OK"},
                                []))

    @utils.et2xml
    @log_exception
    @stopwatch
    @check_auth()
    def DeletePostForID(self, ai, request):
        feed_id = request['feed_id']
        item_no = request['item_no']
        model = entries.load_entry(feed_id, item_no)
        model.delete()
        HandlerRegistry.put_event('DeleteHandler',
                                  ai.uid,
                                   feed_id,
                                   item_no,
                                   0,
                                   model.on_moderation)

        return utils.build_xml(("status",
                                {"code": "OK"},
                                []))

    @utils.et2xml
    @log_exception
    @stopwatch
    def GetPostSafe(self, ai, feed_id, item_no, escape_html=False):
        model = entries.load_entry(feed_id, item_no)
        if ai:
            access_manager = Accesses.Access(ai.uid, feed_id, entry=model)
            access_manager.assert_can_read_post()
        elif model.access_group != GroupType.USER:
            raise PermissionDenied()

        serialized = model.serialize(escape_html=escape_html)
        return serialized
