# -*- coding: utf-8 -*-



import logging

import lxml.etree as ET

from django.conf import settings

import at.aux_.models
from at.common import models as common_models
from at.common import staff
from at.common import const
from at.common import Maillists
from at.common.utils import Status, log_exception, et2xml, get_connection
from at.aux_ import Accesses
from at.aux_ import entries

_log = logging.getLogger(__name__)


def is_email_normal(email):
    """Filter function, filters dismissed employers
    and not confirmed personal emails"""
    _log.debug("Validating email %s", email)
    if Maillists.is_email_ml(email):
        return True
    else:
        api = staff.Api()
        data = api.persons.filter(
            emails__address=email
        ).fields(['login', 'emails', 'official.is_dismissed'])
        if data['total'] == 0:
            # Если нет такой почты в стаффе
            return False
        else:
            # Статус сотрудника
            person = data['result'][0]
            person_mails = [
                el['address'] for el in person['emails']
                if el['source_type'] == 'passport' or el['source_type'] == 'staff'
            ]
            return not person['official']['is_dismissed'] and email in person_mails


class Subscriptions(object):
    """Менеджер подписок на посты и дневники"""

    queries = {
        'get': "SELECT 1 FROM Subscriptions WHERE uid = %s AND feed_id = %s AND item_no = %s",
        'set': """
            INSERT INTO Subscriptions(uid, feed_id, item_no, with_comments, mode) VALUES(%s, %s, %s, %s, %s)
            ON DUPLICATE KEY UPDATE with_comments=VALUES(with_comments), mode=IF(mode='manual', mode, VALUES(mode))""",

        'delete': "DELETE FROM Subscriptions WHERE uid = %s AND feed_id = %s AND item_no = %s AND mode = %s",
        'trackbacks': "SELECT trackback_person_id, trackback_post_no FROM Trackbacks WHERE person_id = %s AND post_no = %s",
        'subscribers': "SELECT uid FROM Subscriptions WHERE feed_id = %s AND item_no = %s",
    }

    @classmethod
    def is_subscribed(cls, ai, feed_id, item_id):
        with get_connection() as connection:
            cursor = connection.execute(cls.queries['get'],
                                        (ai.uid, feed_id, item_id))
        return cursor.rowcount != 0

    @classmethod
    def SubscribeToPost(cls, ai, feed_id, item_id, with_comments=True,
                        mode='manual'):
        return cls.SubscribeToPostUID(ai.uid, feed_id, item_id, with_comments,
                                      mode)

    @classmethod
    def SubscribeToPostUID(cls, uid, feed_id, item_id, with_comments=True,
                           mode='manual'):
        with get_connection() as connection:
            _log.debug("%s %s" % (
            cls.queries['set'], (uid, feed_id, item_id, with_comments, mode)))
            connection.execute(cls.queries['set'],
                               (uid, feed_id, item_id, with_comments, mode))

        if item_id != settings.UNDEFINED_POST_ID:
            cls.SubscribeToChildren(uid, feed_id, item_id, with_comments)

        return Status('Success')

    @classmethod
    def UnSubscribeFromPost(cls, ai, feed_id, item_id, mode=None):
        with get_connection() as connection:
            if mode in [None, 'manual']:
                connection.execute(cls.queries['delete'],
                                   (ai.uid, feed_id, item_id, 'manual'))
            if mode in [None, 'auto']:
                auto_removed = connection.execute(cls.queries['delete'], (
                ai.uid, feed_id, item_id, 'auto')).rowcount

        _log.debug("unsubscribing %s from %s.%s mode %s, auto_removed=%s" % (
        ai.uid, feed_id, item_id, mode, auto_removed))

        # Recurse if:
        #  - we are unsubscribing from a real post, not blog,
        #  and
        #  - we are explicitly unsubscribing from this post (mode != 'auto')
        #  or
        #  - we are processing some other post's children (mode == 'auto') and
        #    user was automatically subscribed to this post (auto_removed is True).

        if item_id != settings.UNDEFINED_POST_ID and (
                mode != 'auto' or auto_removed):
            cls.UnsubscribeFromChildren(ai, feed_id, item_id)

        return Status('Success')

    @classmethod
    def SubscribeToBlog(cls, ai, feed_id, with_comments):
        return cls.SubscribeToPost(ai, feed_id, settings.UNDEFINED_POST_ID,
                                   with_comments)

    @classmethod
    def UnSubscribeFromBlog(cls, ai, feed_id):
        return cls.UnSubscribeFromPost(ai, feed_id, settings.UNDEFINED_POST_ID)

    @classmethod
    def SubscribeToChildren(cls, uid, feed_id, item_no, with_comments):
        _log.debug(
            "subscribing %s to children of %s.%s" % (uid, feed_id, item_no))
        with get_connection() as connection:
            for person_id, post_no in connection.execute(
                    cls.queries['trackbacks'], (feed_id, item_no)):

                # Do not subscribe to uid's own posts, since it's impossible via UI.
                if person_id == uid:
                    continue

                _log.debug("subscribing %s to child of %s.%s - %s.%s" % (
                uid, feed_id, item_no, person_id, post_no))
                cls.SubscribeToPostUID(uid, person_id, post_no, with_comments,
                                       'auto')

    @classmethod
    def UnsubscribeFromChildren(cls, ai, feed_id, item_no):
        uid = ai.uid
        _log.debug(
            "unsubscribing %s from children of %s.%s" % (uid, feed_id, item_no))
        with get_connection() as connection:
            for person_id, post_no in connection.execute(
                    cls.queries['trackbacks'], (feed_id, item_no)):

                # Ignore user's own posts, since we don't subscribe to them automatically.
                if person_id == uid:
                    continue

                _log.debug("unsubscribing %s from child of %s.%s - %s.%s" % (
                uid, feed_id, item_no, person_id, post_no))
                cls.UnSubscribeFromPost(ai, person_id, post_no, 'auto')

    @classmethod
    def PropogateSubscriptions(cls, parent_feed_id, parent_item_no, new_feed_id,
                               new_item_no):
        with get_connection() as connection:
            for person_id, in connection.execute(cls.queries['subscribers'], (
            parent_feed_id, parent_item_no)):
                cls.SubscribeToPostUID(person_id, new_feed_id, new_item_no,
                                       True, 'auto')


class Mailing:
    """Почтовый менеджер

    Генерирует списки почтовых адресов и почтовых настроек для людей, которые хотели бы
    увидеть письмо об указанном действии в своём почтовом ящике.
    """

    queries = {
        'subscribers':
            """
            select s.uid
            from Posts p
            join Subscriptions s on
              (p.person_id = s.feed_id
              and (:comment_id and p.post_no = s.item_no or s.item_no = :undefined_post_id)
              and (s.with_comments = 1 or not :comment_id))
            where p.person_id = :feed_id and p.post_no = :item_no
          """,
    }

    @classmethod
    def GetMailingTargets(cls, feed_id, post_id, comment_id=0,
                          unfiltered_uids=False):
        entry = entries.load_entry(feed_id, post_id, comment_id)
        subscriber_uids = set([uid for uid, in get_connection().execute(
            cls.queries['subscribers'],
            {
                'comment_id': comment_id,
                'undefined_post_id': settings.UNDEFINED_POST_ID,
                'feed_id': feed_id,
                'item_no': post_id
            }
        )])
        if not comment_id:
            # подписчики Мгновенного Дайджеста
            instant_subscribed_followers = at.aux_.models.Follower.objects.filter(
                person_id=feed_id,
                uid_id__mail_settings__digest_mode=const.DIGEST_MODES.ALL,
            ).values_list('uid_id', flat=True)
            subscriber_uids |= set(instant_subscribed_followers)

        if comment_id:
            post = entries.models.load_entry(feed_id, post_id)
            subscriber_uids.add(post.author_id)
        else:
            post = entry

        if entry.parent_comment_id:
            parent = entries.load_entry(feed_id, post_id,
                                        entry.parent_comment_id)
            subscriber_uids.add(parent.author_id)

        # включает в себя проверку того, что пользователь живой
        subscriber_uids = set([uid for uid in set(subscriber_uids) if Accesses.Access(uid, feed_id, post).can_read_post()]
        )

        if not unfiltered_uids:
            subscriber_uids = filter_persons_for_notify(entry, subscriber_uids)

        # слать ли нотификацию автору записи?
        # пост: в своем дневнике не шлем, в клубе шлем, если есть подписка
        # на клуб или фоловинг с DigestMode.all — как всем
        # комент: если включена настройка notify_self — шлем,
        # в остальных случаях не шлем
        if comment_id:
            ums = get_user_mail_settings(entry.author_id)
            if (
                ums.comment_mode == const.COMMENT_MODES.ALL and
                ums.notify_self
            ):
                subscriber_uids.add(entry.author_id)
            else:
                subscriber_uids.discard(entry.author_id)

        return subscriber_uids


    @classmethod
    @log_exception
    @et2xml
    def GetMailingSettingsXML(cls, ai):
        mail_settings = get_user_mail_settings(ai.uid)
        LEGACY = const.LEGACY

        root = ET.Element('UserMailSettings')
        ET.SubElement(root, 'feed_id').text = str(ai.uid)
        for field_old, field_new in list(LEGACY.NOTIFICATIONS_FIELDS_OLD_NEW.items()):
            value_new = getattr(mail_settings, field_new)
            if field_new == const.NOTIFICATION_FIELDS.COMMENT_MODE:
                value_new_str = str(LEGACY.COMMENT_NOTIFICATION_MODE_NEW_OLD[value_new])
            elif field_new == const.NOTIFICATION_FIELDS.DIGEST_MODE:
                value_new_str = str(LEGACY.DIGEST_MODE_NEW_OLD[value_new])
            else:
                value_new_str = str(int(value_new))
            ET.SubElement(root, field_old).text = value_new_str

        return ET.ElementTree(root)

    @classmethod
    @log_exception
    @et2xml
    def SetMailingSettings(cls, ai, request):
        _log.debug(request)
        LEGACY = const.LEGACY

        data = {}
        for field_old, value_old in list(request.items()):
            field_new = LEGACY.NOTIFICATIONS_FIELDS_OLD_NEW[field_old]
            if field_new == const.NOTIFICATION_FIELDS.COMMENT_MODE:
                value_new = LEGACY.COMMENT_MODE_OLD_NEW[value_old]
            elif field_new == const.NOTIFICATION_FIELDS.DIGEST_MODE:
                value_new = LEGACY.DIGEST_MODE_OLD_NEW[value_old]
            else:
                value_new = bool(value_old)
            data[field_new] = value_new

        update_mail_settings(
            person_id=ai.uid,
            data=data,
        )

        return Status('Success')


    @classmethod
    @log_exception
    @et2xml
    def GetPostSubscriptionXML(cls, ai, feed_id, item_no):
        return Subscriptions.is_subscribed(ai, feed_id, item_no) and Status(
            'Success', 'yes') or Status('Success', 'no')

    @classmethod
    @log_exception
    @et2xml
    def GetBlogSubscriptionXML(cls, ai, feed_id):
        root = ET.Element('Subscription')
        ET.SubElement(root, 'Notifications',
                      enabled=Subscriptions.is_subscribed(ai, feed_id,
                                                          settings.UNDEFINED_POST_ID) and 'true' or 'false')
        return ET.ElementTree(root)

    @classmethod
    @log_exception
    @et2xml
    def SetBlogSubscription(cls, ai, request):
        feed_id = request['feed_id']
        notifications = request['notifications']
        if notifications:
            Subscriptions.SubscribeToBlog(ai, feed_id, with_comments=True)
        else:
            Subscriptions.UnSubscribeFromBlog(ai, feed_id)
        return Status('Success')

    @classmethod
    @log_exception
    def SetBlogSubscriptionPublic(cls, ai, request):
        feed_id = request['feed_id']
        new_state = request['new_state']

        _log.debug("Setting blog public subscription: %r %r %r" % (
        ai, feed_id, new_state))
        if new_state:
            Subscriptions.SubscribeToBlog(ai, feed_id, with_comments=False)
        else:
            Subscriptions.UnSubscribeFromBlog(ai, feed_id)

        return Status('Success')

    @classmethod
    @log_exception
    def SetPostSubscriptionPublic(cls, ai, request):
        feed_id = request['feed_id']
        item_no = request['item_no']
        new_state = request['new_state']

        _log.debug("Setting post public subscription: %r %r %r %r" % (
            ai, feed_id, item_no, new_state))

        if new_state:
            Subscriptions.SubscribeToPost(ai, feed_id, item_no,
                                          with_comments=True)
        else:
            Subscriptions.UnSubscribeFromPost(ai, feed_id, item_no)

        return Status('Success')


def get_digest_users():
    return common_models.MailSettings.objects.filter(
        person__has_access=True,
        digest_mode__in=(
            const.DIGEST_MODES.POPULAR,
            const.DIGEST_MODES.FULL,
        )
    ).values_list('person_id', 'digest_mode')


def get_user_mail_settings(person_or_uid):
    # на время миграции, потом можно просто .get() оставить
    ums = common_models.MailSettings.objects.filter(person=person_or_uid).first()
    return ums or common_models.MailSettings(
        **const.NOTIFICATION_FIELDS.DEFAULTS
    )


def get_user_mail_settings_bulk(person_ids):
    return common_models.MailSettings.objects.filter(person__in=person_ids)


def update_mail_settings(person_id, data):
    return common_models.MailSettings.objects.filter(
        person_id=person_id,
    ).update(**data)


def filter_persons_for_notify(entry, person_ids):
    if not entry.comment_id:
        return person_ids

    filtered = set()
    mail_settings = {
        mail_settings.person_id: mail_settings
        for mail_settings in get_user_mail_settings_bulk(person_ids)
    }

    for person_id in person_ids:
        person_mail_settings = mail_settings[person_id]
        mode = person_mail_settings.comment_mode

        if mode == const.COMMENT_MODES.FRIENDS:
            access = Accesses.Access(
                user=entry.author_id,
                feed=person_mail_settings.person_id,
            )
            if access.is_friend():
                filtered.add(person_id)
        elif mode == const.COMMENT_MODES.ALL:
            filtered.add(person_id)
    return filtered


def tmp_migrate_mail_settings(
    mail_settings_cls=common_models.MailSettings,
    person_cls=at.aux_.models.Person,
):
    from at.aux_ import MongoStorage
    mongo_storage = MongoStorage.Storage(
        collection_name='UserMailSettings_storage'
    )
    mongo_storage.connect()

    persons = person_cls.objects.all()
    old_settings = {
        item['feed_id']: item
        for item in mongo_storage.collection.find()
    }

    LEGACY = const.LEGACY
    new_settings_models = []
    for person in persons:
        new_data = const.NOTIFICATION_FIELDS.DEFAULTS.copy()
        # миграция выключает дайджест для людей с потерянными настройками,
        # чтобы не заспамить всех
        new_data[const.NOTIFICATION_FIELDS.DIGEST_MODE] = const.DIGEST_MODES.DISABLED
        if person.person_id in old_settings:
            old_data = old_settings[person.person_id].get('dict_data', {})
            for field_old, value_old in list(old_data.items()):
                field_new = LEGACY.NOTIFICATIONS_FIELDS_OLD_NEW.get(field_old)
                if field_new is None:
                    # email etc
                    continue
                if field_new == const.NOTIFICATION_FIELDS.COMMENT_MODE:
                    value_new = LEGACY.COMMENT_MODE_OLD_NEW[value_old]
                elif field_new == const.NOTIFICATION_FIELDS.DIGEST_MODE:
                    value_new = LEGACY.DIGEST_MODE_OLD_NEW[value_old]
                else:
                    value_new = bool(value_old)
                new_data[field_new] = value_new
        model = mail_settings_cls(
            person=person,
            **new_data
        )
        new_settings_models.append(model)
    return mail_settings_cls.objects.bulk_create(new_settings_models)
