# coding: utf-8

import logging

from django.conf import settings
from pymongo import MongoClient

_log = logging.getLogger(__name__)


class Storage(object):
    connection = None

    @classmethod
    def connect_to_mongo(cls):
        if cls.connection is None:
            nodes = ','.join(settings.MONGO['hosts'])
            repl_set_name = settings.MONGO['repl_set']
            _log.debug("Connecting to %s as %s" % (nodes, repl_set_name))
            max_pool_size = settings.MONGO['max_pool_size']
            network_timeout = settings.MONGO['network_timeout']
            connect_timeout = settings.MONGO['connect_timeout']
            connection_string_tpl = 'mongodb://{username}:{password}@{hosts}/{database}?' + '&'.join([
                '%s={%s}' % (option, option) for option in (
                    'maxPoolSize',
                    'socketTimeoutMS',
                    'connectTimeoutMS',
                    'replicaSet',
                )
            ])
            connection_string = connection_string_tpl.format(
                username=settings.MONGO['user'],
                password=settings.MONGO['password'],
                hosts=nodes,
                database=settings.MONGO['database'],
                maxPoolSize=max_pool_size,
                socketTimeoutMS=network_timeout,
                connectTimeoutMS=connect_timeout,
                replicaSet=repl_set_name
            )
            try:
                cls.connection = MongoClient(connection_string)
                _log.debug('New mongo connection.')
            except Exception:
                _log.exception('Failed to connect to mongo!')
                raise
        return cls.connection

    def __init__(self, collection_name):
        self.collection_name = collection_name
        self._connected = False
        self.db = None
        self.collection = None

    def connect(self):
        self.connect_to_mongo()
        self.db = self.connection[settings.MONGO['database']]
        self.collection = self.db[self.collection_name]
        self._connected = True

    def __getattr__(self, name):
        if not self._connected:
            self.connect()
        return getattr(self.collection, name)

