# -*- coding: utf-8 -*-



import logging

from at.common.utils import get_connection


_log = logging.getLogger(__name__)


class Settings(object):
    __slots__ = ('name', 'enabled')
    def __init__(self, name, enabled=None, disabled=None):
        self.name = name
        self.enabled = enabled if enabled is not None else not disabled
        assert self.enabled is not None

    def get_disabled(self):
        return not self.enabled

    def is_disabled(self):
        return self.get_disabled()
    def is_enabled(self):
        return not self.get_disabled()

    def set_disabled(self, val):
        self.enabled = not val

    disabled = property(get_disabled, set_disabled)


class CheckSettings(object):
    __slots__ = ('person_id', 'table_name', 'all_names')
    table_name = None
    all_names = ()

    def __init__(self, person_id):
        self.person_id = person_id

    def get_all(self):
        with get_connection() as conn:
            cur = conn.execute("SELECT name FROM %s WHERE person_id=%d" % (self.table_name, self.person_id))
            disabled_names = set([c[0] for c in cur])
            return [Settings(name,disabled = name in disabled_names) for name in self.all_names]

    def _check_names(self, names_seq):
        if not set(self.all_names).issuperset( set(names_seq) ):
            raise RuntimeError("Unknown setting names")

    def is_disabled(self, name):
        self._check_names( (name,) )
        with get_connection() as conn:
            cur = conn.execute("SELECT person_id FROM %s WHERE person_id=%d and name=%%s" % (self.table_name, self.person_id), (name,))
            return bool(cur.fetchone())

    def is_enabled(self, name):
        return not self.is_disabled(name)

    def get(self, name):
        return Settings(name, self.is_enabled(name))

    def set(self, setting):
        with get_connection() as conn:
            if not setting.get_disabled():
                conn.execute("DELETE FROM %s WHERE person_id = %d AND name = %%s" % (self.table_name, self.person_id), (setting.name, ))
            else:
                conn.execute("INSERT IGNORE INTO %s(person_id, name) VALUES (%d, %%s)" % (self.table_name, self.person_id), (setting.name, ))

    def set_to(self, name, is_enabled):
        self.set(Settings(name, enabled = is_enabled))

    def disable(self, name):
        self.set_to(name, False)

    def enable(self, name):
        self.set_to(name, True)

    def _set_hidden_settings(self, names_seq):
        self._check_names( names_seq )
        with get_connection() as conn:
                # delete old person settings
                conn.execute("DELETE FROM %s WHERE person_id=%d" % (self.table_name, self.person_id))
                if names_seq: # 
                    insert_part = ",".join( ["(%d,%%s)" % self.person_id]*len(names_seq) )
                    conn.execute("INSERT INTO %s (person_id, name) VALUES %s" % (self.table_name, insert_part), tuple(names_seq) )

    def resolve(self, settings_seq):
        for_hids = [s.name for s in settings_seq if s.disabled]
        self._set_hidden_settings(for_hids)
