# -*- coding: utf-8 -*-


import traceback
import logging

import lxml.etree as ET

from at.common import exceptions
from at.common import Lock
from at.common.utils import get_connection
from at.aux_ import Accesses
from at.aux_ import entries

_log = logging.getLogger(__name__)

SECRET = 'd0n\'th4<|<0urv0t3s1234567890'


class Vote(object):
    def __init__(self, vote_for):
        self.voted = False
        self.vote_for = vote_for
        self.votes = 0
        self.percent = 0
        self.percent_of_max = 0

    def __str__(self):
        return '<option number="%s" votes="%s" percent="%s" percent-of-max="%s" voted="%s"/>' % \
               (self.vote_for, self.votes, self.percent, self.percent_of_max,
                'yes' if self.voted else 'no')

    def node(self):
        return ET.XML(str(self))


class Poll(object):
    def __init__(self, poll_id, options):
        self.poll_id = poll_id
        self.options = options

    def get_uid(self):
        return int(self.poll_id.split('_')[1])

    def store_votes(self, votes):
        uids = set([uid for uid, vote in votes])
        with get_connection() as con:
            try:
                con.execute('DELETE FROM Votes ' \
                            'WHERE poll_id="%s" AND uid IN (%s)' % \
                            (
                            self.poll_id, ','.join([str(uid) for uid in uids])))
                con.execute('INSERT INTO Votes (poll_id, uid, vote) VALUES ' + \
                            ','.join(['("%s", %s, %s)' % \
                                      (self.poll_id, uid, vote) for (uid, vote)
                                      in votes]))
            except:
                _log.error(traceback.format_exc())
                raise

    def get_storage_id(self):
        return (self.get_uid(), self.poll_id)

    @classmethod
    def get_votes(cls, polls):
        sql = 'SELECT poll_id, uid, vote FROM Votes ' \
              'WHERE poll_id IN (%s)'
        param = ','.join(['"%s"' % p[1] for p in polls])
        _log.debug(sql % param)
        d = dict([(p[1], []) for p in polls])
        with get_connection() as conn:
            for row in conn.execute(sql % param):
                d[row[0]].append(tuple(row[1:]))
        return d


class PollManager:
    @classmethod
    def results(self, poll_ids, uid):
        if not poll_ids:
            return ''
        root = ET.Element('polls')
        _log.debug('POLLS: ' + str(poll_ids))
        try:
            polls_votes = Poll.get_votes(
                [Poll(poll_id, []).get_storage_id()
                 for poll_id in poll_ids])
        except:
            _log.error(traceback.format_exc())
            return ''
        _log.debug('VOTES: ' + repr(polls_votes))
        for poll_id, all_votes in list(polls_votes.items()):
            poll = ET.SubElement(root, 'poll', {'id': poll_id})
            try:
                votes = {}
                voters = set()
                for vote in all_votes:
                    vote_for = vote[1]
                    if vote_for not in votes:
                        votes[vote_for] = Vote(vote_for)
                    votes[vote_for].votes += 1
                    if vote[0] == uid:
                        votes[vote_for].voted = True
                    voters.add(vote[0])
                total = sum([v.votes for v in list(votes.values())])
                max_ = max([1] + [v.votes for v in list(votes.values())])
                _log.debug('voters: %r' % (voters,))
                if total:
                    for v in list(votes.values()):
                        v.percent = int(round(100.0 * v.votes / len(voters)))
                        v.percent_of_max = int(round(100.0 * v.votes / max_))
                poll.attrib['voted'] = 'yes' if [v for v in list(votes.values()) if
                                                 v.voted] else 'no'
                for vote in list(votes.values()):
                    poll.append(vote.node())
            except Exception as msg:
                _log.error('Error appending POLL votes for %s: %s' % \
                           (ET.tostring(poll), msg))
        return ET.tostring(root, encoding='utf-8')

    @classmethod
    def vote_for(cls, ai, feed_id, item_no, my_votes):
        if not ai.uid:
            raise exceptions.AccessDenied('Only registered users may vote!')

        post = entries.load_entry(feed_id, item_no)
        Accesses.Access(ai.uid, feed_id, entry=post).assert_can_vote()

        poll_id = post.poll_id
        poll_type = post.poll_type
        if poll_type == 'single':
            my_votes = [my_votes[0]]
        lock = Lock.Lock(poll_id)
        if not lock.acquire(3):
            raise RuntimeError("Can't acquire lock")
        try:
            poll = Poll(poll_id, [])
            poll.store_votes([(ai.uid, int(v)) for v in my_votes])
        finally:
            lock.release()
