# coding=utf-8
# vim: set fileencoding=utf-8
# -*- coding: utf-8 -*-



import datetime
import logging
import traceback
import types
import unicodedata

import lxml.etree as ET
from lxml.builder import ElementMaker
from django.utils import timezone

import at.aux_.models
import at.common.models
from at.common import exceptions
from at.common import Recommendations
from at.common import groups
from at.common import staff
from at.common import utils
from at.common import dbswitch
from at.common.LongJob import LongJob
from at.common.dbswitch import root_rw_session
from at.common.Config.SimpleLayout import (
    load_layouts, get_all_settings, get_default_settings, set_setting,
    xml_setting
)
from at.common.utils import (
    Status, log_exception, stopwatch, check_auth, force_unicode
)
from at.aux_ import Accesses
from at.aux_ import models
from at.aux_ import ProfileStorage
from at.pump import HandlerRegistry
from functools import reduce

TYPE_SERIALIZERS = {
    int: lambda e, v: str(v),
    int: lambda e, v: str(v),
    bool: lambda e, v: str(v).lower(),
    type(None): lambda e, v: '',
}

_log = logging.getLogger(__name__)
EM = ElementMaker(typemap=TYPE_SERIALIZERS)
LAYOUTS_XML = """
<layouts>
    <!-- Дефолтное состояние устанвливается в базе: f7.WidgetNames.enabled -->
    <!-- По умолчанию дефолтное состояние "включен". -->
    <!-- Безусловное "включен" - атрибутом forced="true" -->
    <!-- (Чтобы не реагировал на пользовательское выкл) -->
    <widget type="layout" id="common">
        <widget type="flow" id="common">
            <widget type="pseudo-widget" id="tips"/>
        </widget>
    </widget>

    <widget type="layout" id="friends">
        <widget type="flow" id="left">
            <widget type="pseudo-widget" id="unread-comments"/>
            <widget type="pseudo-widget" id="club-notifications"/>
            <widget type="pseudo-widget" id="unread-likes"/>
            <widget type="pseudo-widget" id="friendsof"/>
            <widget type="pseudo-widget" id="birthday"/>
        </widget>
    </widget>

    <widget type="layout" id="index_widgets">
        <widget type="flow" id="left">
            <widget type="pseudo-widget" id="qu-progress"/>
            <widget type="pseudo-widget" id="add-to-friends"/>
            <widget type="pseudo-widget" id="profile-contacts"/>
            <widget type="pseudo-widget" id="yaru-tsr-clubs"/>
        </widget>
        <widget type="flow" id="center">
            <widget type="pseudo-widget" id="description"/>
            <widget type="pseudo-widget" id="yaru-viewport-blog"/>
        </widget>
        <widget type="flow" id="right">
            <widget type="flow" id="allfriends">
                <widget type="pseudo-widget" id="allfriends-friends"/>
                <widget type="pseudo-widget" id="allfriends-friendsof"/>
            </widget>
            <widget type="pseudo-widget" id="yaru-tsr-polls"/>
            <widget type="pseudo-widget" id="yaru-tsr-posts"/>
        </widget>
    </widget>


    <widget type="layout" id="index_club">
        <widget type="flow" id="left">
            <widget type="pseudo-widget" id="club-menu"/>
            <widget type="flow" id="allfriends">
                <widget type="pseudo-widget" id="club-moderators"/>
                <widget type="pseudo-widget" id="club-members"/>
            </widget>
            <widget type="pseudo-widget" id="club-calendar"/>
            <widget type="pseudo-widget" id="club-news"/>
            <widget type="pseudo-widget" id="club-popular"/>
            <widget type="pseudo-widget" id="club-polls"/>
        </widget>
        <widget type="flow" id="center">
            <widget type="pseudo-widget" id="description"/>
            <widget type="pseudo-widget" id="club-rules"/>
            <widget type="pseudo-widget" id="club-posts"/>
        </widget>
    </widget>
</layouts>
"""
load_layouts(ET.XML(LAYOUTS_XML))


def get_status_change_job(person_id):
    return LongJob(person_id, 'change_blog_status_impl')


_status2handlers = {
    ('normal', 'deleted'): 'DeleteFeedHandler',
    ('deleted', 'normal'): 'UndeleteFeedHandler',
    ('normal', 'dismissed'): 'DismissedFeedHandler',
    ('dismissed', 'normal'): 'ReadmittedFeedHandler',
}


def create_change_status_events(person_id, old_status, new_status):
    HandlerRegistry.put_event('UpdateUserEventsAfterBlogStatusChange', person_id)
    change_handler = _status2handlers.get((old_status, new_status))
    if change_handler:
        HandlerRegistry.put_event(change_handler, person_id)
    else:
        _log.warning("invalid status change (%s, %s) for %s" %
                     (old_status, new_status, person_id))


def _change_blog_status_impl(person_id, status):
    possible_statuses = ['normal', 'deleted', 'dismissed']
    if status not in possible_statuses:
        return Status('Invalid status name')
    with root_rw_session():
        instance = models.Person.get_person(person_id)
        if status == instance.status:
            _log.debug("don't need change blogs status -- already done")
            return Status('Already done')
        job = get_status_change_job(person_id)
        if job.is_active():
            return Status("ERROR", "is_active")
        job.start()
        old_status = instance.status
        instance.status = status
        instance.save()
    create_change_status_events(person_id, old_status, status)
    return Status("Success")


def check_title(title):
    return [_f for _f in [
        unicodedata.category(x) in ['Ll', 'Lu', 'Lt', 'Nd', 'Nl'] \
        for x in title] if _f]


def update_model(model, data):
    allowed_fields = ['sex', 'title', 'title_eng', 'picture_html', 'mood']
    changed = False
    for field in allowed_fields:
        value = data.get(field)
        if value is None:
            continue
        value = force_unicode(value)
        if field == 'sex' and not value:
            value = 'unknown'
        if field in ['title', 'title_eng'] and not check_title(value):
            _log.warn('Prohibited title "%s"' % data['title'])
            continue
        if getattr(model, field) != value:
            changed = True
            setattr(model, field, value)
    return changed


@log_exception
@stopwatch
def UpdateSource(auth_info, data):
    if not auth_info.uid:
        raise exceptions.AccessDenied("Not authorized")
    with root_rw_session():
        source_id = data.pop('feed_id', auth_info.uid)
        model = models.Person.get_person(int(source_id))
        if update_model(model, data):
            model.save()
        else:
            _log.debug('Not changed')
        return model


def maybe_create_blog(uid, data):
    try:
        models.Person.get_person(uid)
        return False
    except exceptions.NotFound:
        pass

    if data.get('memorial'):
        status = 'memorial'
    elif data['official']['is_dismissed']:
        status = 'dismissed'
    else:
        status = 'normal'
    try:
        bday = datetime.datetime.strptime(data['personal']['birthday'],
                                          '%Y-%m-%d')
        bdy, bdm, bdd = bday.year, bday.month, bday.day
    except Exception:
        bdy, bdm, bdd = 0, 0, 0
    with root_rw_session():
        person = models.Person(
            person_id=uid,
            login=data['login'],
            title=data['name']['first']['ru'] + ' ' + data['name']['last'][
                'ru'],
            title_eng=data['name']['first']['en'] + ' ' + data['name']['last'][
                'en'],
            creation_date=timezone.now(),
            modification_date=timezone.now(),
            user_type='profile',
            status=status,
            sex={'male': 'man', 'female': 'woman'}.get(
                data['personal']['gender'], 'male'),
            mood='',
            score=1000,
            qu=1,
            bd_year=bdy, bd_month=bdm, bd_day=bdd,
            email=data['work_email'],
            community_type='NONE'
        )
        person.save()
        at.aux_.models.Friend.objects.create(
            uid=person,
            person_id=person,
            fgroup_id=groups.GroupType.OWNER,
            mutually=1,
        )
        at.aux_.models.Follower.objects.create(
            uid=person,
            person_id=person,
        )
        at.common.models.MailSettings.objects.create(
            person=person,
        )

    HandlerRegistry.put_event('AutofriendshipManager', uid)


def set_profile_group(ai, rd, category, groups, feed_id=None):
    feed_id = feed_id or ai.uid
    access = Accesses.Access(ai.uid, feed_id)
    access.assert_can_update_profile()
    groups = set(groups.split(','))
    # Костыль для сохранения css-стилей и, потенциально, любых значений,
    # которые не надо сплитить по "," AT-1505
    no_split_groups = rd['no_split_groups'].split(',')
    new_items = [
        (arg, value.strip()) for arg, value in list(rd.items())
        if arg in groups and arg in no_split_groups
        ]
    new_items.extend(
        reduce(
            lambda x, y: x + y,
            (
                [(arg, val) for val in value.split(',')]
                for arg, value in list(rd.items())
                if arg in groups and arg not in no_split_groups
            ),
            []
        )
    )
    new_items = set([(x[0], x[1].strip()) for x in new_items])
    new_items = [x for x in new_items if x[1] != '']
    entries_for_add = [(name, value, 'text') for name, value in new_items]
    ProfileStorage.put(feed_id, category, entries_for_add)
    return entries_for_add


class Profile:
    # Widget-related:
    @utils.et2xml
    @log_exception
    def GetPageLayout(self, ai, page, feed_id=None):
        if feed_id is None:
            feed_id = ai.uid
        try:
            settings = get_all_settings(feed_id)
        except:
            _log.error(traceback.format_exc())
            settings = get_default_settings()
        from at.common import Config
        layout = Config.SimpleLayout.Layouts.get(page,
            Config.SimpleLayout.getNullLayout(page))
        return ET.ElementTree(layout.node(settings, {}))

    @utils.et2xml
    @log_exception
    def SetWidgetSettingsByID(self, ai, request):
        feed_id = request['feed_id']
        show = request['show']
        widget_id = request['widget_id']

        access = Accesses.Access(ai.uid, feed_id)
        access.assert_can_update_profile()
        try:
            _id, enabled = show.split('=')
            assert _id == widget_id
            set_setting(feed_id, widget_id, enabled)
            return ET.ElementTree(ET.Element('completed'))
        except Exception:
            _log.error('Unexpected data for SetWidgetSettings: ' \
                       'show = "%s", widget_id = "%s"' % (show, widget_id),
                       exc_info=True)
            return ET.ElementTree(ET.Element('failed'))

    @utils.et2xml
    @log_exception
    def GetRecommendations(self, uid, friends_count, clubs_count, ai):
        # сломалось в какой-то момент
        # XXX This method is only for debugging and evaluation!
        root = ET.Element('recommendations')
        friends_el = ET.Element('friends')
        api = staff.Api.with_ticket_for_user(ai)
        person = api.persons.filter(uid=uid).fields(['location', 'language'])[0]
        colleagues = Recommendations.get_colleagues(uid, friends_count, api=api)
        for uid in colleagues:
            friends_el.append(ET.Element('person', {'uid': str(uid)}))
        clubs_el = ET.Element('clubs')
        clubs = Recommendations.get_recommended_clubs(colleagues, clubs_count)
        if Recommendations.speaks_russian(person, api=api):
            for club_id in Recommendations.RECOMMENDED_CLUBS_RU + \
                    Recommendations.get_office_clubs(
                        person['location']['office']['code']):
                clubs.append((club_id, 0, None))
        for uid, score, _ in clubs:
            clubs_el.append(
                ET.Element('club', {'uid': str(uid), 'score': str(score)}))
        root.append(friends_el)
        root.append(clubs_el)
        return ET.ElementTree(root)

    @utils.et2xml
    @log_exception
    def GetAllWidgetSettings(self, feed_id):
        root = ET.Element('widget-settings')
        for widget, enabled in list(get_all_settings(feed_id).items()):
            root.append(xml_setting(widget, enabled))
        _log.debug(ET.tostring(root))
        return ET.ElementTree(root)

    # / widget info

    @utils.et2xml
    @log_exception
    def GetProfileCategoriesByID(self, ai, feed_id, categories):
        # я не знаю когда это происходит, но в логах мешает
        # когда будет жалоба с воспроизвдением — отладим/уберем
        if ai is None:
            msg = 'ai is None in GetProfileCategoriesByID. feed_id=%s, categories=%s'
            _log.warning(msg, feed_id, categories)
            return '<source-profile feed-id="%s" />' % feed_id

        def set_category(root, catname, data_tuples):
            cat = ET.SubElement(root, 'category', {'name': catname})
            _log.debug(str(data_tuples))
            for data_tuple in data_tuples:
                # XXX before, we were storing 4-tuple in cache,
                # now switching to 3-tuple. After cache invalidates,
                # may write simply "for name, value, ct in data_tuples"
                name, value, content_type = data_tuple[-3:]
                entry = ET.SubElement(cat, 'entry')
                ET.SubElement(entry, 'name').text = name
                ET.SubElement(entry, 'category').text = catname
                if content_type == 'xml':
                    try:
                        entry.append(
                            ET.XML('<value>%s</value>' % force_unicode(value)))
                    except:
                        _log.warn(traceback.format_exc())
                else:
                    ET.SubElement(entry, 'value').text = utils.force_unicode(
                        value)
            return cat

        cats = [c.strip() for c in categories.split(',') if c.strip()]
        root = ET.Element('aux')

        profile = ET.SubElement(root, 'source-profile',
                                {'feed-id': str(feed_id)})
        for cat, data_tuples in ProfileStorage.get(ai, feed_id,
                                                   cats).items():
            set_category(profile, cat, data_tuples)

        requested = ET.SubElement(root, 'requested-categories')
        for cat in cats:
            ET.SubElement(requested, 'category', {'name': cat})
        return ET.ElementTree(root)

    @utils.et2xml
    @log_exception
    @check_auth(1)
    def SetProfileGroupForID(self, ai, request):
        category = request['category']
        groups = request['groups']
        feed_id = request['feed_id']
        set_profile_group(ai, request, category, groups, feed_id)
        return Status("Success")

    @utils.et2xml
    @log_exception
    @check_auth(1)
    def SetBasicFeedInfoForID(self, ai, request):
        feed_id = request['feed_id']
        updated_source = {"feed_id": feed_id}

        def SetSourceEntry(input_name, native_name=None):
            if native_name is None:
                native_name = input_name
            input_value = request.get(input_name) or None
            if input_value is not None:
                updated_source[native_name] = input_value

        SetSourceEntry("title")
        SetSourceEntry("title", "title_eng")
        SetSourceEntry("sex")
        SetSourceEntry("id", "picture_html")
        UpdateSource(ai, updated_source)
        return Status("Success")

    # no http
    @utils.et2xml
    @log_exception
    def DismissBlog(self, uid):
        return _change_blog_status_impl(uid, status='dismissed')

    # no http
    @utils.et2xml
    @log_exception
    def ResurrectBlog(self, uid):
        return _change_blog_status_impl(uid, status='normal')

    @utils.et2xml
    @log_exception
    def CanChangeStatus(self, feed_id):
        job = get_status_change_job(feed_id)
        if job.is_active():
            return '<status_change>running</status_change>'
        else:
            return '<status_change>possible</status_change>'


Profile = utils.decorate(Profile, utils.stopwatch)
