# coding: utf-8



from collections import defaultdict
import logging

from at.common import utils
from at.common import staff
from at.common import dbswitch
from at.aux_ import Accesses
from at.aux_ import models

_log = logging.getLogger ('aux_/profile')


def get_language(uid):
    try:
        setting = get_entry(uid, 'OptionsOthers', 'ui-lang')[0]
        assert setting, "Empty ui-lang means inherit from intranet"
        assert setting in ['ru', 'en'], "Invalid language value"
        return setting
    except (AssertionError, IndexError):
        return utils.blackbox.userinfo(uid, '127.0.0.1',
                [('userinfo.lang.uid', 'lang')])['fields']['lang']


def get_category(ai, feed_id, category):
    data = get( ai, feed_id, [category]).get(category, [])
    d = defaultdict(list)
    for name, value, content_type in data:
        d[name].append((value, content_type))
    return d


def get(ai, feed_id, categories):
    access = Accesses.Access(ai.uid, feed_id)
    access.assert_can_read_blog()

    categories = set(categories)
    result = {}
    if not categories:
        return result

    if not(utils.is_community_id(feed_id)) and \
        ('Minimum' in categories or 'Personal' in categories):
        if 'Minimum' in categories:
            categories.remove('Minimum')
            person = models.Person.get_by(person_id=feed_id)
            if person is not None:
                login = person.login
            else:
                logging.error('Person for feed_id %s was not found', feed_id)
                return result
            result['Minimum'] = [
                    ('website', 'http://staff.yandex.ru/%s' % login, 'text'),
                    ('email', '%s@yandex-team.ru' % login, 'text')
                    ]
        if 'Personal' in categories:
            categories.remove('Personal')
            result.update(_get_from_staff(ai, feed_id))
    if categories:
        for cat, data in _get(feed_id).items():
            if cat in categories: result[cat] = data
    return result


def get_entry(feed_id, category, entry):
    sql = 'SELECT value FROM Profile WHERE feed_id = %s AND category = %s AND name = %s'
    with utils.get_connection() as conn:
        return [r[0] for r in conn.execute(sql, (feed_id, category, entry))]


# ==== Private methods
def _get_from_staff(auth_info, feed_id):
    try:
        api = staff.Api.with_ticket_for_user(auth_info)
        user = api.persons.filter(uid=feed_id).\
                fields(['name', 'personal', 'login'])[0]
        lang = get_language(auth_info.uid)
        if lang not in user['name']['first']:
            # Uses first existing language from staff
            lang = list(user['name']['first'].keys())[0]
        bday = user['personal'].get('birthday') or ''
        data = {
                'birth_year': bday[0:4],
                'birth_month': bday[5:7],
                'birth_day': bday[8:10],
                'sex': 'woman' if user['personal']['gender'] == 'F' else 'man',
                'first_name': user['name']['first'][lang],
                'last_name': user['name']['last'][lang],
                }
        data['signature'] = data['first_name'] + ' ' + data['last_name']
        return {'Personal': [(k, v, 'text') for (k, v) in data.items()]}
    except Exception:
        # Warning потому что из-за анонимных запросов API
        # у нас слишком много этих записей в логе
        _log.warning('User %s profile not retrieved from Staff API', feed_id, exc_info=1)
        return {}


def _get(feed_id):
    sql = 'SELECT category, name, value, content_type '\
            'FROM Profile WHERE feed_id = %s'
    data = defaultdict(list)
    with utils.get_connection() as conn:
        for row in conn.execute(sql, (feed_id,)):
            data[row[0]].append(tuple(row[1:]))
    return data


def put(feed_id, category, rows):
    with utils.get_connection() as conn:
        conn.execute(
            'DELETE FROM Profile WHERE feed_id = %s AND category = %s',
            (feed_id, category)
        )
        if rows:
            conn.execute(
                'INSERT INTO Profile VALUES ' + \
                (','.join(['(%s, %s, %s, %s, %s)'] * len(rows))),
                sum([(feed_id, category) + r for r in rows], ())
            )
