# -*- coding: utf-8 -*-


from itertools import groupby
import logging

import lxml.etree as ET

from at.common import dbswitch
from at.common.Lock import Lock
from at.common.utils import (
    log_exception, stopwatch, Status, check_auth, get_connection, et2xml
)
from at.aux_.UserEvents import EventList


_log = logging.getLogger ('aux_')


def get_entry_tuple(entry_id):
    parts = entry_id.split('.')
    if len(parts) == 2:
        parts += (0,)
    if len(parts) == 3:
        return tuple(int(x) for x in parts)
    else:
        return None


#=====================================================================


@stopwatch
@log_exception
def IncrementCounter(connection, person_id, value):
    sql = 'INSERT INTO RecentActionsCounter(person_id, type, count, answered) '\
            'VALUES (%s, "comment_me", %s, 1) '\
            'ON DUPLICATE KEY UPDATE count = count + %s'
    connection.execute(sql, (person_id, value, value))

@stopwatch
@log_exception
def DecrementCounter(connection, person_id, value = 1):
    sql = 'UPDATE RecentActionsCounter SET count = count - %s '\
            'WHERE person_id = %s AND type = "comment_me" AND answered = 1'
    connection.execute(sql, (value, person_id))


def RecentActionsLock(person_id):
    return Lock("recent_actions_%d" % person_id)

def ids2str(ids):
    ret = ','.join(['(%d, %d, %d)' % tuple(_id) for _id in ids if len(_id) == 3])
    if ret:
        return ret
    return "(null, null, null)"



class RecentActions:
    @check_auth(1)
    @stopwatch
    @et2xml
    def ReceivedCommentsTotalCount(self, ai, tag):
        with get_connection() as connection:
            query = """SELECT sum(count) FROM RecentActionsCounter
                        WHERE person_id = %s AND type = 'comment_me'"""
            count, = connection.execute(query, (ai.uid,)).fetchone()
            node = ET.Element(tag, {'uid': str(ai.uid)})
            node.text = str(count or 0)
            return ET.ElementTree(node)
        

    @check_auth(1)
    @stopwatch
    def MarkAllNotificationsAsRead(self, ai, request):
        person_id = ai.uid
        _log.debug("Mark every notification for %s", person_id)
        comment_notifications = EventList(person_id,
                    ['CommentNotification', 'MentionNotification'], ['seen'])
        for event in comment_notifications:
            try:
                event.state.update(('seen', ))
            except Exception:
                _log.warning(
                    "Can't mark event %s for person_id %s as seen",
                    event.id, person_id)
        comment_notifications.save()
        return Status('Success')

    @et2xml
    @log_exception
    @check_auth(1)
    @stopwatch
    def MarkCommentNotificationsByItemIDs(self, ai, request):
        ids_str = request['ids_str']
        mark_str = request['mark_str']
        marks = mark_str.split(',')
        ids = set(get_entry_tuple(id_str) for id_str in ids_str.split(','))
        ids.discard(None)
        is_saved = False
        for (feed_id, item_no), group in groupby(ids, key=lambda t: (t[0], t[1])):
            # normally it should always be only one group
            comment_ids = set( comment_id for _, _, comment_id in group )
            events = EventList(ai.uid, ['CommentNotification', 'MentionNotification', 'Like'],
                               ['seen']).filter(feed_id=feed_id, item_no=item_no)
            if events:
                [
                    event.state.update(marks) for event in events
                    if event.object.get('comment_id', 0) in comment_ids
                ]
                events.save()
                is_saved = True

        return Status('Success' if is_saved else 'NotFound')


