# -*- coding: utf-8 -*-



import datetime
import time
import logging
from collections import defaultdict

from lxml import etree as ET

from at.aux_.models import Friend
from at.common import utils
from at.common.groups import GroupType

_log = logging.getLogger ('aux_/relations')


def _cleanFriendGroups(uid, person_id, groups):
    Friend.objects.filter(fgroup_id__in=groups, uid=uid, person_id=person_id).delete()


def addRelation(uid, person_id, group=GroupType.FRIEND, clear_all_groups=False):
    if clear_all_groups:
        groups = list(GroupType._num2str.keys())
    else:
        groups = GroupType.expand_role(group)
        groups.add(GroupType.BANNED)
        # костыль для AT-2018 – в других местах нам не надо,
        # чтоб expand(MODERATOR) возвращал MODERATOR_INVITED, а тут надо
        if group == GroupType.MODERATOR:
            groups.add(GroupType.MODERATOR_INVITED)
    _log.warn(str(groups))
    _cleanFriendGroups(uid, person_id, groups)
    Friend.objects.create(uid_id=uid, person_id_id=person_id, fgroup_id=group)


def removeRelation(uid, person_id, group=GroupType.FRIEND):
    groups = GroupType.upscale_role(group)
    _cleanFriendGroups(uid, person_id, groups)


class Relation(object):
    __slots__ = ('uid', 'feed_id', 'role', 'add_date', 'mutual')

    def __init__(self, uid, feed_id, role, add_date, mutual):
        self.uid = uid
        self.feed_id = feed_id
        self.role = role
        # to unixtime for some reason
        if isinstance(add_date, datetime.datetime):
            self.add_date = time.mktime(add_date.timetuple())
        else:
            self.add_date = add_date
        self.mutual = int(mutual)

    @property
    def important(self):
        return (self.uid, self.feed_id, self.role)

    def __eq__(self, other):
        return self.important == other.important

    def __hash__(self):
        # XXX Очень плохо! Работает только при условии,
        # что мы всегда сразу заполняем все поля объекта и не меняем потом
        return hash(self.important)

    def __str__(self):
        return 'Role [%s]' % str(self.important)

    def to_xml(self, node='member'):
        node = ET.Element(node)
        for field in self.__slots__:
            ET.SubElement(node, field).text = str(getattr(self, field))
        ET.SubElement(node, 'role_type').text = GroupType.get_str(self.role)
        return node


class RelationsList(list):
    def __init__(self, iterable=None):
        self.__set = set()
        for el in iterable or []:
            self.append(el)

    def append(self, item):
        if item not in self.__set:
            self.__set.add(item)
            super(RelationsList, self).append(item)

    def extend(self, items):
        for item in items:
            self.append(item)


    def to_xml(self, root_node='members', member_node='member'):
        node = ET.Element(root_node)
        for relation in self:
            node.append(relation.to_xml(member_node))
        return node

    def __str__(self):
        return 'RelationsList<%s>' % ', '.join(map(str, self))

    def expand(self):
        for rel in self[:]:
            # FIXME переписать более эффективно, группируя по пользователю
            # м.б. вообще переделать RelationsSet, чтоб он был строго для одной пары (uid, feed_id)
            # но тогда NB надо убрать флажок flat в Relations.get_relations_multi
            for role in GroupType.expand_roles([rel.role]):
                self.append(Relation(rel.uid, rel.feed_id, role, rel.add_date, rel.mutual))


def _join(ids):
    return ','.join(map(str, ids))


ZERO_RELATION = lambda uid, feed_id: Relation(uid, feed_id, GroupType.USER, 0, 1)


def get_relations_multi(uids, feed_ids, flat = False):
    # NB: Не экспандит роли!
    if not feed_ids and not uids:
        _log.warning('Empty feed_ids and uids in get_relations_multi', feed_ids)
        return []
    with utils.get_connection() as conn:
        sql = 'SELECT uid, fgm.person_id, fgroup_id, add_date, mutually '\
                'FROM FriendGroupMember fgm join persons p ON (p.person_id = fgm.uid) '\
                'WHERE ' + \
                ( ('fgm.person_id IN (%s) AND ' % _join(feed_ids)) if feed_ids else '') + \
                ( ('fgm.uid IN (%s) AND ' % _join(uids)) if uids else '') + \
                'p.status = "normal" '\
                'ORDER BY fgm.uid, fgm.person_id'
        # Это можно написать изящно на itertools и comprehensions,
        # но получается совсем нечитаемо:
        result = {}
        key = None
        for row in conn.execute(sql):
            value = tuple(row)
            if not flat:
                key = value[:2]
            if not key in result:
                result[key] = RelationsList([ZERO_RELATION(value[0], value[1])])
            result[key].append(Relation(*value))
        if flat:
            return result.get(None, RelationsList())
        else:
            return result

def get_relations(uid, feed_id):
    # NB: Не экспандит роли!
    return get_relations_multi([uid], [feed_id]).get(
            (uid, feed_id),
            RelationsList([ZERO_RELATION(uid, feed_id)])
    )

def get_relations_sql_condition(roles, inverted=False, status='normal', mutually=None):
    all_roles = GroupType.upscale_roles(roles)
    if not inverted:
        feed, people = 'person_id', 'uid'
    else:
        feed, people = 'uid', 'person_id'
    return  ('FROM FriendGroupMember fgm join persons p ON (p.person_id = fgm.%s) '\
            'WHERE fgm.%s = %%s AND fgroup_id IN (%s) ' +\
            ('AND mutually = %d ' % int(mutually) if mutually is not None else '') +\
            'AND p.status = "%s" ') % (people, feed, _join(all_roles), status)

def get_relations_counters(feed_id, roles=[GroupType.FRIEND], inverted=False, mutually=None):
    ''' Счётчик с группировкой по ролям '''
    if type(roles) == int:
        roles = [roles]
    sql = 'select fgroup_id, mutually, count(*) ' + \
            get_relations_sql_condition(roles, inverted, 'normal', mutually) + \
            'GROUP BY fgroup_id, mutually '
    res = defaultdict(int)
    try:
        with utils.get_connection() as conn:
            for row in conn.execute(sql, (feed_id,)):
                res[row[:2]] = row[2]
    except Exception:
        _log.error('get_relations_counters failed with %s', locals())
    return res

def get_total_relations_counters(feed_id, roles=[GroupType.FRIEND], inverted=False, mutually=None):
    '''Суммарные счётчики людей с ролями не ниже roles '''
    if type(roles) == int:
        roles = [roles]
    result = defaultdict(int)
    counters = get_relations_counters(feed_id, roles, inverted, mutually)
    for role in roles:
        result[role] = sum([v for (k, _), v in list(counters.items())
                            if k in GroupType.upscale_role(role)], 0)
    return result


def list_feed_relations(feed_id, roles, mutually=None, count=0, page=0,
                        inverted=False, order_by='add_date DESC', status='normal'):
    """ Список релейшенов, которые в данном фиде имеют роль не ниже данной.
        Если inverted, то наоборот – где данный фид имеет роль. """
    if type(roles) == int:
        roles = [roles]
    sql = 'SELECT uid, fgm.person_id, fgroup_id, unix_timestamp(add_date), mutually ' + \
            get_relations_sql_condition(roles, inverted, status, mutually) + \
            ('ORDER BY %s ' % order_by) + \
            ('LIMIT %d, %d ' % (page * count, count) if count else '')
    with utils.get_connection() as conn:
        return RelationsList([Relation(*tuple(row)) for row in conn.execute(sql, (feed_id, ))])


def serialize_grouplist(data):
    root = ET.Element('aux')
    for key, (count, users) in data.items():
        aux = ET.SubElement(ET.SubElement(root, key), 'aux')
        ET.SubElement(aux, 'count').text = str(count)
        aux.append(users.to_xml())
    return root

