# coding: utf-8



import re
import logging

from lxml import etree as ET

from django.conf import settings

from at.common import startrek_client
from at.common import oauth
from at.common import isearch
from at.common.utils import *
from at.common import assertions

_log = logging.getLogger(__name__)

# Identifier of the role with per-project assignees.
ASSIGNEE_ROLE = 10001
# Identifier of issue type "Bug"
TYPE_BUG = '1'
# Identifier of the CC field
CUSTOM_FIELD_CARBON_COPY = 'customfield_10170'

MATCH_KEY_PRIO = 0
MATCH_NAME_PRIO = 1

STARTREK_PATH = settings.STARTREK_API_URL


class StarTracker(object):
    key = 'startrek'
    reg = re.compile('https?://st.yandex-team.ru/(\w+)-(\d+)$', re.U | re.I)

    def create_issue(self, projectKey, reporter, assignee, summary, body,
                     copy_to=None):
        _log.debug('CREATE ISSUE %s',
                   (projectKey, reporter.login, assignee, summary, body, copy_to))
        client = startrek_client.Startrek(reporter.user_ticket, 'At-backend', STARTREK_PATH)
        issue_dict = {
            'queue': projectKey,
            'summary': summary,
            'description': body,
        }
        if assignee:
            issue_dict['assignee'] = getAuthInfo(login=assignee).uid
        if copy_to is not None:
            issue_dict['followers'] = [getAuthInfo(login=copy_to).uid]
        _log.debug('Will create issue %s', issue_dict)
        try:
            issue = client.create_issue(issue_dict)
        except startrek_client.models.NotFound:
            assertions.assertion(False, 'project', 'NOT_VALID', 'jira')
        except startrek_client.models.StartrekError as er:
            if er.messages.get('summary'):  # XXX looks suspicious
                assertions.assertion(False, 'title', 'NOT_VALID', 'jira')
            else:
                # неизвестная ошибка, залогируем и ререйзнем чтобы отвалиться с красной плашкой
                _log.error('Unknown error while creating ticket %s', projectKey,
                           exc_info=True)
                raise

        return issue.key


STARTREK = StarTracker()


class Trackers:
    @et2xml
    @log_exception
    def GetTrackersProjects(self, ai, query):
        """ Autocomplete projects """
        results = isearch.get_queues_suggest(auth=ai, text=query)

        root = ET.Element('projects')
        for result in results:
            item = ET.SubElement(root, 'project')
            ET.SubElement(item, 'key').text = result['id']
            ET.SubElement(item, 'name').text = result['title']
        return ET.ElementTree(root)

    @et2xml
    @log_exception
    def GetTrackersCandidates(self, queue, uid, ai):
        users = []
        project_lead = ''
        try:
            client = startrek_client.Startrek(
                ai.user_ticket,
                'At-backend',
                STARTREK_PATH,
            )
            queue = client.get_queue(queue)
            project_lead = queue.get_lead()
            for user in queue.get_team():
                try:
                    users.append(getAuthInfo(login=user.id))
                except:
                    pass
        except startrek_client.models.StartrekError:
            _log.warn("Error while getting queue info '%s' with uid %s",
                      queue,
                      uid,
                      exc_info=1
                      )
        root = ET.Element('actors')
        for user in users:
            item = ET.SubElement(root, 'actor').text = str(user.uid)
        root.set('lead', project_lead)
        return ET.ElementTree(root)
