# -*- coding: utf-8 -*-



import os
from collections import defaultdict
import logging
import traceback


import lxml.etree as ET

from at.common.utils import (
    stopwatch, log_exception, et2xml, Status, get_connection
)
from at.common.item_loader import per_item_loader
from at.aux_ import Polls
from at.aux_ import Tags
from at.aux_ import UserEvents
from at.aux_.Items import parse_item_id
from at.aux_.CommunityOptions import CommunityStat
from at.aux_.Likes import get_like_counts

log = logging.getLogger(__name__)


def _get_uids(uid_str):
    return list(set([int(uid.strip()) for uid in uid_str.split(',') if uid]))


class XDecorators(object):

    @et2xml
    @stopwatch
    @log_exception
    @et2xml
    def deco_GetPostRubrics(self, ids_string):
        id_tuples = [
            tuple(id_str.split('.'))
            for id_str in ids_string.split(",")
        ]
        sql = """
            SELECT person_id, post_no, rubric_id, r.metarubric
            FROM Posts p JOIN CommunitiesRubrics r USING (rubric_id)
            WHERE %s
        """ % ' OR '.join([
            'person_id=%s AND post_no=%s' % id_tuple
            for id_tuple in id_tuples
        ])
        rows = get_connection().execute(sql)

        root = ET.Element("posts-rubric-info")
        for person_id, post_no, rubric_id, metarubric in rows:
            feed_item = '%s.%s' % (person_id, post_no)
            ET.SubElement(root, "post", {
                "feed-item": feed_item,
                'rubric': str(rubric_id),
                'metarubric': metarubric
            })
        return ET.tostring(root)


    @stopwatch
    @log_exception
    def deco_GetSubscriptions(self, ai, ids_string):
        # я не знаю когда это происходит, но в логах мешает
        # когда будет жалоба с воспроизвдением — отладим/уберем
        if ai is None:
            log.warning(
                'ai is None in deco_GetSubscriptions. ids=%s', ids_string)
            return '<subscriptions/>'

        def load_subscriptions(ids_set):
            query = """SELECT feed_id, item_no, 1 FROM Subscriptions
                        WHERE uid = %s AND (%%s)""" % (ai.uid,)
            return per_item_loader(
                ids_set,
                query,
                ['feed_id', 'item_no'],
            )

        xml = ET.Element('subscriptions')
        ids_set = ids_string.split(",")
        subscriptions = load_subscriptions(ids_set)
        for id in ids_set:
            status = subscriptions.get(id, 0)
            xml.append(
                xml.makeelement(
                    'post',
                    {'feed-item': id, 'status': str(status)}
                )
            )
        return ET.tostring(xml)


    @et2xml
    @stopwatch
    @log_exception
    def deco_GetLikeCounts(self, ai, ids_string):
        # К id номеров постов нужно добавлять .0, чтобы лайки
        # постов и комментариев можно
        # было получать одним запросом,
        # используя общий код per_item_loader.
        full_ids = ['%s.%s.%s' % parse_item_id(id) for id in ids_string.split(',')]
        counts = get_like_counts(full_ids)

        # Check if the user liked the posts himself
        my_likes = defaultdict(int)

        if ai and ai.uid and counts:
            feed_item_ids = set( key for key, value in counts.items() if value[0] )
            if feed_item_ids:
                template = 'select feed_id, item_no, comment_id, value from Likes where (%s) and uid = %s'
                condition = ' or '.join(
                    '(feed_id=%s and item_no=%s and comment_id=%s)' %
                    tuple(id.split('.')) for id in feed_item_ids
                )
                with get_connection() as connection:
                    cursor = connection.execute(template % (condition, ai.uid))
                    my_likes.update(('%s.%s.%s' % tuple(row)[:-1], row[3]) for row in cursor)

        # Converting to XML
        root = ET.Element('likes')
        list(map(root.append,
                [root.makeelement('like', {'id': id.partition('.0')[0], 'count': str(count), 'total': str(total), 'i-like': str(my_likes[id])})
                for (id, (total, count)) in counts.items()]))

        # Voila!
        return ET.tostring(root)

    @et2xml
    @stopwatch
    @log_exception
    def deco_MarkModeration(self, ai, post_coord_str):
        if not ai.uid:
            return Status('OK')
        for coord in set(post_coord_str.split(',')):
            feed_id, item_no = list(map(int, coord.split('.')))
            events = UserEvents.EventList(ai.uid, ['ClubNotification'], ['seen']).\
                    filter(feed_id=feed_id, item_no=item_no)
            for e in events: e.state.update({'seen'})
            events.save()
        return Status('OK')

    @et2xml
    @stopwatch
    @log_exception
    def deco_PollVotes(self, ai, poll_id_str):
        try:
            poll_ids = list(set([p for p in poll_id_str.split(',')]))
            return Polls.PollManager.results(poll_ids, ai.uid)
        except:
            log.error(traceback.format_exc())
            return "<polls/>"

    @et2xml
    @stopwatch
    @log_exception
    def deco_FeedItemsTags(self, feed_item_str):
        try:
            feed_items = set( feed_item for feed_item in feed_item_str.split(',') )
            feedItemsTags = Tags.t.getFeedItemsTagList(list(feed_items))
            rootElement = ET.Element("posts-tags")
            for feed_item, tag_id, title in feedItemsTags:
                tagElement = ET.SubElement(rootElement, 'post-tag')
                tagElement.attrib['feed-item'] = feed_item
                ET.SubElement(tagElement, 'id').text = str(tag_id)
                ET.SubElement(tagElement, 'title-tag').text = title
            return ET.tostring(rootElement)
        except:
            log.error(traceback.format_exc())
            return "<posts-tags/>"

    @et2xml
    @stopwatch
    @log_exception
    def deco_CommunityStat(self, uid_str):
        try:
            uids = _get_uids(uid_str)
            root = ET.Element("community-stat")
            for cs in CommunityStat.select(uids):
                cs.append_to(root)
            return ET.tostring(root)
        except:
            log.error(traceback.format_exc())
            return "<sources/>"

