# coding: utf-8



from django.contrib import admin
from django import forms

from at.aux_ import models
from at.common.groups import GroupType


class PersonAdmin(admin.ModelAdmin):

    list_display = (
        'login',
        'person_id',
        'community_type',
        'title',
    )
    search_fields = (
        'login',
        'person_id',
    )
    list_filter = (
        'community_type',
        'status',
    )


class FollowerAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'uid_login',
        'person_id_login',
    )
    search_fields = (
        'person_id__login',
        'uid__login',
    )
    list_select_related = (
        'uid',
        'person_id',
    )

    raw_id_fields = (
        'uid',
        'person_id',
    )

    show_full_result_count = False

    def uid_login(self, obj):
        return obj.uid.login

    def person_id_login(self, obj):
        return obj.person_id.login


class FriendAdmin(admin.ModelAdmin):
    list_display = (
        'id',
        'uid_login',
        'person_id_login',
        'mutually',
        'add_date',
    )
    search_fields = (
        '^person_id__login',
        '^uid__login',
    )
    list_select_related = (
        'uid',
        'person_id',
    )

    raw_id_fields = (
        'uid',
        'person_id',
    )
    list_filter = (
        'fgroup_id',
    )

    def uid_login(self, obj):
        return obj.uid.login

    def person_id_login(self, obj):
        return obj.person_id.login

    def fgroup_id_str(self, obj):
        return GroupType._num2str[obj.fgroup_id] + ' (%s)' % str(obj.fgroup_id)


class Moderator(models.Friend):

    class Meta:
        proxy = True
        verbose_name = 'Moderator'
        verbose_name_plural = 'Moderators'


class ModeratorForm(forms.ModelForm):

    fgroup_id = forms.Field(required=False, widget=forms.HiddenInput)

    def clean_fgroup_id(self):
        return GroupType.MODERATOR

    class Meta:
        exclude = [
            'add_date',
            'mutually',
        ]
        labels = {
            'uid': 'Person',
            'person_id': 'Club',
        }


class ModeratorAdmin(FriendAdmin):

    list_display = (
        'id',
        'person_login',
        'club_slug',
        'add_date',
    )

    list_filter = ()

    form = ModeratorForm

    def person_login(self, obj):
        return obj.uid.login

    def club_slug(self, obj):
        return obj.person_id.login

    def get_queryset(self, request):
        queryest = super(ModeratorAdmin, self).get_queryset(request)
        return queryest.filter(fgroup_id=GroupType.MODERATOR)

    def save_model(self, request, obj, form, change):
        obj.fgroup_id = GroupType.MODERATOR
        super(ModeratorAdmin, self).save_model(request, obj, form, change)


admin.site.register(models.Follower, FollowerAdmin)
admin.site.register(models.Friend, FriendAdmin)
admin.site.register(Moderator, ModeratorAdmin)
admin.site.register(models.Person, PersonAdmin)
