# coding: utf-8


import logging

from django.conf import settings
from xml.sax.saxutils import escape

from at.common import groups
from at.common import Types
from at.common import BodyFormatter
from at.common import utils
from at.common import exceptions
from at.common.Context import ContextBinder
from at.aux_.entries import formatters
from at.aux_.entries import fields
from at.aux_.entries import repositories
from at.aux_.entries import serializers


PREGENERATED_SNIPPET_LENGTH = 40

log = logging.getLogger(__name__)


class MetaPost(type):
    post_types = {}

    def __new__(metakls, kls_name, bases, attrs):
        metakls.create_fields(bases, attrs)
        new_kls = super(MetaPost, metakls).__new__(metakls, kls_name, bases, attrs)
        metakls.register_post_type_class(new_kls)
        return new_kls

    @staticmethod
    def create_fields(bases, attrs):
        """Collect each fields.Field attribute to 'fields'
        attribute regarding last parent 'fields' attribute"""
        parent_type_fields = dict(getattr(bases[-1], 'fields', {}))
        obj_fields = parent_type_fields or {}
        for attr_name, attr_value in list(attrs.items()):
            if isinstance(attr_value, fields.Field):
                obj_fields[attr_name] = attr_value
                default = attr_value.options.get('default', None)
                attrs[attr_name] = default
        attrs['fields'] = obj_fields

    @classmethod
    def register_post_type_class(metakls, new_kls):
        """Fill registry for dispatching class by post_type"""
        post_type = getattr(new_kls, 'type', None)
        if post_type:
            metakls.post_types[post_type] = new_kls


class Tag(object):

    def __init__(self, id, title):
        self.id = id
        self.title = utils.force_unicode(title)

    @property
    def is_unresolved(self):
        return self.id is None

    def __repr__(self):
        return '<Tag: %s: %s>' % (self.id, utils.force_str(self.title))


class Post(object, metaclass=MetaPost):
    do_trackback = False

    default_content_type = "text/wiki"

    feed_id = fields.IntegerField()
    item_no = fields.IntegerField()
    comment_id = fields.IntegerField()
    author_id = fields.IntegerField()
    rubric_id = fields.IntegerField()

    store_time = fields.DatetimeField()
    store_time_microseconds = fields.IntegerField()

    # для поста — время редактирования
    # для коммента это поле почему-то традиционно
    # не хранится и используется только store_time
    item_time = fields.DatetimeField()

    block_comments = fields.BooleanField(integer=True)
    # читать как: «опубликовано минимум один раз»
    published = fields.BooleanField(default=False, integer=True)
    on_moderation = fields.BooleanField(default=0, integer=True)
    original_access_type = fields.StringField()  # TODO: переделать на флажок
    moderated_by = fields.StringField()
    deleted = fields.IntegerField(default=0, serialize_as_bool=True)

    pinned_after = fields.IntegerField()

    parent_comment_id = fields.IntegerField(default=0)
    top_level_parent_comment_id = fields.IntegerField()
    tb_feed_id = fields.IntegerField()
    tb_item_no = fields.IntegerField()
    tb_rubric_id = fields.IntegerField()
    tb_access_type = fields.StringField()
    reply_to_feed_id = fields.IntegerField()
    reply_to_item_no = fields.IntegerField()
    reply_to_comment_id = fields.IntegerField()
    children_count = fields.IntegerField(default=0)

    _content_type = fields.StringField()
    content_version = fields.StringField()

    entry_type = fields.StringField()
    _access_type = fields.StringField(default='')
    access_group = fields.IntegerField()
    cc_address = fields.StringField()
    _title = fields.StringField(default='')
    _body = fields.StringField(default='')
    body_original = fields.StringField(default='')
    _snippet = fields.StringField()

    form_id = fields.StringField()

    score = fields.IntegerField(default=0)
    store_time_interesting_usec = fields.IntegerField()
    store_time_most_interesting_usec = fields.IntegerField()

    repositories = {
        'mysql': repositories.MysqlRepository(),
    }
    default_repository = repositories['mysql']

    def __init__(self, feed_id, item_no=None, comment_id=0, is_new=True):
        self.feed_id = feed_id
        self.item_no = item_no
        self.comment_id = comment_id
        self._snippet = None

        self._trackback = None
        self._parent = None
        self._reply_to = None

        self._formatter = None
        self._content_type = self.default_content_type
        self._tags = []

        # если is_new == True, значит объект еще не сохранен в базу
        self.is_new = is_new

    def __repr__(self):
        return 'models.%s(%d, %d, %d)' % (self.type.capitalize(),
                self.feed_id, self.item_no or 0, self.comment_id or 0)

    def save(self, using=None):
        repository = self.repositories.get(using, self.default_repository)
        assert repository, "Set default repository before using .save"
        return repository.save(self)

    def delete(self, using=None):
        repository = self.repositories.get(using, self.default_repository)
        assert repository, "Set default repository before using .delete"
        return repository.delete(self)

    def validate(self):
        for attr_name, field in list(self.fields.items()):
            field.validate(getattr(self, attr_name))

    @property
    def serializer(self):
        return serializers.get_xml_serializer(self)

    def serialize(self, escape_html=False, pretty=False):
        self.validate()
        return self.serializer.serialize(escape_html, pretty=pretty)

    def serialize_xml_content(self, escape_html=False, pretty=False):
        self.validate()
        return self.serializer.serialize_xml_content(escape_html, pretty=pretty)

    def serialize_item(self, escape_html=False, pretty=False):
        self.validate()
        result = self.serializer.serialize_item(escape_html, pretty=pretty)
        if isinstance(result, bytes):
            result = result.decode('utf-8')
        return result

    def serialize_reply(self, escape_html=False, pretty=False):
        self.validate()
        return self.serializer.serialize_reply(escape_html=escape_html)

    def update_fields(self, data):
        for attr_name, field_cls in list(self.fields.items()):
            value = data.get(attr_name)
            if value is not None:
                setattr(self, attr_name, value)
        if 'tags' in data:
            self.tags = data['tags']

    def _get_content_type(self):
        return self._content_type

    def _set_content_type(self, content_type):
        self._formatter = None
        self._content_type = content_type

    content_type = property(_get_content_type,
                            _set_content_type)

    def clone(self):
        instance_clone = self.__class__(
            feed_id=self.feed_id,
            item_no=self.item_no,
            comment_id=self.comment_id
        )
        for field in self.fields:
            setattr(instance_clone, field, getattr(self, field))
        instance_clone.tags = self.tags
        return instance_clone

    def mark_deleted(self, manually=True):
        self.deleted = 1 if manually else 2

    def mark_undeleted(self):
        self.deleted = 0

    @property
    def formatter(self):
        if self._formatter is None:
            # Очень костыльно но санитайзеры назначаются при определении
            # а контент тайп может поменяться в рантайме
            if self.content_type == "text/wiki":
                self._formatter = formatters.WikiFormatter()
            elif self.content_type in ("text/plain", "text/html"):
                raise RuntimeError('Old formatters are removed! Change content type and save as wiki doc')
        return self._formatter

    def _get_body(self):
        if not self._body:
            self._set_body(self.body_original)
        return self._body

    def _set_body(self, value):
        self.body_original = value
        self._body = self.formatter.format_body(value)
        if self.has_trackback:
            self.trackback.body = value

        # если тайтл пустой, то сниппет равен первой строке body
        # и нужно его сбросить
        if not self.title:
            self._snippet = self._make_snippet()

    body = property(_get_body, _set_body)

    def _get_access_type(self):
        if self.access_group is None:
            return None
        if not self._access_type:
            self._access_type = groups.get_str_access(self.access_group)
        return self._access_type

    def _set_access_type(self, value):
        self._access_type = value
        self.access_group = groups.get_int_access(self._access_type)

    access_type = property(_get_access_type, _set_access_type)

    def is_deleted_manually(self):
        return self.deleted == 1

    @property
    def snippet(self):
        if self._snippet is None:
            self._snippet = self._make_snippet()
        return self._snippet

    def _make_snippet(self, snippet_len=PREGENERATED_SNIPPET_LENGTH):
        context = ContextBinder(
            feed_id=self.feed_id,
            item_no=self.item_no,
            comment_id=self.comment_id,
        )
        with context:
            return BodyFormatter.get_firstline(
                # в других местах title экранируется сериализатором
                body=escape(self.title) or self.body,
                maxlen=snippet_len,
            )

    def _get_title(self):
        return self._title

    def _set_title(self, value):
        self._title = self.formatter.format_title(value)
        # title изменился -> _snippet неактуальный
        self._snippet = self._make_snippet()

    title = property(_get_title, _set_title)

    def _set_cc_list(self, value):
        self.cc_address = None if value is None else ','.join(value)

    def _get_cc_list(self):
        if self.cc_address is None:
            return None
        elif self.cc_address:
            return self.cc_address.split(',')
        else:
            return []

    cc_list = property(_get_cc_list, _set_cc_list)

    def _get_tags(self):
        """
        При обращении к tags — через запрос к базе резолвим теги.
        """
        self.resolve_tags()
        return self._tags

    def _set_tags(self, value):
        self._tags = []
        for element in value:
            if isinstance(element, Tag):
                self._tags.append(element)
            elif isinstance(element, tuple):
                id, name = element
                if name:
                    self._tags.append(Tag(id=id, title=name))
            elif isinstance(element, str):
                if element:
                    self._tags.append(Tag(id=None, title=element))
            elif isinstance(element, int):
                self._tags.append(Tag(id=element, title=None))

    def resolve_tags(self, connection=None):
        unresolved = [tag.title for tag in self._tags if tag.is_unresolved]
        if unresolved:
            if self.has_trackback:
                feed_id = self.tb_feed_id
            else:
                feed_id = self.feed_id
            resolved = self.default_repository.resolve_tags(feed_id, unresolved)
            for tag in self._tags:
                if tag.is_unresolved:
                    tag.id = resolved.get(tag.title)

    tags = property(_get_tags, _set_tags)

    @property
    def is_comment(self):
        if self.is_new:
            # у нового комента есть item_no, а у поста нет
            # как обычно неясно 0 тут или None
            return bool(self.item_no)
        else:
            # у загруженного из базы коммента есть comment_id, у поста нет
            return bool(self.comment_id)

    @property
    def has_trackback(self):
        return bool(self.tb_item_no and self.tb_feed_id)

    @property
    def has_parent(self):
        return bool(self.parent_comment_id)

    @property
    def is_trackback(self):
        return bool(self.reply_to_feed_id and self.reply_to_item_no)

    @property
    def trackback(self):
        if self.has_trackback:
            if self._trackback is None:
                self._trackback = load_entry(self.tb_feed_id, self.tb_item_no)
            return self._trackback

    @property
    def in_reply_to(self):
        if self.is_trackback:
            if self._reply_to is None:
                try:
                    self._reply_to = load_entry(
                        feed_id=self.reply_to_feed_id,
                        item_no=self.reply_to_item_no,
                        comment_id=self.reply_to_comment_id,
                    )
                except exceptions.NotFound:
                    return
            return self._reply_to

    @property
    def parent(self):
        if self.has_parent:
            if self._parent is None:
                self._parent = load_entry(
                    feed_id=self.feed_id,
                    item_no=self.item_no,
                    comment_id=self.parent_comment_id,
                )
            return self._parent

    @property
    def type_id(self):
        return Types.ItemTypes().by_name(self.type)['id']

    def __eq__(self, other):
        return (
            self.__class__ == other.__class__
            and self.item_no == other.item_no
            and self.feed_id == other.feed_id
            and self.comment_id == other.comment_id
        )

    @property
    def _web_url(self):
        """
        Для удобства отладки (посмотреть пост/коммент в браузере),
        не для использования в коде
        """
        return '%s://clubs.%s/%s/%s/%s' % (
            settings.AT_SCHEME,
            settings.AT_HOST,
            self.feed_id,
            self.item_no or 0,
            self.comment_id or 0,
        )

    @property
    def _db_query(self):
        """
        Для удобства отладки (посмотреть запись в базе),
        не для использования в коде
        """
        if self.comment_id is None:
            sql = """
                SELECT content.*, post.*
                FROM Posts post
                INNER JOIN EntryXmlContent as content
                  ON
                    post.person_id = content.feed_id AND
                    post.post_no = content.post_no AND
                    content.comment_id = 0
                WHERE
                  post.person_id = %s and
                  post.post_no = %s
            """ % (
                self.feed_id,
                self.item_no,
            )
        else:
            sql = """
                SELECT content.*, comment.*
                FROM Comments comment
                INNER JOIN EntryXmlContent as content
                  ON
                    comment.person_id = content.feed_id AND
                    comment.post_no = content.post_no AND
                    comment.comment_id = content.comment_id
                WHERE
                  comment.person_id = %s and
                  comment.post_no = %s and
                  comment.comment_id = %s
            """ % (
                self.feed_id,
                self.item_no,
                self.comment_id,
            )
        import textwrap
        return textwrap.dedent(sql)


class Text(Post):
    type = "text"


class Summon(Post):
    type = "summon"

    summon_address = fields.StringField()
    summon_uid = fields.IntegerField()


class Status(Post):
    type = "status"


class Link(Post):
    type = "link"

    url = fields.StringField()
    shared_comment_id = fields.IntegerField(default=0)
    shared_feed_id = fields.IntegerField()
    shared_item_no = fields.IntegerField()


class Poll(Post):
    type = "poll"

    options = None

    poll_type = fields.StringField()
    expires = fields.DateField()
    hidden = fields.BooleanField()
    poll_id = fields.StringField()

    def update_fields(self, data):
        super(Poll, self).update_fields(data)
        self.options = data.get('options', [])


class Congratulation(Post):
    type = "congratulation"

    event = fields.StringField()
    whom = fields.IntegerField()


class Friend(Post):
    type = "friend"
    friender = fields.IntegerField()
    friendee = fields.IntegerField()


class Unfriend(Post):
    type = "unfriend"
    friender = fields.IntegerField()
    friendee = fields.IntegerField()


class Join(Post):
    type = "join"
    person = fields.IntegerField()
    club = fields.IntegerField()


class Unjoin(Post):
    type = "unjoin"
    person = fields.IntegerField()
    club = fields.IntegerField()


class Ticket(Post):
    type = "jira"
    issue = fields.StringField()


def load_entry(feed_id, item_no, comment_id=0, using=None):
    repository = Post.repositories.get(using, Post.default_repository)
    return repository.load(feed_id, item_no, comment_id)


def load_posts_by_ids(id_list, using=None):
    repository = Post.repositories.get(using, Post.default_repository)
    return repository.load_posts_by_ids(id_list)


def load_posts_by_condition(condition, using=None):
    repository = Post.repositories.get(using, Post.default_repository)
    return repository.load_posts_by_condition(condition)


def load_comments_by_ids(id_list, with_deleted=False, using=None):
    repository = Post.repositories.get(using, Post.default_repository)
    return repository.load_comments_by_ids(id_list, with_deleted=with_deleted)


def load_comments_by_condition(condition, join_tables='', using=None):
    repository = Post.repositories.get(using, Post.default_repository)
    return repository.load_comments(condition, join_tables)


def load_entries_by_ids(id_list):
    posts_ids = []
    comments_ids = []
    for id in id_list:
        feed_id, item_no, comment_id = id
        if comment_id:
            comments_ids.append(tuple(id))
        else:
            posts_ids.append((feed_id, item_no))
    entries = []
    if posts_ids:
        entries.extend(load_posts_by_ids(posts_ids))
    if comments_ids:
        entries.extend(load_comments_by_ids(comments_ids))
    return entries


def get_post_class(post_type):
    return MetaPost.post_types.get(post_type, Text)
