# coding: utf-8


import hashlib
import logging
from at.common import Types
from at.common import dbswitch
from at.common import utils
from at.common import exceptions
from at.common.groups import AccessLevel
from at.common.utils import get_connection
from .. import serializers


from .base_repository import BaseRepository


log = logging.getLogger(__name__)


generate_entry_item_no_sql = """
    CALL generate_bulca_item_no(:feed_id)
"""

upsert_post_meta_sql = """
  INSERT INTO Posts
  (
    person_id,
    author_uid,
    post_no,
    store_time,
    store_time_month_year,
    item_time,
    post_type,
    deleted,
    store_time_usec,
    freezed,
    last_updated,
    rubric_id,
    access_group,
    score,
    store_time_interesting_usec,
    store_time_most_interesting_usec,
    on_moderation,
    pinned_after
  )
  VALUES
  (
    :person_id,
    :author_uid,
    :item_no,
    :store_time,
    DATE_FORMAT(:store_time,'%%Y-%%m-01'),
    :item_time,
    :post_type,
    :deleted,
    :store_time_usec,
    :freezed,
    :store_time,
    :rubric_id,
    :access_group,
    :score,
    :store_time_interesting_usec,
    :store_time_most_interesting_usec,
    :on_moderation,
    :pinned_after
  )
  ON DUPLICATE KEY UPDATE
    author_uid=VALUES(author_uid),
    store_time=VALUES(store_time),
    store_time_month_year=VALUES(store_time_month_year),
    item_time=VALUES(item_time),
    post_type=VALUES(post_type),
    deleted=VALUES(deleted),
    store_time_usec=VALUES(store_time_usec),
    is_imported=VALUES(is_imported),
    freezed=VALUES(freezed),
    rubric_id=VALUES(rubric_id),
    access_group=VALUES(access_group),
    score=VALUES(score),
    store_time_interesting_usec=VALUES(store_time_interesting_usec),
    store_time_most_interesting_usec=VALUES(store_time_most_interesting_usec),
    on_moderation=VALUES(on_moderation),
    pinned_after=VALUES(pinned_after)
"""

upsert_post_content_sql = """
  INSERT INTO EntryXmlContent
  (
    feed_id,
    post_no,
    comment_id,
    xml,
    wf_version
  )
  VALUES
  (
    :feed_id,
    :item_no,
    :comment_id,
    :xml,
    :wf_version
  )
  ON DUPLICATE KEY UPDATE
    xml=VALUES(xml),
    wf_version=VALUES(wf_version)
"""

select_trackback_comment_id_sql = """
  SELECT person_id, post_no, comment_id
  FROM Trackbacks
  WHERE
    trackback_person_id = :feed_id AND
    trackback_post_no = :item_no
"""

upsert_post_store_data_sql = """
  INSERT INTO PostStoreData
  (
    feed_id,
    item_no,
    link_md5
  )
  VALUES
  (
    :feed_id,
    :item_no,
    :link_md5
  )
  ON DUPLICATE KEY UPDATE
    link_md5=VALUES(link_md5)
"""
# NB: PRIMARY KEY (`feed_id`,`link_md5`)

select_post_store_data_sql = """
  SELECT item_no FROM PostStoreData
  WHERE
    feed_id = :feed_id
    AND link_md5 = :link_md5
"""

resolve_feed_tags_sql_tpl = """
    SELECT id, title_tag
    FROM PostCategory
    WHERE title_tag IN (%s)
    AND person_id = :feed_id
"""

delete_post_tags_sql = """
    DELETE FROM PostCategories
    WHERE
      feed_id = :feed_id AND
      post_no = :item_no
"""

create_new_tags_sql_tpl = """
    INSERT INTO PostCategory
    (
        person_id,
        title_tag
    )
    VALUES
      %s
"""

insert_post_tags_sql_tpl = """
    REPLACE INTO PostCategories
    (
        feed_id,
        post_no,
        cat_id,
        deleted
    )
    VALUES
      %s
"""

upsert_comment_meta_sql = """
    INSERT INTO Comments
    (
        person_id,
        author_uid,
        post_no,
        comment_id,
        parent_id,
        store_time,
		deleted,
		children_count,
		comment_type
    )
    VALUES
    (
        :person_id,
        :author_uid,
        :item_no,
        :comment_id,
        :parent_id,
        FROM_UNIXTIME(:store_time),
        :deleted,
        :children_count,
        :comment_type
    )
    ON DUPLICATE KEY UPDATE
        author_uid=VALUES(author_uid),
		parent_id=VALUES(parent_id),
		store_time=VALUES(store_time),
		deleted=VALUES(deleted),
		children_count=children_count,
        comment_type=VALUES(comment_type)
"""

set_comment_parents_sql = """
    INSERT INTO CommentsFullParents
	(
	    SELECT
	        person_id,
	        post_no,
	        parent_id,
	        :comment_id,
	        level_diff + 1
	    FROM CommentsFullParents
	    WHERE comment_id=:parent_id
	    AND person_id=:person_id
	    AND post_no=:item_no
	)
	UNION
	(
	    SELECT
	      :person_id,
	      :item_no,
	      :parent_id,
	      :comment_id,
	      1
	)
"""

update_parents_comments_count_sql_tpl = """
    UPDATE
        Comments c,
        CommentsFullParents cpf
    SET
        children_count = children_count %s 1
	WHERE
        c.person_id = cpf.person_id AND
        c.person_id = :feed_id AND
        c.post_no = cpf.post_no AND
        c.post_no = :item_no AND
        c.comment_id = cpf.parent_id AND
        cpf.comment_id = :comment_id
"""

select_post_last_comment_sql = """
    SELECT comment_id, author_uid, store_time
    FROM Comments
    WHERE
        deleted = 0 AND
        person_id = :feed_id AND
        post_no = :item_no
        ORDER BY comment_id DESC
        LIMIT 1
"""

update_post_pin_after = """
    UPDATE Posts
    SET
      pinned_after = :new_pinned_after
    WHERE
      person_id = :feed_id AND
      pinned_after = :old_pinned_after AND
      post_no != :exclude_item_no
"""

count_post_comments_sql = """
    SELECT COUNT(*)
    FROM Comments
    WHERE
      deleted = 0
      AND person_id = :feed_id
      AND post_no = :item_no
      AND comment_id <> 0
"""

update_post_comments_meta_sql = """
    UPDATE Posts
    SET
        children_count = :comments_count,
        last_comment_id = :last_comment_id,
        last_updated = :last_updated,
        last_author_uid = :last_author_uid
    WHERE
      person_id = :feed_id AND
      post_no = :item_no

"""

insert_trackback_sql = """
    INSERT INTO Trackbacks
    VALUES
    (
        :feed_id,
        :item_no,
        :comment_id,
        :trackback_feed_id,
        :trackback_item_no
    )
"""

delete_post_sql = """
    UPDATE Posts
    SET deleted = :deleted
    WHERE
        person_id = :feed_id AND
        post_no = :item_no
"""

delete_comment_sql = """
    UPDATE Comments
    SET deleted = 1
    WHERE
        person_id = :feed_id AND
        post_no = :item_no AND
        comment_id = :comment_id
"""

select_posts_by_condition_sql_tpl = """
    SELECT
      post.person_id,
      post.post_no,
      post.author_uid,
      post.store_time,
      post.item_time,
      post.children_count,
      post.post_type,
      post.freezed,
      post.store_time_usec,
      post.last_updated,
      post.rubric_id,
      post.last_author_uid,
      post.last_comment_id,
      post.access_group,
      post.score,
      post.store_time_interesting_usec,
      post.store_time_most_interesting_usec,
      post.on_moderation,
      post.pinned_after,
      tb.person_id as reply_to_feed_id,
      tb.post_no as reply_to_item_no,
      tb.comment_id as reply_to_comment_id,
      content.xml
    FROM Posts post
    INNER JOIN EntryXmlContent as content
      ON
        post.person_id = content.feed_id AND
        post.post_no = content.post_no AND
        content.comment_id = 0
    LEFT JOIN Trackbacks as tb
      ON
        post.person_id = tb.trackback_person_id AND
        post.post_no = tb.trackback_post_no
    WHERE
      deleted = 0 AND
      %s
"""


class MysqlRepository(BaseRepository):

    def save(self, entry):
        """
        Сохраняет пост или коментарий.
        При создании новых постов или коментариев генерит id объектов
        (в т.ч. id трекбека для коментария) и проставляет их в нужные
        атрибуты модели.

        Про трекбеки.
        Чтобы не сойти с ума делаем так:
            1. Создать трекбек можно только, создавая новый коментарий
                * is_comment == True
                * должны быть проставлены tb_feed_id, tb_item_no, tb_rubric_id
                * у поста-трекбека могут быть теги, у модели комента должны
                  быть проставлены tags
            2. Отредактировать трекбек можно только, редактируя пост
                * is_comment == False

            Редактирование трекбека-поста при редактировании коментария пока
            нам не нужно и, чтобы не усложнять все, не реализовано.
        """
        with dbswitch.root_rw_session():
            if entry.is_comment:
                self._save_comment(entry)
            else:
                self._save_post(entry)
        entry.is_new = False

    def _generate_entry_id(self, entry):
        """
        Сгенерить item_no для поста или comment_id для комментария.
        """
        rows = get_connection().execute(generate_entry_item_no_sql, {
            'feed_id': entry.feed_id,
        })

        return rows.fetchone()[0]

    def _insert_entry_content(self, entry):
        """
        Put data in EntryXmlContent table
        """
        from at.common import wf
        xml = entry.serialize_xml_content()

        if entry.content_type == 'text/wiki':
            wf_version = wf.VERSION
        else:
            wf_version = None

        get_connection().execute(upsert_post_content_sql, {
            'feed_id': entry.feed_id,
            'item_no': entry.item_no,
            'comment_id': entry.comment_id or 0,
            'xml': xml,
            'wf_version': wf_version,
        })

    def _get_trackback_comment_id(self, post):
        """
        Для поста-трекбека вернуть id связанного коммента.
        """
        cursor = get_connection().execute(select_trackback_comment_id_sql, {
            'feed_id': post.feed_id,
            'item_no': post.item_no,
        })
        row = cursor.fetchone()
        if row is None:
            raise exceptions.NotFound(str((post.feed_id, post.item_no)))
        return row

    def _save_post(self, post):
        """
        Put post data into Posts, PostStoreData, EntryXMLContent.
        Also add info about access into PostAccesses.
        """
        if not post.item_no:
            stored_item_no = self._get_post_store_data(post)
            if stored_item_no:
                post.item_no = stored_item_no
            else:
                post.item_no = self._generate_entry_id(post)

        self._insert_post_meta(post)
        self._insert_fake_comment(post)
        self._insert_entry_content(post)
        self._insert_post_tags(post)
        self._correct_feed_pins(post)

        if post.is_new and not post.is_trackback:
            self._insert_post_store_data(post)

        if not post.is_new and post.is_trackback:
            self._update_post_trackback_data(post)

    def _insert_post_meta(self, post):
        """
        Put data in Posts table
        """
        get_connection().execute(upsert_post_meta_sql, {
            'person_id': post.feed_id,
            'author_uid': post.author_id,
            'item_no': post.item_no,
            # TODO: все преобразования желательно перенести в
            # методы/property модели
            'store_time': utils.dt2isoformat(post.store_time),
            'item_time': utils.dt2isoformat(post.item_time),
            'post_type': post.type_id,
            'deleted': post.deleted,
            'store_time_usec': utils.usec(post.store_time),
            'freezed': post.block_comments,
            'rubric_id': post.rubric_id,
            'access_group': post.access_group,
            'score': post.score,
            'store_time_interesting_usec': post.store_time_interesting_usec,
            'store_time_most_interesting_usec': post.store_time_most_interesting_usec,
            'on_moderation': post.on_moderation,
            'pinned_after': post.pinned_after,
        })

    def _insert_fake_comment(self, post):
        """
        Put post in Comments table
        """
        get_connection().execute(upsert_comment_meta_sql, {
            'person_id': post.feed_id,
            'author_uid': post.author_id,
            'item_no': post.item_no,
            'comment_id': 0,
            'parent_id': 0,
            'store_time': utils.dt2timestamp(post.store_time),
            'deleted': post.deleted,
            'comment_type': post.type_id,
            'children_count': post.children_count,
        })

    @staticmethod
    def get_form_id_hash(form_id):
        return hashlib.md5(form_id.encode('utf-8')).hexdigest()[:16]

    def _get_post_store_data(self, post):
        if not post.form_id:
            return
        result = get_connection().execute(select_post_store_data_sql, {
            'feed_id': post.feed_id,
            'link_md5': self.get_form_id_hash(post.form_id)
        })
        row = result.fetchone()
        return row and row[0]

    def _insert_post_store_data(self, post):
        """
        Put data in PostStoreData table
        """
        if not post.form_id:
            return
        get_connection().execute(upsert_post_store_data_sql, {
            'feed_id': post.feed_id,
            'item_no': post.item_no,
            'link_md5': self.get_form_id_hash(post.form_id),
        })

    def _insert_post_tags(self, post):
        """
        Put data in PostCategories table
        """
        if not post.is_new:
            get_connection().execute(delete_post_tags_sql, {
                'feed_id': post.feed_id,
                'item_no': post.item_no,
            })

        if not post.tags:
            return

        new_tags = [tag for tag in post.tags if tag.is_unresolved]
        if new_tags:
            create_new_tags_sql = create_new_tags_sql_tpl % ', '.join([
                "(%s, '%s')" % (post.feed_id, tag.title)
                for tag in new_tags
            ])
            get_connection().execute(create_new_tags_sql)

        # резолвим только что созданные теги
        # нужно это делать в той же транзакции
        post.resolve_tags()

        insert_post_tags_sql = insert_post_tags_sql_tpl % ', '.join([
            '(%s, %s, %s, %s)' % (post.feed_id, post.item_no, tag.id, post.deleted)
            for tag in post.tags
        ])
        get_connection().execute(insert_post_tags_sql)

    @staticmethod
    def resolve_tags(feed_id, titles):
        tag_string = ', '.join('"' + name + '"' for name in titles)
        sql = resolve_feed_tags_sql_tpl % tag_string

        result = get_connection().execute(sql, {
            'feed_id': feed_id,
        })
        return dict((title, id) for id, title in result.fetchall())

    def _update_post_trackback_data(self, post):
        """
        Обновить EntryXmlContent для комента, соответсвующего посту
        """
        tb_comment_id = self._get_trackback_comment_id(post)
        tb_comment = post.clone()
        tb_comment.feed_id, tb_comment.item_no, tb_comment.comment_id = tb_comment_id
        self._insert_entry_content(tb_comment)

    def _correct_feed_pins(self, post):
        pinned_after_old = getattr(post, 'pinned_after_old', None)
        if pinned_after_old == post.pinned_after:
            return

        if pinned_after_old is not None:
            # если был закрпеплен, но открепился или переехал —
            # нужно найти указатель на этот пост и исправить
            get_connection().execute(update_post_pin_after, {
                'feed_id': post.feed_id,
                'old_pinned_after': post.item_no,
                'new_pinned_after': pinned_after_old,
                'exclude_item_no': post.item_no,
            })
        if post.pinned_after is not None:
            # если закрепился за каким-то постом —
            # нужно исправить возможый указатель на него
            get_connection().execute(update_post_pin_after, {
                'feed_id': post.feed_id,
                'old_pinned_after': post.pinned_after,
                'new_pinned_after': post.item_no,
                'exclude_item_no': post.item_no,
            })

    def _save_comment(self, comment):
        """
        Put comment data into Comments, CommentsFullParents and EntryXMLContent
        """
        if not comment.comment_id:
            comment.comment_id = self._generate_entry_id(comment)

        self._insert_comment_meta(comment)
        self._insert_entry_content(comment)

        if comment.is_new:
            self._insert_comment_full_parents(comment)
            self._update_parents_children_count(comment)
            self._update_post_comment_fields(comment)

        if comment.do_trackback:
            # Редактирование трекбека происходит всегда через пост,
            # а создание через комент.
            self._insert_comment_trackback_data(comment)

    def _insert_comment_meta(self, comment):
        get_connection().execute(upsert_comment_meta_sql, {
            'person_id': comment.feed_id,
            'author_uid': comment.author_id,
            'item_no': comment.item_no,
            'comment_id': comment.comment_id,
            'parent_id': comment.parent_comment_id,
            'store_time': utils.dt2timestamp(comment.store_time),
            'deleted': comment.deleted,
            'children_count': comment.children_count,
            'comment_type': Types.ItemTypes().by_name(comment.type)['id'],
        })

    def _insert_comment_full_parents(self, comment):
        get_connection().execute(set_comment_parents_sql, {
            'person_id': comment.feed_id,
            'item_no': comment.item_no,
            'parent_id': comment.parent_comment_id,
            'comment_id': comment.comment_id,
        })

    def _update_parents_children_count(self, comment):
        update_parents_comments_count_sql = update_parents_comments_count_sql_tpl % (
            '-' if comment.deleted else '+'
        )
        get_connection().execute(update_parents_comments_count_sql, {
            'feed_id': comment.feed_id,
            'item_no': comment.item_no,
            'comment_id': comment.comment_id,
        })

    def _update_post_comment_fields(self, comment):
        last_comment_data = self._get_post_last_comment_data(feed_id=comment.feed_id, item_no=comment.item_no)

        if last_comment_data:
            last_comment_id, last_author_uid, last_updated = last_comment_data
        else:
            last_comment_id, last_author_uid, last_updated = None, None, None

        comments_count = self._get_post_comments_count(feed_id=comment.feed_id, item_no=comment.item_no)

        get_connection().execute(update_post_comments_meta_sql, {
            'feed_id': comment.feed_id,
            'item_no': comment.item_no,
            'last_comment_id': last_comment_id,
            'last_author_uid': last_author_uid,
            'last_updated': last_updated,
            'comments_count': comments_count,
        })

    def _get_post_last_comment_data(self, feed_id, item_no):
        cursor = get_connection().execute(select_post_last_comment_sql, {
            'feed_id': feed_id,
            'item_no': item_no,
        })
        row = cursor.fetchone()
        if row:
            last_comment_id, last_author_uid, last_updated = row
            return last_comment_id, last_author_uid, last_updated
        else:
            return None

    def _get_post_comments_count(self, feed_id, item_no):
        cursor = get_connection().execute(count_post_comments_sql, {
            'feed_id': feed_id,
            'item_no': item_no,
        })
        row = cursor.fetchone()
        count, = row
        return count

    def _insert_comment_trackback_data(self, comment):
        """
        Создание поста-трекбека на основе комментария.
        """
        tb_post = comment.clone()
        tb_post.feed_id = comment.tb_feed_id
        tb_post.item_no = comment.tb_item_no
        tb_post.access_type = comment.tb_access_type
        if tb_post.access_type == AccessLevel.MODERATORS:
            tb_post.on_moderation = True
        tb_post.rubric_id = comment.tb_rubric_id
        tb_post.reply_to_feed_id = comment.feed_id
        tb_post.reply_to_item_no = comment.item_no
        tb_post.reply_to_comment_id = comment.comment_id
        tb_post.comment_id = 0

        self._save_post(tb_post)

        get_connection().execute(insert_trackback_sql, {
            'feed_id': comment.feed_id,
            'item_no': comment.item_no,
            'comment_id': comment.comment_id,
            'trackback_feed_id': tb_post.feed_id,
            'trackback_item_no': tb_post.item_no,
        })

        comment.tb_item_no = tb_post.item_no

    def delete(self, entry, **kwargs):
        super(MysqlRepository, self).delete(entry, **kwargs)
        with dbswitch.root_rw_session():
            if entry.is_comment:
                self._delete_comment(entry)
            else:
                self._delete_post(entry)

    def _delete_post(self, post):
        get_connection().execute(delete_post_sql, {
            'feed_id': post.feed_id,
            'item_no': post.item_no,
            'deleted': post.deleted,
        })
        get_connection().execute(delete_comment_sql, {
            'feed_id': post.feed_id,
            'item_no': post.item_no,
            'comment_id': 0,
        })

    def _delete_comment(self, comment):
        get_connection().execute(delete_comment_sql, {
            'feed_id': comment.feed_id,
            'item_no': comment.item_no,
            'comment_id': comment.comment_id,
        })
        self._update_parents_children_count(comment)
        self._update_post_comment_fields(comment)

    def load(self, feed_id, item_no, comment_id=0):
        if not comment_id:
            entry = self.load_post(feed_id=feed_id, item_no=item_no)
        else:
            entry = self.load_comment(feed_id=feed_id, item_no=item_no, comment_id=comment_id)
        return entry

    def load_post(self, feed_id, item_no):
        loaded_posts = self.load_posts_by_ids(id_list=[(feed_id, item_no)])
        try:
            return loaded_posts[0]
        except IndexError:
            raise exceptions.NotFound(str((feed_id, item_no, 0)))

    def load_posts_by_condition(self, condition):
        sql = select_posts_by_condition_sql_tpl % condition

        results = get_connection().execute(sql)

        raw_posts = {}
        columns = list(results.keys())
        for row in results.fetchall():
            row_dict = dict((key, val) for key, val in zip(columns, row))
            raw_posts[(row_dict['person_id'], row_dict['post_no'])] = row_dict

        id_list = list(raw_posts)

        posts_tags = self._load_posts_tags(list(id_list))

        loaded_posts = []
        for post_id, data in list(raw_posts.items()):
            model = self._build_post_model(data)
            model.tags = posts_tags.get((model.feed_id, model.item_no), [])
            loaded_posts.append(model)
        return loaded_posts

    def load_posts_by_ids(self, id_list):
        if not id_list:
            return []

        id_list = list(map(tuple, id_list))

        id_condition = '(' + ' OR '.join([
            'post.person_id = %s AND post.post_no = %s' % post_id
            for post_id in id_list
        ]) + ')'
        return self.load_posts_by_condition(condition=id_condition)

    def _build_post_model(self, data):
        post_type_str = Types.ItemTypes().by_id(data['post_type'])['name']
        feed_id = data['person_id']
        item_no = data['post_no']
        model = self.get_entry_cls(post_type_str)(
            feed_id=feed_id,
            item_no=item_no,
            is_new=False,
        )
        model.author_id = data['author_uid']
        model.store_time = data['store_time']
        model.item_time = data['item_time']
        model.children_count = data['children_count']
        model.block_comments = bool(data['freezed'])
        model.store_time_microseconds = data['store_time_usec']
        model.deleted = False
        model.rubric_id = data['rubric_id']
        model.last_author_uid = data['last_author_uid']
        model.last_comment_id = data['last_comment_id']
        model.access_group = data['access_group']
        model.reply_to_feed_id = data['reply_to_feed_id']
        model.reply_to_item_no = data['reply_to_item_no']
        model.reply_to_comment_id = data['reply_to_comment_id']
        model.score = int(data['score'])
        model.store_time_interesting_usec = data['store_time_interesting_usec']
        model.store_time_most_interesting_usec = data['store_time_most_interesting_usec']
        model.on_moderation = bool(data['on_moderation'])
        model.pinned_after = data['pinned_after']

        try:
            xml_data = serializers.deserialize_xml_content(
                entry_cls=model.__class__, source=data['xml'])
        except Exception:
            log.error(
                'Failed to deserialize post (%s, %s)',
                feed_id, item_no
            )
            raise
        model.update_fields(xml_data)
        return model

    def _load_posts_tags(self, id_list):
        if not id_list:
            return {}

        sql_tpl = """
            SELECT feed_id, post_no, p.id, title_tag
                FROM PostCategories pc
                JOIN PostCategory p
                ON pc.cat_id = p.id
                AND pc.`feed_id` = p.`person_id`
            WHERE deleted = 0 AND
            (%s)
        """
        id_condition = ' OR '.join([
            'feed_id = %s AND post_no = %s' % post_id
            for post_id in id_list
        ])
        sql = sql_tpl % id_condition

        cursor = get_connection().execute(sql)

        loaded_tags = {}
        for feed_id, item_no, tag_id, title in cursor.fetchall():
            post_tags = loaded_tags.setdefault((feed_id, item_no), [])
            post_tags.append((tag_id, title))
        return loaded_tags


    def load_comment(self, feed_id, item_no, comment_id):
        """
        Вернуть комментарий.

        В результате "parent_comment_id" будет всегда равен 0.

        @rtype: entries.models.Post
        """
        loaded_comments = self.load_comments_by_ids(id_list=[(feed_id, item_no, comment_id)])
        try:
            return next(loaded_comments)
        except (IndexError, StopIteration):
            raise exceptions.NotFound(str((feed_id, item_no, comment_id)))

    def _build_comment_model(self, row):
        """
        Преобразовать результат работы SQL из load_comments_by_parent в модель.

        @param row: tuple-like row
        @rtype: Post
        """
        author_id, feed_id, item_no, children_count, deleted, store_time, \
        comment_type, content, comment_id, parent_id, tb_feed_id, tb_item_no = tuple(row)
        comment_type_str = Types.ItemTypes().by_id(comment_type)['name']
        model = self.get_entry_cls(comment_type_str)(
            feed_id=feed_id,
            item_no=item_no,
            comment_id=comment_id,
            is_new=False,
        )
        model.children_comments = []
        model.store_time = store_time
        model.author_id = author_id
        model.deleted = bool(deleted)
        model.children_count = children_count
        model.parent_comment_id = parent_id
        model.tb_feed_id = tb_feed_id
        model.tb_item_no = tb_item_no

        try:
            content_data = serializers.deserialize_xml_content(
                entry_cls=model.__class__, source=content)
        except Exception:
            log.error(
                'Failed to deserialize comment (%s, %s, %s)',
                feed_id, item_no, comment_id
            )
            raise

        model.update_fields(content_data)
        return model

    def load_comments(self, condition, join_tables='', with_deleted=False):
        sql = """
            SELECT
                c.author_uid, c.person_id, c.post_no, c.children_count, c.deleted,
                c.store_time, c.comment_type, content.xml, c.comment_id, c.parent_id,
                tb.trackback_person_id, tb.trackback_post_no
            FROM Comments c """ + join_tables + \
            """ JOIN EntryXmlContent content ON (
                    c.comment_id = content.comment_id AND
                    c.person_id = content.feed_id AND
                    c.post_no = content.post_no
                )
                LEFT JOIN Trackbacks tb ON (
                    tb.person_id = c.person_id AND
                    tb.post_no = c.post_no AND
                    tb.comment_id = c.comment_id
                )
            WHERE c.comment_id != 0 AND """ + \
            ('' if with_deleted else '(c.deleted = 0 or c.children_count > 0) AND ') + \
            condition
        for row in get_connection().execute(sql):
            yield self._build_comment_model(row)


    def load_comments_by_ids(self, id_list, with_deleted=False):
        condition = ' OR '.join([
            'c.person_id = %s AND c.post_no = %s AND c.comment_id = %s' %
            entry_id for entry_id in id_list
        ])
        return self.load_comments('(%s)' % condition, with_deleted=with_deleted)

    @staticmethod
    def get_trackback_children_count(mapping):
        # подстановка children_count из трекбэка нужна только при отображении дерева комментариев,
        # поэтому не встраиваю её в основной код загрузки комментов.
        # mapping содержит соответствие (tb_feed_id, tb_item_no) номеру коммента в исходном посте
        # (поскольку этот метод используется только при отображении дерева комментариев к _одному посту_,
        # то номер коммента является достаточным идентификатором).
        if not mapping:
            return []
        sql = 'select person_id, post_no, children_count from Posts where (person_id, post_no) in (%s)'
        pairs = ','.join('(%d,%d)' % tb_coords for tb_coords in mapping.keys())
        return [ (mapping[(row[0], row[1])], row[2]) for row in get_connection().execute(sql % pairs) ]
