# coding: utf-8

import functools

from lxml import etree as ET

from at.common import utils
from . import fields as fields_helpers


def serialized(function):
    @functools.wraps(function)
    def wrapper(*args, **kwargs):
        as_object = kwargs.pop('as_object', False)
        pretty = kwargs.pop('pretty', False)

        xml_object = function(*args, **kwargs)

        if as_object:
            return xml_object

        return ET.tostring(xml_object, encoding="utf-8", pretty_print=pretty)
    return wrapper


def source_as_str_or_obj(function):
    @functools.wraps(function)
    def wrapper(self, source):
        if isinstance(source, str):
            source = ET.fromstring(source)
        return function(self, source)
    return wrapper


class Deserializer(object):
    def __init__(self, entry_cls):
        self.entry_cls = entry_cls

    def parse_content_type(self, source):
        """Parse content-type"""
        raise NotImplementedError


class Meta(object):
    common_fields = [
        ('AccessType', 'access_type'),
        ('StoreTime', 'store_time'),
        ('PostType', 'type_id'),
        ('id/uid', 'feed_id'),
        ('id/item_no', 'item_no'),
        ('id/asString', 'as_string'),
        ('UpdateTime', 'item_time'),
        ('CommentCount', 'children_count'),
        ('on_moderation', 'on_moderation'),
        ('pinned_after', 'pinned_after'),
    ]

    common_reply_fields = [
        ('CommentCount', 'children_count'),
        ('StoreTime', 'store_time'),
        ('deleted', 'deleted'),
        ('id/uid', 'feed_id'),
        ('id/item_no', 'item_no'),
        ('id/comment_id', 'comment_id'),
        ('id/asString', 'as_string'),
        ('id/asStringFull', 'as_string_full'),
        ('id/trackback-id/uid', 'tb_feed_id'),
        ('id/trackback-id/item_no', 'tb_item_no'),
        ('id/trackback-id/asString', 'tb_as_string'),
    ]

    item_fields = [
        ('author/uid', 'author_id'),
        ('block-comments', 'block_comments'),
        # верстка требует поля и для комента-трекбека и для поста-трекбека
        ('trackback/in-reply-to-id/uid', 'tb_in_reply_feed_id'),
        ('trackback/in-reply-to-id/item_no', 'tb_in_reply_item_no'),
        ('trackback/in-reply-to-id/comment_id', 'tb_in_reply_comment_id'),
        ('trackback/in-reply-to-id/asString', 'tb_in_reply_as_string'),
    ]

    parent_fields = [
        ('comment_id', 'parent_comment_id'),
        ('item_no', 'item_no'),
        ('uid', 'feed_id'),
    ]

    entry_fields = [
        ('cc_addresses', 'cc_address'),
        # edit_time нужен в xml для фронтенда, но не нужен в EntryXmlContent.xml
        # но из-за неразличимости этих сериализаций, попадает в оба. В модели
        # поля edit_time нет, так что из EntryXmlContent.xml edit_time не
        # используется.
        ('edit_time', 'item_time'),
        ('meta_type', 'entry_type'),
        ('type', 'entry_type'),
        ('filter_type', 'entry_type'),
        ('moderated-by', 'moderated_by'),
        ('published', 'published')
    ]

    content_fields = [
        ('body-original', 'body_original'),
        ('snippet', '_snippet'),
        ('title', '_title'),
        ('original-access-type', 'original_access_type')
    ]

    trackback_fields = [
        ('item_no', 'tb_item_no'),
        ('uid', 'tb_feed_id'),
    ]

    in_reply_to_fields = [
        ('comment_id', 'reply_to_comment_id'),
        ('item_no', 'reply_to_item_no'),
        ('uid', 'reply_to_feed_id'),
    ]

    meta_fields = {

    }

    # Поля, которые будут добавлены в xml пустыми,
    # если в соответствующем поле модели None
    xml_empty_fields = [
        'title',
        'body-original'
    ]


class XMLDeserializer(Deserializer):
    meta = Meta

    def parse_entry(self, source):
        return self._parse_section_fields(
            source=source,
            suffix='',
            fields=self.meta.entry_fields,
        )

    def parse_content(self, source):
        content_fields = self._parse_section_fields(
            source=source,
            suffix='content/',
            fields=self.meta.content_fields
        )
        parsed_body = self.parse_body(source)
        if parsed_body is not None:
            content_fields['_body'] = parsed_body
        return content_fields

    def parse_body(self, source):
        body_node = source.find('content/body')
        if body_node is None:
            return

        # у body могут быть атрибуты, например type
        attrs = list(body_node.attrib)
        for attr in attrs:
            del body_node.attrib[attr]

        body_node_as_str = ET.tounicode(body_node)
        # топорный, но, кажется, самый простой и надежный способ превратить
        # внутренности тега body в строку, ничего не потеряв.
        # Пустой тег <body/> превратится в пустую строку, как можно догадаться.
        body_node_as_str = body_node_as_str.strip()
        return body_node_as_str[len('<body>'):-len('</body>')]

    def parse_content_type(self, source):
        result = {}
        try:
            result["_content_type"] = utils.force_unicode(
                source.find('content/body-original').attrib['type']
            )
        except (AttributeError, KeyError):
            result["_content_type"] = "text/plain"
        return result

    def parse_meta(self, source):
        return self._parse_section_fields(
            source=source,
            suffix='meta/',
            fields=self.meta.meta_fields,
        )

    def parse_common(self, source):
        return self._parse_section_fields(
            source=source,
            suffix='',
            fields=self.meta.common_fields
        )

    def parse_item(self, source):
        return self._parse_section_fields(
            source=source,
            suffix='item/',
            fields=self.meta.item_fields
        )

    def _parse_section_fields(self, source, suffix, fields):
        result = {}

        for xml_name, attr_name in fields:
            xpath = '%s%s' % (suffix, xml_name)
            node = source.find(xpath)
            field_object = self.entry_cls.fields.get(attr_name)
            if field_object is None:
                continue
            if node is not None:
                prepared_value = fields_helpers.from_literal(field_object, value=node.text)
                result[attr_name] = prepared_value
        return result

    def parse_tags(self, source):
        tags = source.find('item/tag-list') or []
        return {"tags": [int(tag_node.attrib['id']) for tag_node in tags]}

    def parse_parent(self, source):
        return self._parse_section_fields(
            source=source,
            suffix='item/parent/',
            fields=self.meta.parent_fields,
        )

    def parse_in_reply_to(self, source):
        return self._parse_section_fields(
            source=source,
            suffix='item/trackback/in-reply-to-id/',
            fields=self.meta.in_reply_to_fields,
        )

    def parse_trackback(self, source):
        return self._parse_section_fields(
            source=source,
            suffix='id/trackback-id/',
            fields=self.meta.trackback_fields,
        )

    @source_as_str_or_obj
    def deserialize_xml_content(self, source):
        """
        source — нода entry
        """
        result = {}
        result.update(self.parse_entry(source))
        result.update(self.parse_meta(source))
        result.update(self.parse_content(source))
        result.update(self.parse_content_type(source))
        result.update(self.add_extra(source))
        return result

    def add_extra(self, source):
        """
        Для переопределения в наследниках
        """
        return {}


class Serializer(object):
    def __init__(self, object):
        self.object = object

    def serialize(self):
        """Serialize object"""
        raise NotImplementedError


class XMLSerializer(Serializer):

    meta = Meta

    @serialized
    def serialize_xml_content(self, escape_html=False):
        return self.build_entry_node(escape_html=escape_html)

    @serialized
    def serialize_item(self, escape_html=False):
        return self.build_item_node(escape_html=escape_html)

    @serialized
    def serialize_reply(self, escape_html=False):
        return self.build_reply_node(escape_html=escape_html)

    @serialized
    def serialize(self, escape_html=False):
        return self.build_feed_node(escape_html=escape_html)

    @serialized
    def serialize_for_preview(self, escape_html=False):
        return self.build_aux_node(escape_html=escape_html)

    def build_entry_meta_node(self):
        meta = ET.Element('meta')
        self.serialize_section(meta, self.meta.meta_fields)
        return meta

    def build_entry_node(self, escape_html=False):
        """
        Для переопределения в дочерних классах.
        """
        entry = ET.Element('entry')
        self.serialize_section(entry, self.meta.entry_fields)

        if self.meta.meta_fields:
            meta = self.build_entry_meta_node()
            entry.append(meta)

        content = ET.Element('content')
        self.serialize_section(content, self.meta.content_fields)

        body = self.serialize_body(escape_html=escape_html)
        content.append(body)

        self.build_content_type_node(content)
        entry.append(content)
        return entry

    def build_feed_node(self, escape_html=False):
        feed = ET.Element('feed')
        outer_item = self.build_item_node(escape_html=escape_html)
        feed.append(outer_item)
        return feed

    def build_parent_node(self, name='parent'):
        parent = ET.Element(name)
        section = [
            ('uid', 'feed_id'),
            ('item_no', 'item_no'),
            ('asString', 'as_string')
        ]
        if getattr(self.object, 'parent_comment_id', None):
            section.extend([
                ('comment_id', 'parent_comment_id'),
                ('asStringFull', 'as_string_full_parent')
            ])
        self.serialize_section(parent, section)
        return parent

    def build_reply_node(self, escape_html=False):
        root = ET.Element('reply')
        inner_item = ET.Element('item')
        self.serialize_section(inner_item, self.meta.item_fields)
        inner_item.append(self.build_parent_node())
        inner_item.append(self.build_entry_node(escape_html=escape_html))
        outer_item = ET.SubElement(root, 'item')
        self.serialize_section(outer_item, self.meta.common_reply_fields)
        outer_item.append(inner_item)
        if hasattr(self.object, 'children_comments'):
            for child in self.object.children_comments:
                root.append(child.serializer.build_reply_node(escape_html=escape_html))
        return root

    def build_item_node(self, escape_html=False):
        entry = self.build_entry_node(escape_html=escape_html)
        inner_item = ET.Element('item')
        self.serialize_section(inner_item, self.meta.item_fields)

        tag_list = ET.Element('tag-list')
        self.build_tag_list_node(tag_list)
        inner_item.append(tag_list)
        inner_item.append(entry)

        outer_item = ET.Element('item')
        self.serialize_section(outer_item, self.meta.common_fields)
        outer_item.append(inner_item)

        return outer_item

    def build_aux_node(self, escape_html=False):
        """
        Для переопределения в дочерних классах.
        """
        feed = self.build_feed_node(escape_html=escape_html)

        # это нужно сделать в превью
        posts_tags = ET.Element('posts-tags')
        self.build_posts_tags_node(posts_tags)

        aux = ET.Element('aux')
        aux.append(feed)
        aux.append(posts_tags)

        return aux

    def add_node(self, root, name, value):
        parts = name.split('/')
        for part in parts[:-1]:
            if root.find(part) is not None:
                root = root.find(part)
                continue
            else:
                root = ET.SubElement(root, part)
        name = parts[-1]

        el = ET.SubElement(root, name, {})
        el.text = utils.force_unicode(value)

    def serialize_section(self, root, fields):
        for xml_name, attr_name in fields:
            # можно переопределить сериализацию какого-то атрибута
            # или просто запилить метод _serialize_что_угодно и упомянуть в
            # meta ключ что_угодно
            method_name = '_serialize_' + attr_name
            if hasattr(self, method_name):
                value = getattr(self, method_name)()
            else:
                value = getattr(self.object, attr_name, None)

            insert_empty = xml_name in self.meta.xml_empty_fields
            if value is None and not insert_empty:
                continue

            field_object = self.object.fields.get(attr_name)
            if field_object is not None:
                xml_value = fields_helpers.to_literal(field_object, value)
            else:
                xml_value = value

            if xml_value is not None:
                self.add_node(
                    root=root,
                    name=xml_name,
                    value=xml_value,
                )

    def _serialize_as_string(self):
        return '%s.%s' % (
            self.object.feed_id,
            self.object.item_no
        )

    def _serialize_as_string_full(self):
        return '%s.%s.%s' % (
            self.object.feed_id,
            self.object.item_no,
            self.object.comment_id
        )

    def _serialize_tb_as_string(self):
        if self.object.has_trackback:
            return '%s.%s' % (
                self.object.tb_feed_id,
                self.object.tb_item_no,
            )

    def _serialize_tb_in_reply_feed_id(self):
        if self.object.is_trackback:
            return self.object.reply_to_feed_id
        elif self.object.has_trackback:
            return self.object.feed_id

    def _serialize_tb_in_reply_item_no(self):
        if self.object.is_trackback:
            return self.object.reply_to_item_no
        elif self.object.has_trackback:
            return self.object.item_no

    def _serialize_tb_in_reply_comment_id(self):
        if self.object.is_trackback:
            in_reply_to = self.object.in_reply_to
            return in_reply_to and in_reply_to.parent_comment_id or None
        elif self.object.has_trackback:
            return self.object.comment_id

    def _serialize_tb_in_reply_as_string(self):
        if self.object.is_trackback or self.object.has_trackback:
            return '%s.%s' % (
                self._serialize_tb_in_reply_feed_id(),
                self._serialize_tb_in_reply_item_no(),
            )

    def _serialize_as_string_full_parent(self):
        return '%s.%s.%s' % (
            self.object.feed_id,
            self.object.item_no,
            self.object.parent_comment_id or 0
        )

    def build_content_type_node(self, node):
        # As bad code as our content_type storing way
        for content_node in node.getchildren():
            if content_node.tag.endswith('-original'):
                content_node.attrib['type'] = self.object.content_type

    def serialize_body(self, escape_html=False):
        body_open = '<body>'
        body_close = '</body>'
        if escape_html:
            body_open += '<![CDATA['
            body_close = ']]>' + body_close
        body_node_str = body_open + self.object._body + body_close
        xml_body = ET.fromstring(body_node_str)

        if escape_html:
            xml_body.text = ET.CDATA(xml_body.text)
        return xml_body

    def build_posts_tags_node(self, node):
        for tag in self.object.tags:
            tag_node = ET.Element('post-tag')
            tag_node.attrib['feed-item'] = '%s.%s' % (
                self.object.author_id,
                self.object.item_no
            )

            id = ET.Element('id')
            if tag.id:
                id.text = str(tag.id)
            else:
                # тег может не существовать в базе в режиме превью,
                # для того, чтобы фронтенд его отрисовал ставим фейковый id
                id.text = '0'

            title = ET.Element('title-tag')
            title.text = tag.title

            tag_node.append(id)
            tag_node.append(title)

            node.append(tag_node)

    def build_tag_list_node(self, node):
        for tag in self.object.tags:
            tag_node = ET.Element('tag')
            tag_node.attrib['id'] = str(tag.id)
            node.append(tag_node)


class LinkMeta(Meta):
    meta_fields = [
        ('URL', 'url'),
    ]

    shared_post_fields = [
        ('comment_id', 'shared_comment_id'),
        ('feed_id', 'shared_feed_id'),
        ('item_no', 'shared_item_no'),
    ]


class LinkDeserializer(XMLDeserializer):
    meta = LinkMeta

    def parse_shared_post(self, source):
        # Separate method for shared post parsing because we store
        # shared_post not as everything else
        node = source.find('meta/shared_post_id')
        if node is not None:
            result = {}
            for xml_name, attr_name in self.meta.shared_post_fields:
                field_object = self.entry_cls.fields[attr_name]
                attr_value = node.attrib.get(xml_name)
                if attr_value:
                    result[attr_name] = fields_helpers.from_literal(
                    field_object, node.attrib.get(xml_name) or '')
            return result

    def add_extra(self, source):
        result = {}
        shared_post_info = self.parse_shared_post(source)
        if shared_post_info:
            result.update(shared_post_info)
        return result


class LinkSerializer(XMLSerializer):
    meta = LinkMeta

    def build_entry_meta_node(self):
        meta = super(LinkSerializer, self).build_entry_meta_node()

        shared_post_data = ET.Element("shared_post_id")
        attrs = {}
        for xml_name, attr_name in self.meta.shared_post_fields:
            value = getattr(self.object, attr_name, None)
            if value:
                attrs[xml_name] = str(value)
        if attrs:
            shared_post_data.attrib.update(attrs)
            meta.append(shared_post_data)

        return meta


class StatusSerializer(XMLSerializer):

    def build_entry_node(self, escape_html=False):
        entry = super(StatusSerializer, self).build_entry_node(escape_html=escape_html)
        content_body = entry.find("content/body")
        content_body.attrib['type'] = 'text/html'
        return entry


class PollMeta(Meta):
    meta_fields = [
        ('expires', 'expires'),
        ('hidden', 'hidden'),
        ('poll_id', 'poll_id'),
        ('poll_type', 'poll_type'),
    ]


class PollDeserializer(XMLDeserializer):
    meta = PollMeta

    def parse_poll_options(self, source):
        options = source.findall('meta/options/option')
        return {'options': [element.text for element in options]}

    def add_extra(self, source):
        result = {}
        poll_options = self.parse_poll_options(source)
        if poll_options:
            result.update(poll_options)
        return result


class PollSerializer(XMLSerializer):
    meta = PollMeta

    def build_entry_meta_node(self):
        meta = super(PollSerializer, self).build_entry_meta_node()
        options = ET.Element("options")
        meta.append(options)
        for option in self.object.options:
            self.add_node(options, "option", option)
        return meta


class SummonMeta(Meta):
    meta_fields = [
        ('address', 'summon_address'),
        ('uid', 'summon_uid'),
    ]


class SummonSerializer(XMLSerializer):
    meta = SummonMeta


class SummonDeserializer(XMLDeserializer):
    meta = SummonMeta


class CongratulationMeta(Meta):
    content_fields = Meta.content_fields[:]
    content_fields.extend(
        [
            ('whom/uid', 'whom'),
            ('event', 'event'),
        ]
    )


class CongratulationSerializer(XMLSerializer):
    meta = CongratulationMeta


class CongratulationDeserializer(XMLDeserializer):
    meta = CongratulationMeta


class FriendMeta(Meta):
    content_fields = Meta.content_fields[:]
    content_fields.extend([('friendee/uid', 'friendee'),
                          ('friender/uid', 'friender')])


class FriendSerializer(XMLSerializer):
    meta = FriendMeta
UnfriendSerializer = FriendSerializer


class FriendDeserializer(XMLDeserializer):
    meta = FriendMeta
UnfriendDeserializer = FriendDeserializer


class JoinMeta(Meta):
    content_fields = Meta.content_fields[:]
    content_fields.extend(
        [
            ('friendee/uid', 'club'),
            ('friender/uid', 'person'),
        ]
    )


class JoinSerializer(XMLSerializer):
    meta = JoinMeta
UnjoinSerializer = JoinSerializer


class JoinDeserializer(XMLDeserializer):
    meta = JoinMeta
UnjoinDeserializer = JoinDeserializer


class TicketMeta(Meta):
    entry_fields = Meta.entry_fields[:]
    entry_fields.extend(
        [
            ('issue', 'issue'),
        ])


class TicketSerializer(XMLSerializer):
    meta = TicketMeta


class TicketDeserializer(XMLDeserializer):
    meta = TicketMeta


def get_serializer(model):
    cls_name = '%s%s' % (model.__class__.__name__, 'Serializer')
    cls = globals().get(cls_name, XMLSerializer)
    return cls(object=model)


def get_deserializer(entry_cls):
    cls_name = entry_cls.__name__ + 'Deserializer'
    cls = globals().get(cls_name, XMLDeserializer)
    return cls(entry_cls=entry_cls)
