# coding: utf-8


import logging
import time
from datetime import datetime, timedelta

from django.core.exceptions import PermissionDenied
from lxml import etree as ET
from dateutil.relativedelta import relativedelta

from at.common import utils
from at.aux_ import Accesses
from at.aux_.entries import models
from at.aux_.feeds.loaders import friends
from at.aux_.feeds.loaders import popular
from at.aux_.feeds.loaders import feed
from at.aux_.feeds.loaders import calendar
from at.aux_.feeds.loaders import comments
from at.common.groups import GroupType

logger = logging.getLogger(__name__)


class Feeds(object):
    TAG_NAME = 'posts'

    @utils.et2xml
    @utils.log_exception
    @utils.stopwatch
    def GetFeedCalendar(self, ai, feed_id, post_types, cat_id):
        loader = calendar.CalendarLoader(
            viewer_id=ai.uid,
            feed_id=feed_id,
            post_types=post_types,
            tag_id=cat_id,
        )

        root = ET.Element("calendar")
        current_year, current_year_node = None, None

        def datetime_to_str_timestamp(dt):
            return str(int(time.mktime(dt.timetuple())))

        for month, posts_count in loader.posts_data:
            month_num = month.month
            month_year = month.year

            if month_year != current_year:
                current_year_node = ET.SubElement(
                    root, "year", {"number": str(month_year)}
                )
                current_year = month_year
            month_node = ET.SubElement(current_year_node, "month")

            month_begin = datetime(month_year, month_num, 1, 0, 0, 0)
            month_end = month + relativedelta(months=1) - relativedelta(seconds=1)

            ET.SubElement(month_node, "month-begin").text = datetime_to_str_timestamp(month_begin)
            ET.SubElement(month_node, "month-end").text = datetime_to_str_timestamp(month_end)
            ET.SubElement(month_node, "posts-count").text = str(posts_count)
        return ET.ElementTree(root)


    @utils.log_exception
    @utils.et2xml
    @utils.stopwatch
    def GetPopularPosts(self, ai, tb, count, category, metarubric, escape_html=False):
        loader_cls = {
            'all': popular.AllPostsLoader,
            'interesting': popular.InterestingPostsLoader,
            'top': popular.MostInterestingPostsLoader,
        }.get(category, popular.AllPostsLoader)

        params = {
            'viewer_id': ai.uid,
            'metarubric': metarubric,
        }
        if count:
            params['limit'] = count
        if tb:
            params['tb'] = tb
        loader = loader_cls(**params)
        response = self._wrap_response(loader, escape_html=escape_html)
        self.debug_response(response)
        return response

    @utils.et2xml
    @utils.stopwatch
    @utils.log_exception
    def ReadFriendsLight(self, ai, person_id, tb, count, escape_html=False):
        params = {
            'viewer_id': ai.uid,
            'host_id': person_id,
        }
        if count:
            params['limit'] = count
        if tb:
            params['tb'] = tb
        loader = friends.FriendsPostsLoader(**params)
        response = self._wrap_response(loader, escape_html=escape_html)
        self.debug_response(response)
        return response

    @staticmethod
    def handle_date(dt_str):
        # фронтенд может присылать 0 — интерпретируем как None
        return datetime.fromtimestamp(dt_str) if dt_str else None

    @utils.et2xml
    @utils.log_exception
    def GetFeed2Light(self, ai, source_id, escape_html=False, post_types_str='', cat_id=None,
                      count=None, tb=None, min_store_time=None, max_store_time=None):

        count = count or 20
        tb = tb or 0
        if post_types_str or cat_id:
            pins_mode = 'all_posts'
        else:
            pins_mode = 'pins_first'

        if ai is None:
            # странный режим, когда видны только посты с публичным доступом,
            # на нулевой uid просто не матчится ничего из Followers/fgm
            # сделано только для rss на вики
            # https://st.yandex-team.ru/AT-2438#1485793009000
            viewer_id = 0
        else:
            viewer_id = ai.uid
        return self._load_feed(
            viewer_id=viewer_id,
            feed_id=source_id,
            tag_id=cat_id,
            post_types_str=post_types_str,
            tb=tb,
            limit=count,
            min_store_time=self.handle_date(min_store_time),
            max_store_time=self.handle_date(max_store_time),
            order='store_time',
            pins_mode=pins_mode,
            escape_html=escape_html,
        )

    @utils.et2xml
    @utils.log_exception
    def GetFeedOnModeration(self, ai, source_id,
            count=20, tb=0, min_store_time=None, max_store_time=None):

        return self._load_feed(
            viewer_id=ai.uid,
            feed_id=source_id,
            tb=tb,
            limit=count,
            min_store_time=self.handle_date(min_store_time),
            max_store_time=self.handle_date(max_store_time),
            order='store_time',
            on_moderation=True,
        )


    @utils.stopwatch
    @utils.et2xml
    @utils.log_exception
    def GetFeedTrends(self, ai, source_id, post_types_str='', count=None, tb=None):
        count = count or 5
        tb = tb or 0
        return self._load_feed(
            viewer_id=ai.uid,
            feed_id=source_id,
            post_types_str=post_types_str,
            tb=tb,
            limit=count,
            min_store_time=datetime.now() - timedelta(days=90),
            max_store_time=datetime.now(),
            order='score',
        )

    def _load_feed(self, **params):
        if not params['feed_id']:
            return utils.Status("error", "NOT_FOUND")

        if params.get('tb') == 0:
            del params['tb']

        if params.get('limit') == 0:
            del params['limit']

        viewer_id = params.pop('viewer_id')

        if 'post_types_str' in params:
            post_types_str = params.pop('post_types_str')
            params['post_types'] = [
                type for type in post_types_str.split(',') if type]

        loader = feed.FeedLoader(viewer_id=viewer_id, **params)
        response = self._wrap_response(loader, escape_html=params.get('escape_html'))
        self.debug_response(response)
        return response

    def _wrap_response(self, loader, escape_html=False):
        posts_id_list = [(data[0], data[1]) for data in loader.posts_data]

        posts = models.load_posts_by_ids(posts_id_list)
        posts = sorted(
            posts,
            key=lambda model: posts_id_list.index((
                model.feed_id,
                model.item_no,
            ))
        )
        response = '\n'.join(post.serialize_item(escape_html=escape_html) for post in posts)
        pager = loader.pager
        if pager and (pager.before or pager.after):
            response += '<navigation>'
            if pager.before is not None:
                response += '<before tb="%s"/>' % pager.before
            if pager.after is not None:
                response += '<after tb="%s"/>' % pager.after
            response += '</navigation>'

        response = '<feed>' + response + '</feed>'

        if pager and pager.has_more:
            response += '<has-more/>'

        response = '<aux>' + response + '</aux>'
        return response

    @utils.stopwatch
    @utils.et2xml
    @utils.log_exception
    def GetFeedPins(self, ai, source_id):
        loader = feed.FeedLoader(
            viewer_id=ai.uid,
            feed_id=source_id,
            pins_mode='pins_only',
        )
        posts_id_list = [(data[0], data[1]) for data in loader.posts_data]
        posts = models.load_posts_by_ids(posts_id_list)
        posts = sorted(
            posts,
            key=lambda model: posts_id_list.index((
                model.feed_id,
                model.item_no,
            ))
        )
        root = ET.Element('feed')
        for post in posts:
            item = ET.Element('item')
            item.append(ET.fromstring('<item_no>%s</item_no>' % post.item_no))
            item.append(ET.fromstring('<snippet>%s</snippet>' % post.snippet))
            item.append(ET.fromstring('<item_time>%s</item_time>' % post.item_time))
            root.append(item)
        return ET.tostring(root)

    @utils.stopwatch
    @utils.et2xml
    @utils.log_exception
    def GetCommentsEx(self, ai, feed_id, item_no, parent_id=None, last_seen_id=None,
                      with_parent=None, limit=None, escape_html=False):
        post = models.load_entry(feed_id, item_no)
        if ai:
            access = Accesses.Access(ai.uid, feed_id, entry=post)
            access.assert_can_read_post()
        else:
            if post.access_group != GroupType.USER:
                raise PermissionDenied(f"Post {feed_id}/{item_no} is not opened")

        if parent_id is None:
            parent_id = 0
        if with_parent is None:
            with_parent = True
        if limit is None:
            limit = 100

        comment = models.load_entry(feed_id, item_no, parent_id)
        parents, last_comment_id, extra_count = comments.build_comment_tree(
            comment, last_seen_id, limit)

        srlzr = comment.serializer
        root_node = ET.Element('aux')
        parent_item = ET.SubElement(ET.SubElement(root_node, 'parent'), 'item')
        srlzr.serialize_section(parent_item, srlzr.meta.common_reply_fields)

        replies_node = ET.SubElement(root_node, 'replies')

        args = ET.SubElement(replies_node, 'args')
        ET.SubElement(args, 'from').text = str(last_seen_id)
        ET.SubElement(args, 'limit').text = str(limit)
        # внезапно, используется для построения ссылки "уровень выше"
        ET.SubElement(args, 'with-parent').text = str(int(with_parent))

        args.append(srlzr.build_parent_node('parent-id')) # todo проверить
        if parent_id and not with_parent:
            parents = parents[0].children_comments
        for r in parents:
            replies_node.append(r.serializer.build_reply_node(escape_html=escape_html))
        if last_comment_id:
            navigation = ET.SubElement(replies_node, 'more')
            ET.SubElement(navigation, 'from').text = str(last_comment_id)
            ET.SubElement(navigation, 'extra-count').text = str(extra_count)
        return ET.ElementTree(root_node)

    @utils.stopwatch
    @utils.et2xml
    @utils.log_exception
    def GetSingleComment(self, ai, feed_id, item_no, comment_id, escape_html=False):
        post = models.load_entry(feed_id, item_no)
        access = Accesses.Access(ai.uid, feed_id, entry=post)
        access.assert_can_read_post()
        comment = models.load_entry(feed_id, item_no, comment_id)
        return comment.serialize_reply(escape_html=escape_html)

    def debug_response(self, response):
        preview = response[:100]
        if len(response) > 100:
            preview += '...' + response[-100:]
        logger.debug(preview)
