# coding: utf-8

import logging

from at.common import Types
from at.aux_.feeds import loaders


logger = logging.getLogger(__name__)


class FeedLoader(loaders.PostsLoader):
    """
    Посты для фида
    """

    pager_type = 'numeric'

    defaults = {
        'tag_id': None,
        'post_types': list,
        'limit': None,
        'tb': 0,
        # TODO: универсально принимать даты и юникстайм
        'min_store_time': None,
        'max_store_time': None,
        'order': 'store_time',
        'on_moderation': None,
        'pins_mode': 'all_posts',  # 'pins_first'|'pins_only',
    }

    def __init__(self, viewer_id, feed_id, **params):
        self.feed_id = feed_id
        self._all_pinned_rows = None
        super(FeedLoader, self).__init__(viewer_id, **params)

    def get_query_args(self):
        args = super(FeedLoader, self).get_query_args()
        args['feed_id'] = self.feed_id

        if self.tag_id:
            args['tag_id'] = self.tag_id
        if self.min_store_time:
            args['min_store_time'] = self.min_store_time
        if self.max_store_time:
            args['max_store_time'] = self.max_store_time
        return args

    def get_from_lines(self):
        lines = [
            'FROM Posts',
            self.ACCESS_JOIN,
        ]
        if self.tag_id:
            lines.append("""
                LEFT JOIN `PostCategories` tags ON (
                    Posts.person_id = tags.feed_id AND
                    Posts.post_no = tags.post_no
                )
            """)
        return lines

    def get_where_lines(self):
        lines = [
            'WHERE Posts.person_id=%(feed_id)s',
            self.ACCESS_CONDITION,
        ]

        if self.post_types:
            post_type_ids = []
            for post_type_name in self.post_types:
                id = Types.ItemTypes().by_name(post_type_name)['id']
                if id:
                    post_type_ids.append(id)
            if post_type_ids:
                lines.append(
                    'AND post_type IN (%s)' % ', '.join(map(str, post_type_ids)))

        if self.tag_id:
            lines.append('AND tags.cat_id = %(tag_id)s')

        if self.min_store_time:
            lines.append('AND Posts.store_time >= %(min_store_time)s')

        if self.max_store_time:
            lines.append('AND Posts.store_time <= %(max_store_time)s')

        if self.on_moderation is not None:
            lines.append('AND Posts.on_moderation = %s' % int(self.on_moderation))

        return lines

    def get_offset_lines(self, offset=None):
        if offset is None:
            offset = self.tb
        if offset:
            return ['OFFSET %s' % offset]
        return []

    def get_post_rows(self):
        rows = []
        if self.pins_mode == 'pins_only':
            rows.extend(self.all_pinned_rows)
        elif self.pins_mode == 'pins_first':
            rows.extend(self.pinned_rows)
            rows.extend(self.select_other_rows())
        elif self.pins_mode == 'all_posts':
            rows.extend(self.select_other_rows())
        return rows

    @property
    def pinned_rows(self):
        return self.all_pinned_rows[self.tb:self.tb + self.get_query_limit()]

    @property
    def all_pinned_rows(self):
        if self._all_pinned_rows is None:
            rows = self.select_pinned_rows()
            rows = self.sort_pinned_rows(rows)
            rows = [(row[0], row[1]) for row in rows]
            self._all_pinned_rows = rows
        return self._all_pinned_rows

    def select_pinned_rows(self):
        return self.select_rows(
            query=' '.join(
                self.get_select_lines(
                    fields=list(self.select_fields) + ['pinned_after']
                ) +
                self.get_from_lines() +
                self.get_where_lines() +
                ['AND pinned_after IS NOT NULL']
            ),
            query_args=self.get_query_args(),
        )

    @staticmethod
    def sort_pinned_rows(rows):
        pointers = {
            pinned_after: (feed_id, item_no, pinned_after)
            for (feed_id, item_no, pinned_after) in rows
        }

        sorted_pins = []

        current_item_no = 0
        while True:
            post_after_current = pointers.get(current_item_no)
            if post_after_current is None:
                break
            sorted_pins.append(post_after_current)
            feed_id, item_no, pinned_after = post_after_current
            current_item_no = item_no

        return sorted_pins

    def select_other_rows(self):
        if self.pins_mode == 'pins_first':
            limit = max(0, self.get_query_limit() - len(self.pinned_rows))
            offset = max(0, self.tb - len(self.all_pinned_rows))
            where_append_lines = ['AND pinned_after IS NULL']
        else:
            limit = None
            offset = None
            where_append_lines = []
        return self.select_rows(
            query=' '.join(
                self.get_select_lines() +
                self.get_from_lines() +
                self.get_where_lines() +
                where_append_lines +
                self.get_order_lines() +
                self.get_limit_lines(limit=limit) +
                self.get_offset_lines(offset=offset)
            ),
            query_args=self.get_query_args(),
        )
