# coding: utf-8



import logging
import datetime

from django.conf import settings
from at.common import utils
from at.common import Rubrics
from at.aux_.feeds import loaders
from at.aux_ import Likes
from at.aux_ import entries



logger = logging.getLogger(__name__)


class PopularLoader(loaders.PostsLoader):

    pager_type = 'timestamp'
    select_fields = (
        'Posts.person_id',
        'Posts.post_no',
        'Posts.store_time',
        'Posts.store_time_usec',
    )

    defaults = {
        'limit': None,
        'tb': utils.usec,
        'metarubric': None
    }

    rubric = None

    direction = '<'
    tb_field = None
    no_filter_metarubrics = ['all', '']

    def __init__(self, viewer_id, **params):
        super(PopularLoader, self).__init__(viewer_id, **params)
        if self.metarubric and self.metarubric not in self.no_filter_metarubrics:
            self.rubric = Rubrics.Rubrics().by_metarubric(self.metarubric)

    def get_query_args(self):
        args = super(PopularLoader, self).get_query_args()

        args['store_time_usec'] = self.tb
        if self.rubric:
            args['rubric_id'] = self.rubric.rubric_id
        return args

    def get_from_lines(self):
        return [
            'FROM Posts',
            self.ACCESS_JOIN,
        ]

    def get_where_lines(self):
        return [
            'WHERE',
            self.tb_field, self.direction, '%(store_time_usec)s',
            self.rubric_filter,
            self.ACCESS_CONDITION,
        ]

    def get_order_lines(self):
        return [
            'ORDER BY', self.tb_field, 'DESC'
        ]

    @property
    def hidden_rubrics(self):
        return Rubrics.get_all_hidden_rubrics()

    @property
    def rubric_filter(self):
        if self.rubric is None:
            hide_rubrics = [-1] + self.hidden_rubrics
            hidden_rubrics_str = ','.join(map(str, hide_rubrics))
            return "AND rubric_id NOT IN (%s)" % hidden_rubrics_str
        return "AND rubric_id = %(rubric_id)s"


class AllPostsLoader(PopularLoader):
    """Загрузка постов для категории все посты"""
    tb_field = "store_time_usec"

    @property
    def hidden_rubrics(self):
        return []


class InterestingPostsLoader(PopularLoader):
    """Загрузка постов для категории интересное"""
    tb_field = "store_time_interesting_usec"


class MostInterestingPostsLoader(PopularLoader):
    """Загрузка постов для категории самое интересное"""
    tb_field = "store_time_most_interesting_usec"


class PostScorer(object):
    MAP = [
        (
            'store_time_interesting_usec',
            settings.AT_INTERESTING_SCORE,
            datetime.timedelta(days=settings.AT_INTERESTING_LIFETIME)
        ),
        (
            'store_time_most_interesting_usec',
            settings.AT_MOST_INTERESTING_SCORE,
            datetime.timedelta(days=settings.AT_MOST_INTERESTING_LIFETIME)
        )
    ]

    def __init__(self, feed_id, post_no):
        self.feed_id = feed_id
        self.post_no = post_no

    def score(self, connection):
        entry = entries.load_entry(self.feed_id, self.post_no)
        likes = _get_likes(self.feed_id, self.post_no)
        commenters = _get_commenters(connection, self.feed_id, self.post_no)
        if hasattr(entry, 'poll_id'):
            voters = _get_voters(connection, entry.poll_id)
        else:
            voters = 0
        entry.score = round(likes / 2 + voters / 10 + commenters)
        logger.info('Scorer for (%d, %d): %d', self.feed_id, self.post_no, entry.score)
        now = datetime.datetime.now()
        now_usec = utils.usec()
        for field, min_score, lifetime in self.MAP:
            if getattr(entry, field) is None:
                if entry.score >= min_score and entry.store_time + lifetime > now:
                    setattr(entry, field, now_usec)
        entry.save()

def _get_likes(feed_id, post_no):
    try:
        return int(list(Likes.get_like_counts([
            '%s.%s.0' % (feed_id, post_no)
        ]).values())[0][1])
    except (IndexError,ValueError):
        return 0

def _get_commenters(connection, feed_id, post_no):
    return connection.execute('''
            select count(distinct author_uid) from Comments
            where
                person_id = %s
            and post_no = %s
            and comment_id > 0
            and deleted = 0
        ''', (feed_id, post_no)).fetchone()[0]

def _get_voters(connection, poll_id):
    return connection.execute('''
            select count(distinct uid)
            from Votes
            where
                poll_id = %s
        ''', (poll_id,)).fetchone()[0]
