# encoding: utf-8


from django.utils import timezone
from django.conf import settings
from django.db import models
import yenv

from at.common import exceptions
from at.common import Inflector
from at.common import utils
from at.common import dbswitch
from at.common.Context import context
from at.common.groups import GroupType


class Follower(models.Model):
    class Meta:
        db_table = 'Followers'
        unique_together = (
            'uid',
            'person_id',
        )

    uid = models.ForeignKey(
        'aux_.Person',
        db_column='uid',
        related_name='follow_on',
    )
    person_id = models.ForeignKey(
        'aux_.Person',
        db_column='person_id',
        related_name='followers',
    )


class Friend(models.Model):
    class Meta:
        db_table = 'FriendGroupMember'

    uid = models.ForeignKey(
        'aux_.Person',
        db_column='uid',
        related_name='friends',
    )
    person_id = models.ForeignKey(
        'aux_.Person',
        db_column='person_id',
        related_name='friend_by',
    )
    fgroup_id = models.IntegerField(default=GroupType.FRIEND)
    add_date = models.DateTimeField(null=False, default=timezone.now)
    mutually = models.IntegerField(null=False, default=0)

    unique_together = (
        'uid',
        'person_id',
        'fgroup_id',
    )


class Person(models.Model):
    __inflections = None

    class Meta:
        db_table = 'persons'

    person_id = models.BigIntegerField(primary_key=True, null=False)
    login = models.CharField(max_length=255, null=False, default='')
    title = models.CharField(max_length=255, null=False, default='', db_index=True)
    title_eng = models.CharField(max_length=255, null=False, default='')
    picture_html = models.CharField(max_length=255, null=False, default='')
    description = models.TextField(max_length=255, null=False, default='')
    creation_date = models.DateTimeField(null=False, default=timezone.now)
    modification_date = models.DateTimeField(null=False, default=timezone.now)

    COMMUNITY = 'community'
    PROFILE = 'profile'
    USER_TYPES = (
        (COMMUNITY, COMMUNITY),
        (PROFILE, PROFILE)
    )

    user_type = models.CharField(max_length=255, choices=USER_TYPES, null=False, default=PROFILE)

    STATUSES = (
        ('normal', 'normal'),
        ('memorial', 'memorial'),
        ('deleted', 'deleted'),
        ('dismissed', 'dismissed')
    )

    status = models.CharField(max_length=255, choices=STATUSES, null=False, default='normal')

    SEX = (
        ('man', 'man'),
        ('woman', 'woman'),
        ('unknown', 'unknown')
    )

    sex = models.CharField(max_length=255, choices=SEX, default='man')
    mood = models.CharField(max_length=255, null=False, default='')
    qu = models.IntegerField(null=False, default=1)
    qu_up_time = models.DateTimeField(null=False, default='1970-01-01 00:00:00+0000')
    bd_year = models.SmallIntegerField(null=False, default=0)
    bd_month = models.SmallIntegerField(null=False, default=0)
    bd_day = models.SmallIntegerField(null=False, default=0)
    score = models.IntegerField(null=False, default=0)
    email = models.CharField(max_length=255, null=False, default='')

    COMMUNITY_TYPES = (
        ('OPENED_COMMUNITY', 'OPENED_COMMUNITY'),
        ('PREMODERATED_COMMUNITY', 'PREMODERATED_COMMUNITY'),
        ('CLOSED_COMMUNITY', 'CLOSED_COMMUNITY'),
        ('PREMODERATED_CLOSED_COMMUNITY', 'PREMODERATED_CLOSED_COMMUNITY'),
        ('NONE', 'NONE')
    )

    community_type = models.CharField(max_length=255, choices=COMMUNITY_TYPES, null=False)
    has_access = models.BooleanField(null=False, default=False)

    def __eq__(self, other):
        if not isinstance(other, Person):
            return False
        return self.person_id == other.person_id

    def is_community(self):
        return self.user_type == 'community'

    def is_premoderated(self):
        return self.community_type in ['PREMODERATED_COMMUNITY', 'PREMODERATED_CLOSED_COMMUNITY']

    def is_private_community(self):
        return self.community_type in ['CLOSED_COMMUNITY', 'PREMODERATED_CLOSED_COMMUNITY']

    @property
    def inflections(self):
        if self.__inflections is None:
            self.__inflections = Inflector.inflect(self.title, self.sex)
        return self.__inflections

    @property
    def is_rw(self):
        return False

    @property
    def metatype(self):
        return utils.is_community_id(self.person_id) and "group" or "personal"

    @property
    def picture_url(self):
        if self.picture_html:
            return '%s/%s/' % (settings.AVATAR_GET_URL, self.picture_html)
        else:
            return self.default_picture

    @property
    def default_picture(self):
        # Вроде как используется только для клубов:
        return "%s/%s/" % (settings.AVATAR_GET_URL, self.person_id)

    def get_title(self, inflection=None):
        res = None
        if context.language == 'en':
            res = self.title_eng or self.title
        else:
            if self.metatype == 'personal':
                if self.title:
                    res = self.inflections.get(inflection) or self.title
            else:
                res = self.title
        return res or self.login

    @property
    def dismissed(self):
        return self.status == 'dismissed'

    @staticmethod
    def get_person(person_id):
        try:
            persons = Person.get_persons([person_id])
            # Тут и далее long кастуется в str, см. AT-1012
            person = persons[str(person_id)]
        except KeyError:
            raise exceptions.NotFound('User %s not found' % person_id)
        return person

    @staticmethod
    def get_persons(person_id_list):
        person_id_list = [int(person_id) for person_id in person_id_list]
        persons = Person.objects.filter(person_id__in=person_id_list).all()
        return {str(person.person_id): person for person in persons}

    @staticmethod
    def get_by(**kwargs):
        return Person.objects.filter(**kwargs).first()

    # for django admin
    REQUIRED_FIELDS = []
    # login is not unique (clubs and persons)
    USERNAME_FIELD = 'person_id'

    @property
    def is_active(self):
        return self.status == 'normal'

    @property
    def is_anonymous(self):
        return False

    @property
    def is_authenticated(self):
        return True

    @property
    def is_staff(self):
        if yenv.type != 'production':
            return True
        return self.login in settings.ADMINS

    @property
    def is_superuser(self):
        return self.is_staff

    def has_module_perms(self, *args, **kwargs):
        return self.is_staff

    def has_perm(self, *args, **kwargs):
        return self.is_staff


def GetClubID(s):
    try:
        return int(s)
    except:
        try:
            return Person.get_by(login=s, user_type='community').person_id
        except AttributeError:
            raise exceptions.NotFound(s)
