# *-* encoding: utf-8 *-*

import logging
import re
import urllib.request, urllib.parse, urllib.error

from lxml import etree as ET
from .xslt.transform import XSLTransformer, xslt_custom_function

from django.conf import settings
from at.common import resizer
from at.common import exceptions
from at.common import utils
from at.common import postlink
from at.common import Context
from at.common import Maillists
from at.aux_ import models
from at.aux_ import ProfileStorage


_log = logging.getLogger(__name__)

SNIPPET_DOTS = ' [...]'


def fix_text(text, template='body-formatter', result_as_string=True):
    if isinstance(text, ET._Element):
        text = ET.tostring(text)
    return XSLTransformer.getInstance().apply_template(
                '<div>%s</div>' % text, # оборачиваю в дополнительный тег, чтобы всегда был корневой тег
                template, result_as_string)


def remove_tags(text):
    transform = XSLTransformer.getInstance()
    processed_text = fix_text(text)
    return (transform.apply_template(processed_text,
            'textify', False).getroot().text or '').strip()


def get_firstline(body, maxlen=78, default_text=SNIPPET_DOTS):
    firstline = remove_tags(body)
    if len(firstline) <= maxlen:
        return firstline or default_text
    return firstline[:maxlen].rsplit(' ', 1)[0] or default_text


@xslt_custom_function('get-feed-title') # NB: adds one more parameter (ctx) in the first place
def get_feed_title_xslt(feed_id, inf=None, person=True):
    # XXX NB a parameter, if passed, is always a list! Should pop from it!
    feed_id = feed_id.pop()
    is_club = (person is not True and person and not person.pop())
    if inf is not None:
        inf = inf.pop() if inf else None
    try:
        return makeRepresentation(feed_id, is_club).get_title(inf)
    except exceptions.NotFound:
        return feed_id
    except Exception as e:
        _log.error('get_feed_title_xslt failed', exc_info=True)
        return feed_id


MAX_IMAGE_WIDTH = 600

@xslt_custom_function('resize-image') # NB: adds one more parameter (ctx) in the first place
def resize_image(url):
    if isinstance(url, list):
        url = url[0]
    url = utils.force_unicode(url)
    if url.startswith('data:'):
        return url
    if re.match('https?://(img-)?fotki\.yandex.*', url):
        return re.sub('_orig|_XL|_XXL', '_L', url)
    elif re.match('https?://.*(yandex-team.ru).*', url):
        return url
    return resizer.get_resizer().resize(url, MAX_IMAGE_WIDTH)


@xslt_custom_function('current-language') # NB: adds one more parameter (ctx) in the first place
def current_language():
    return Context.context.language


@xslt_custom_function('current-item-link')
def current_item_link():
    return postlink.make_url(Context.context.feed_id,
                             Context.context.item_no,
                             Context.context.comment_id)


@xslt_custom_function('remove-hide-ref')
def remove_hide_ref(url):
    # XXX NB a parameter, if passed, is always a list! Should pop from it!
    # XXX NB: url[0] is not string and not unicode, it's lxml inner type
    url = utils.force_unicode(url[0])
    if 'h.yandex-team.ru' in url:
        # unquoting url from hideref query params
        return urllib.parse.unquote(re.sub('(https?:)?\/\/h\.yandex-team\.ru/\?', '', url))
    return url


class Representation(object):
    def __init__(self):
        self.__lang = None
        self.uid = None

    def get_language(self):
        raise NotImplementedError

    def __get_lang(self):
        if self.__lang is None:
            self.__lang = self.get_language()
        return self.__lang
    language = property(__get_lang)

    def get_to_header(self):
        return

    @property
    def title(self):
        return self.get_title()

    @property
    def short_title(self):
        return self.get_short_title()

    @property
    def yauser(self):
        return self.get_yauser()

    def __eq__(self, other):
        if self.uid is None and other.uid is None:
            return self.url == other.url
        else:
            return self.uid == other.uid

    def __hash__(self):
        return self.uid


class MailListRepresentation(Representation):
    ENGLISH_MAILLISTS = settings.ENGLISH_MLS

    def __init__(self, email):
        super(MailListRepresentation, self).__init__()
        self.email = email
        self.url = 'https://ml.yandex-team.ru/lists/' + self.email
        self.type = 'mail'

    def get_language(self):
        return ['ru', 'en'][int(self.email in self.ENGLISH_MAILLISTS)]

    def get_email(self):
        return self.__email

    def get_title(self):
        return self.email.split('@')[0]

    def get_short_title(self):
        return self.title # XXX может быть, надо title сделать полным адресом, а тут оставить укороченный

    def get_yauser(self, inflection = None):
        return '<a href="%s">%s</a>' % (self.url, self.email)


class FeedRepresentation(Representation):

    def __init__(self, uid):
        super(FeedRepresentation, self).__init__()
        self.uid = uid
        self.__source = None

    def get_language(self):
        return ProfileStorage.get_language(self.uid)

    def get_title(self, inflection=None):
        title = self.source.get_title(inflection)
        title = re.compile(r'[^\w ]+', flags=re.UNICODE).sub('', title)
        return title

    def get_short_title(self):
        return self.title.split(' ')[0]

    def get_url(self, item_no = None, comment_id = None):
        url = postlink.make_url(
                item_no=item_no,
                comment_id=comment_id,
                source=self.source,
                )
        if comment_id:
            return '%s?for_reply=text#reply-%s-%s' % (url, self.login, comment_id)
        return url

    url = property(get_url)

    def link(self, page):
        return '/'.join((self.url.strip('/'), page))

    def get_yauser(self, inflection = None):
        return '<a href="%s">%s</a>' % (self.url, self.source.get_title(inflection))

    @property
    def newsfeed(self):
        return postlink.newsfeed()

    @property
    def type(self):
        return self.source.metatype

    @property
    def is_club(self):
        return self.type == 'group'

    @property
    def source(self):
        if self.__source is None:
            self.__source = models.Person.get_person(self.uid)
        return self.__source

    def __getattr__(self, attr):
        s = self.source
        return getattr(s, attr)



def makeRepresentation(target, is_club=None):
    if isinstance(target, Representation):
        return target
    if isinstance(target, int):
        return FeedRepresentation(target)
    if isinstance(target, str):
        if target.isdigit():
            return FeedRepresentation(int(target))
        if '@' in target:
            if not Maillists.is_email_ml(target):
                login = target.split('@', 1)[0]
                try:
                    uid = utils.getAuthInfo(login=login).uid
                except:
                    raise ValueError('Neither maillist nor user: "%s"' % target)
                else:
                    return FeedRepresentation(uid)
            else:
                return MailListRepresentation(target)
        if is_club:
            return FeedRepresentation(models.GetClubID(target))
        try:
            return FeedRepresentation(utils.getAuthInfo(login=target).uid)
        except exceptions.NotFound:
            return FeedRepresentation(models.GetClubID(target))
    raise ValueError('unexpected value "%s" of type %s' % (target, type(target)))
