# -*- coding: utf-8 -*-

import logging

import lxml.etree as ET

from at.common.utils import get_connection
from at.common.MagicTable import MagicTable


_log = logging.getLogger('aux/SimpleLayout')

WidgetNames = MagicTable('WidgetNames', 'widget_id, name, enabled')
id2name = lambda i: WidgetNames().by_widget_id(i)['name']
name2id = lambda name: WidgetNames().by_name(name)['widget_id']

widget_setting_pattern = '<widget id="%s" disabled="%s"></widget>'


def xml_setting(name, enabled):
    return ET.XML(
            widget_setting_pattern % (name, enabled and 'false' or 'true'))


def get_default_settings():
    return dict([(r['name'], r['enabled']) for r in WidgetNames()])


def get_setting(feed_id, widget_name):
    sql = 'SELECT enabled FROM WidgetSettings WHERE person_id=%s AND widget_id=%s'
    w = WidgetNames().by_name(widget_name)
    widget_id, default = w['widget_id'], w['enabled']
    with get_connection() as connection:
        rows = connection.execute(sql, (feed_id, widget_id)).fetchall()
        if len(rows):
            return bool(rows[0][0])
        else:
            return default


def get_all_settings(feed_id):
    sql = 'SELECT widget_id, enabled FROM WidgetSettings WHERE person_id=%s'
    with get_connection() as connection:
        cursor = connection.execute(sql, (feed_id,))
        ret = dict([row for row in cursor])
    return dict([
                (r['name'], ret.get(r['widget_id'], r['enabled']))
                        for r in WidgetNames()
                ])


def set_setting(feed_id, widget_name, enabled):
    def chk(v, l):
        return str(v).lower() in l
    if chk(enabled, ['false', 'no', '0']):
        enabled = 0
    elif chk(enabled, ['true', 'yes', '1']):
        enabled = 1
    else:
        raise AssertionError('Unexpected setting value %s' % enabled)
    widget_id = name2id(widget_name)
    sql = 'REPLACE INTO WidgetSettings VALUES (%s, %s, %s)'
    with get_connection() as connection:
        connection.execute(sql, (feed_id, widget_id, enabled))

def reset_setting(feed_id, widget_name):
    sql = 'DELETE FROM WidgetSettings WHERE person_id = %s AND widget_id = %s'
    widget_id = name2id(widget_name)
    with get_connection() as connection:
        _log.error("WILL %s %s", feed_id, widget_id)
        connection.execute(sql, (feed_id, widget_id))

class WidgetInfo(object):
    def __init__(self, node):
        a = node.attrib
        self.type = a['type']
        self.id = a['id']
        self.forced = a.get('forced', 'false') == 'true'
        self.text = node.text
        if self.type in ('flow', 'layout'):
            self.items = [WidgetInfo(w) for w in node]
        else:
            self.items = []

    def to_iterate(self, settings, forced):
        return [i for i in self.items if forced or i.forced \
                or settings.get(i.id, False) or self.type == 'layout']

    def node(self, settings, widgets, forced=False):
        forced = forced or self.forced # in case the root is forced
        root = ET.Element('{urn:yaru-widgets}' + self.type, {'id': self.id}, nsmap = {'widgets': 'urn:yaru-widgets'})
        for i in self.to_iterate(settings, forced):
            root.append(i.node(settings, widgets, forced or i.forced))
        if self.type == 'widget':
            root.text = widgets.get(self.id, '') or self.text
        return root

    def get_ids(self, settings, forced=None, typ='widget'):
        forced = self.forced if forced is None else forced
        ids = []
        for i in self.to_iterate(settings, forced):
            ids.extend(i.get_ids(settings, forced or i.forced, typ))
        if self.type == typ:
            ids.append(self.id)
        return ids
    get_wids = lambda self, s, f=None: self.get_ids(s, f)


def getNullLayout(page):
    return WidgetInfo(ET.XML('<widget type="layout" id="%s"/>' % page))

Layouts = {}

def load_layouts(layouts_node):
    for node in layouts_node:
        if isinstance(node, ET._Comment):
            continue
        assert node.tag == 'widget'
        assert node.attrib['type'] == 'layout', node.attrib
        Layouts[node.attrib['id']] = WidgetInfo(node)
            


layout_xml = """
<layouts>
    <widget type="layout" id="with-wids">
        <widget type="flow" id="with-wids-left">
            <widget type="widget" id="id2"/>
            <widget type="widget" id="id3"/>
            <widget type="pseudo-widget" id="id4"/>
        </widget>
    </widget>
    <widget type="layout" id="with-flow">
        <widget type="flow" id="left">
            <widget type="flow" id="inner">
                <widget type="pseudo-widget" id="birthday"/>
            </widget>
            <widget type="pseudo-widget" id="id3"/>
        </widget>
    </widget>
    <widget type="layout" id="flow-forced">
        <widget type="flow" id="left" forced="true">
            <widget type="pseudo-widget" id="birthday"/>
            <widget type="pseudo-widget" id="id3"/>
        </widget>
        <widget type="flow" id="right">
            <widget type="pseudo-widget" id="birthday"/>
            <widget type="pseudo-widget" id="id3"/>
        </widget>
    </widget>
    <widget type="layout" id="forced" forced="true">
        <widget type="flow" id="left">
            <widget type="pseudo-widget" id="birthday"/>
            <widget type="pseudo-widget" id="id3"/>
        </widget>
        <widget type="flow" id="right"/>
    </widget>
</layouts>
"""


layout_node = ET.XML(layout_xml)
import unittest
class TrueDict(dict):
    def __init__(self, value=True):
        self.v = value
    def __getitem__(self, key):
        return self.v
    def get(self, a,b):
        return self.v

class TestLayout(unittest.TestCase):
    def __init__(self, *args, **kw):
        import test_data
        unittest.TestCase.__init__(self, *args, **kw)
        self.ai = test_data.ai

    def setUp(self):
        Layouts.clear()
        load_layouts(layout_node)

    def testBasic(self):
        "Layout парсится более-менее без ошибок"
        assert set(Layouts.keys()) == \
                set(['with-flow', 'flow-forced', 'forced', 'with-wids']),\
                'Unexpected keys of Layouts: %s' % list(Layouts.keys())

    def testFristLevel(self):
        "Контейнеры верхнего уровня отдаются вне зависимости от настроек"
        node = Layouts['with-wids'].node(TrueDict(False), {})
        assert len(node) == 1, ET.tostring(node)
        assert len(node[0]) == 0, ET.tostring(node)

    def testSettings(self):
        "Настройки имеют значение"
        settings = {'id2': 0, 'id3': 1}
        node = Layouts['with-wids'].node(settings, {})
        assert len(node) == 1, ET.tostring(node)
        assert len(node[0]) == 1, ET.tostring(node)
        assert node[0][0].attrib['id'] == 'id3'


    def testWIDs(self):
        "Айдишники тру виджетов собираются правильно"
        wids = set(Layouts['with-wids'].get_wids(TrueDict()))
        assert wids == set(['id2', 'id3']), wids
        wids = set(Layouts['with-wids'].get_wids(TrueDict(False)))
        assert wids == set([]), 'Not empty: %s' % wids


    def testForced(self):
        "forced работает рекурсивно, на разных уровнях"
        node = Layouts['forced'].node(TrueDict(False), {})
        assert len(node) == 2, ET.tostring(node)
        assert len(node[0]) == 2, ET.tostring(node)
        node = Layouts['flow-forced'].node(TrueDict(False), {})
        assert len(node) == 2, ET.tostring(node)
        assert len(node[0]) == 2, ET.tostring(node)
        assert len(node[1]) == 0, ET.tostring(node)

    def testGetSetSetting(self):
        "Настройка виджета ставится и читается"
        set_setting(self.ai.uid, 'photos-me', 0)
        assert get_setting(self.ai.uid, 'photos-me') == False
        set_setting(self.ai.uid, 'photos-me', 1)
        assert get_setting(self.ai.uid, 'photos-me') == True

    def testDefaultSetting(self):
        "Работают дефолтные настройки и reset"
        def ddiff(d1, d2):
            return '\n'.join(['%s %s %s\n' % (k, d1.get(k), d2.get(k)) for k in set(d1.keys()).union(set(d2.keys()))])
        defaults = get_default_settings()
        for k in list(defaults.keys()):
            reset_setting(self.ai.uid, k)
        d = get_all_settings(self.ai.uid)
        assert defaults == d, ddiff(defaults, d)
        p = list(defaults.items())[0]
        set_setting(self.ai.uid, p[0], not p[1])
        d = get_all_settings(self.ai.uid)
        assert defaults != d, ddiff(defaults, d)
        reset_setting(self.ai.uid, p[0])
        d = get_all_settings(self.ai.uid)
        assert defaults == d, ddiff(defaults, d)

# end.

