# coding: utf-8

import threading


class Context(threading.local):

    __default_values = {'language': 'ru'}
    
    def __init__(self):
        self.__bindings = set()

    def __getattr__(self, attr_name):
        if attr_name in self.__default_values:
            value = self.__default_values[attr_name]
            setattr(self, attr_name, value)
            return value
        else:
            raise AttributeError('Context has no attribute %s and no default value' % attr_name)

    def __setattr__(self, attr_name, value):
        super(Context, self).__setattr__(attr_name, value)
        if not attr_name.startswith('_'):
            self.__bindings.add(attr_name)
    
    def set(self, key, value):
        setattr(self, key, value)

    def unset(self, key):
        delattr(self, key)

    def __delattr__(self, attr_name):
        super(Context, self).__delattr__(attr_name)
        if not attr_name.startswith('_'):
            self.__bindings.remove(attr_name)

    def get_bindings(self):
        return dict((key, getattr(self, key)) for key in self.__bindings)

    def get(self, key, factory=None):
        try:
            return getattr(self, key)
        except AttributeError:
            if factory is not None:
                value = factory()
                setattr(self, key, value)
                return value
            else:
                raise

# тут не через Singleton, потому что он не дружит с threading.local
context = Context()


class ContextBinder(object):

    def __init__(self, **kwargs):
        self.new_values = kwargs
        self.old_values = context.get_bindings()

    def __enter__(self):
        for attr, value in list(self.new_values.items()):
            context.set(attr, value)
        return context

    def __exit__(self, exc_type, exc_value, exc_tb):
        for attr in list(self.new_values.keys()):
            context.unset(attr)
        for attr, value in list(self.old_values.items()):
            context.set(attr, value)
