# *-* encoding: utf-8 *-*

import re
import logging

from django.conf import settings
from ids.utils import https

from at.common import utils
from at.common import decorators

_log = logging.getLogger(__name__)


grammar_mappings = {
            'им': 'im',
            'род': 'rod',
            'дат': 'dat',
            'вин': 'vin',
            'твор': 'tvor',
            'пр': 'pred',
            'местн': 'mestn'
            }


@decorators.cache_to_memcache(prefix='inflector_', ttl_seconds=60*60*24*7)
def _do_inflector_request(name, surname, gender):
    params = {
        'action': 'inflector',
        'text': 'persn{%s}famn{%s}gender{%s};fio=1' % (name, surname, gender),
        'wizclient': 'at',
    }
    url = settings.INFLECTOR_URL
    session = https.get_secure_session()
    return session.get(url, params=params, timeout=0.5).json()


@utils.timeit
def inflect(title, sex):
    def parse_answer(inflector_answer):
        return ' '.join(
                    re.match('persn{([^}]+)}famn{([^}]+)}.+', inflector_answer).groups()
                )
    def convert_grammars(grammar):
        return (grammar_mappings[grammar['Grammar']], parse_answer(grammar['Text']))
    try:
        name, surname = utils.force_unicode(title.strip()).split(' ', 1)
    except ValueError:
        return {}
    try:
        res = _do_inflector_request(name, surname, ['m', 'f'][sex=='woman'])
        return dict(convert_grammars(grammar) for grammar in res['Form']
            if grammar['Grammar'] in grammar_mappings)
    except Exception as e:
        _log.warning('Invalid answer from inflector for "%s"', title, exc_info=1)
        return {}
