# -*- coding: utf-8 -*-

import logging
from collections import defaultdict
import itertools
import math
import operator

from django.conf import settings
from at.common import utils
from at.common import dbswitch
from at.common import staff
from at.common.Checker import Checker
from functools import reduce

from at.common.groups import GroupType

_log = logging.getLogger(__name__)


def get_members(person_list):
    return [
            int(p['person']['uid']) for p in person_list
            if not p['person']['official']['is_dismissed']
            and not p['person']['official']['is_robot']
            ]


def get_superiors(group_list):
    return list(itertools.chain(
        *(
            get_members(grp['department']['heads'])
            for grp in group_list
            if not grp.get('is_deleted', False) # in case we filter in request
        )
    ))


def _iterate_subordinates_smart(groups, is_boss=False):
    # bosses are more interested in subordinate bosses,
    # regular people – in whoever is closer.
    # For a boss, first iterate over the bosses of all groups,
    # then over employees.
    # For regular employee, iterate the closest level bosses,
    # then over closest level employees, then same way over
    # lower/higher level groups.
    prev_level = None
    employees = []
    for group in groups:
        level = group['group']['department']['level']
        if prev_level is not None and level != prev_level:
            if not is_boss:
                while employees: yield employees.pop(0)
            prev_level = level
        for b in get_superiors([group['group']]):
            yield b
        for e in get_members([group]):
            employees.append(e)
    for e in employees: yield e


# XXX This wrapper is BAD, because it transforms an iterator into a list.
# So, it shouldn't be used with potentially large or infinite generators
# to prevent memory leak.
def unique_yield(iterator):
    yielded = set()
    for i in iterator:
        if i not in yielded:
            yield i
            yielded.add(i)

def iterate_subordinates_smart(groups, is_boss=False):
    return (i for i in unique_yield(
            _iterate_subordinates_smart(groups, is_boss)
        ))

def speaks_russian(person, api):
    """ Make a guess whether uid speaks russian according to their staff data."""
    if isinstance(person, int):
        # UID passed
        data = api.persons.filter(uid=person).fields(['location', 'language'])[0]
    else:
        # staff data passed
        data = person
    # XXX Currently neither language.native nor language.content contain reliable
    # information, so I use whatever I can.
    RKUB_CODES = ['ru', 'ua', 'kz', 'by']
    # Those in foreign offices or in the home office judge according to ui lang:
    if data['location']['office']['code'] == 'home' or \
            data['location']['office']['city']['country']['code'] not in RKUB_CODES:
        return data['language']['ui'] == 'ru'
    else:
        return True

# Club helpers:
def get_shared_clubs(uids):
    """ Returns the clubs that uids are members of, and how many members in each. """
    sql = """
    SELECT
        fgm.person_id
    FROM
        FriendGroupMember AS fgm JOIN
        Communities AS c ON fgm.person_id = c.feed_id JOIN
        persons AS p ON c.feed_id = p.person_id
    WHERE
        fgm.fgroup_id = %d AND
        c.adult = "public" AND
        c.rubric_id != 99 AND
        NOT p.community_type in ("CLOSED_COMMUNITY", "PREMODERATED_CLOSED_COMMUNITY") AND
        fgm.uid in (%s) AND fgm.person_id >= %s
    """
    # count делаем в питоне, потому что запрос шардированный не по тому полю
    share_info = defaultdict(int)
    with utils.get_connection() as conn:
        for row in conn.execute(
                sql %
                (GroupType.MEMBER, ','.join(map(str, uids)), settings.COMMUNITY_START_ID)
        ):
            share_info[row[0]] += 1
    return dict((k, {'friends': v}) for k, v in share_info.items())


def get_club_stats(feed_ids, days):
    if not feed_ids:
        return {}

    result = defaultdict(lambda: {'members': 0, 'posts': 0, 'comments': 0})
    from at.aux_ import Community # XXX causes circular import
    for feed_id in feed_ids:
        result[feed_id]['members'] = Community.Community()._count_community_users(feed_id)
    sql = 'select person_id, COUNT(IF(comment_id = 0, 1, NULL)), count(*) '\
            'FROM Comments '\
            'WHERE person_id in (%s) '\
            'AND store_time > date_sub(now(), interval %s day) '\
            'GROUP BY person_id'
    with utils.get_connection() as conn:
        for row in \
                conn.execute(sql % (",".join(map(str, feed_ids)), days)):
            feed_id = row[0]
            result[feed_id].update({'posts': row[1], 'comments': row[2]})
    return result


def get_colleagues(person_id, api, MAX_COLLEAGUES=50):
    my_group = api.groupmembership.filter(
        person__uid=person_id,
        group__type='department'
    ).fields([
        'group.id',
        'group.is_deleted',
        'group.department.level',
        'group.department.heads.person.uid',
        'group.department.heads.person.official.is_dismissed',
        'group.department.heads.person.official.is_robot',
        'group.ancestors.id',
        'group.ancestors.is_deleted',
        'group.ancestors.department.level',
        'group.ancestors.department.heads.person.uid',
        'group.ancestors.department.heads.person.official.is_dismissed',
        'group.ancestors.department.heads.person.official.is_robot',
    ])
    if not my_group:
        return []

    my_group = my_group[0].get('group')
    if not my_group:
        return []

    # First, get all superiors with deputees (заместителями):
    superiors = get_superiors([my_group] + my_group['ancestors'])
    is_boss = person_id in superiors
    members = api.groupmembership.filter(
        group__id=my_group['id'],
        group__type='department',
        person__official__is_dismissed=False,
    ).fields(['person.uid', 'person.official.is_dismissed', 'person.official.is_robot'])
    colleagues = set(get_members(members) + superiors)

    # If number of superiors is unsufficient, add members of
    # subordinate deparments starting from higher level.
    # If number is still insufficient, add members of adjacent and containing groups
    parent_id = sorted(my_group['ancestors'],key=lambda g: g['department']['level'])[-1]['id']

    def wrp(desc=True, **kwargs):
        return api.groupmembership.filter(person__official__is_dismissed=False, **kwargs).fields([
            'person.uid',
            'person.official.is_dismissed',
            'person.official.is_robot',
            'group.department.level',
            'group.department.heads.person.uid',
            'group.department.heads.person.official.is_dismissed',
            'group.department.heads.person.official.is_robot',
        ]).sort(('-' if not desc else '') + 'group__department__level')

    for another_colleague in iterate_subordinates_smart(
                itertools.chain(
                                wrp(group__ancestors__id=my_group['id']),
                                #wrp(desc=False, id=ancestors_ids), # group members of parent groups cause a lot of cocaine in tools
                                wrp(group__id=parent_id),
                                wrp(group__ancestors__id=parent_id)
                ), is_boss):
        if len(colleagues) >= MAX_COLLEAGUES:
            break
        colleagues.add(another_colleague)
    colleagues.difference_update([person_id]) # not colleagues.remove(person_id), b/c it raises KeyError if not present
    return list(colleagues)


def score(friends, members, posts, comments):
    if not members: return 0 # should never happen
    return float(friends) \
            * math.log(3*posts + comments + 2, 2) \
            / math.pow(members, 0.67)


RECOMMENDED_CLUBS_RU = [
        4611686018427387915,    # tools
        4611686018427388043,    # seminar
        4611686018427388363,    # mag
]

RECOMMENDED_CLUBS_EN = [
        4611686018427388493,    # mag-en
        4611686018427388052,    # turkey
]

BLACKLIST_CLUBS = [
        4611686018427388109,    # events, b/c duplicates seminar
        4611686018427387976,    # bbs
        4611686018427388219,    # HR, very dull
]


OFFICE_CLUBS = {
        # Moscow, club Life:
        'rrm': [4611686018427388094],
        'redrose': [4611686018427388094],
        'rrs': [4611686018427388094],
        'rp': [4611686018427388094],
        'au': [4611686018427388094],

        'kievleo': [4611686018427388178], # ukraine
        'od2': [4611686018427388178, 4611686018427388176], # ukraine, odessa
        'simf': [4611686018427388137], # simf
        'spb': [4611686018427388195], # piter
        'eburg': [4611686018427388135], # ekb
        'rnd': [4611686018427388165], # rostov
        'kzn': [4611686018427388193], # kzn
        'nsk': [4611686018427388194], 'tech': [4611686018427388194], # nsk-live
        'minsk': [4611686018427388238], # minsk
        'nn': [4611686018427388434], # nizhny-novgorod
        }
def get_office_clubs(office_key):
    return OFFICE_CLUBS.get(office_key, [])


IGNORED_CLUBS = RECOMMENDED_CLUBS_RU + RECOMMENDED_CLUBS_EN + \
                BLACKLIST_CLUBS + reduce(operator.add, list(OFFICE_CLUBS.values()))

def get_recommended_clubs(uids, count=5, days=90, reqs=None, score_fn=score):
    requirements = {'friends': 4, 'posts': 9, 'comments': 30, 'members': 0}
    requirements.update(reqs or {})
    checker = Checker(requirements)
    checker.load(get_shared_clubs(uids))
    # Apply requirements[friends] before calling get_club_stats() to minimize DB load
    checker.filter('friends', IGNORED_CLUBS)
    checker.load(get_club_stats(checker.get_ids(), days))
    checker.filter()
    return sorted([
                (feed_id, score_fn(**checker[feed_id]), checker[feed_id])
                for feed_id in checker.get_ids()
            ], key=operator.itemgetter(1), reverse=True)[:count]



