# *-* coding: utf-8 *-*

__all__ = ["retry_x_times_at", "cache_to_memcache", "or_none"]

import functools
import logging
import hashlib

from django.core.cache import cache

_log = logging.getLogger(__name__)


def retry_x_times_at(exception, x=5):
    """Retry function X times on :exception:, raise if other"""
    def wrapper(func):
        @functools.wraps(func)
        def inner_wrapper(*args, **kwargs):
            tries = x
            while tries >= 0:
                try:
                    tries -= 1
                    return func(*args, **kwargs)
                except exception:
                    if tries > 0:
                        continue
                    else:
                        raise
        return inner_wrapper
    return wrapper


def _build_args_hash(args, kwargs, prefix, name):

    args_hash = ''.join(map(str, [hash(arg) for arg in args] + \
            [hash(key) for key in kwargs] + \
            [hash(value) for value in list(kwargs.values())]))
    hash_string = hashlib.md5(args_hash.encode('utf-8')).hexdigest().encode('utf-8')
    if isinstance(prefix, str):
        prefix = prefix.encode('utf-8')
    if isinstance(name, str):
        name = name.encode('utf-8')
    key = b'%s%s(%s)' % (prefix, name, hash_string)
    return key


def cache_to_memcache(prefix='', ttl_seconds=60*60*24, key_func=_build_args_hash):
    """Store func result in memcache"""
    def wrapper(func):
        @functools.wraps(func)
        def inner_wrapper(*args, **kwargs):
            key = key_func(args, kwargs, prefix, func.__name__)
            result = cache.get(key)
            if result is None:
                result = func(*args, **kwargs)
                _log.debug('Cache miss (setting) %s=%s', key, result)
                cache.set(key, result, timeout=ttl_seconds)
            else:
                _log.debug('Cache hit %s=%s', key, result)
            return result
        return inner_wrapper
    return wrapper


def or_none(func):
    """Return None on fail"""
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception:
            return None
    return wrapper
