# coding: utf-8

import cgi
import re
import logging
import itertools

import yenv

from django.conf import settings
from at.common import exceptions
from at.common import utils
from at.aux_.models import Person, GetClubID

_log = logging.getLogger(__name__)

SCHEME = settings.AT_SCHEME
DOMAIN = settings.AT_HOST


class MyUrlParser(object):
    flags = re.DOTALL | re.IGNORECASE | re.UNICODE
    scheme = r'https?://'
    domain = r'\.at\.yandex-team\.ru'
    # so that in development both local and production urls are recognised:
    if yenv.type != 'production':
        domain = r'(?:[\w-]*\.at\.[.\w-]+\.|\.at\.)yandex-team\.ru'
    club = scheme + 'clubs' + domain + r'/(?P<club>[\w-]+)'
    blog = scheme + r'(?!clubs)(?P<login>[^\.]+)' + domain
    host = '(?:%s|%s)' % (blog, club)
    stop = r'(?:$|/?(?![/\d]))'

    yauser_regexp = re.compile(r'<ya [^/]*user="(?P<login>[^"]+)"[^/]*/(ya)?>', flags)

    blog_regexp = re.compile(blog + stop, flags)
    club_regexp = re.compile(club + stop, flags) # Only for testing

    post_regexp = re.compile(host + r'/replies\.xml\?item_no=(?P<item_no>\d+)', flags)
    comment_regexp = re.compile(host + r'/replies\.xml\?parent_id=(?P<comment_id>\d+)(&|&amp;|&amp;amp;)item_no=(?P<item_no>\d+)', flags)
    new_post_regexp = re.compile(host + r'/(?P<item_no>\d+)' + stop, flags)
    new_comment_regexp = re.compile(host + r'/(?P<item_no>\d+)/(?P<comment_id>\d+)' + stop, flags)
    def __init__(self, default_blog_id = None, default_club_id = None):
        self.default_blog_id = default_blog_id
        self.default_club_id = default_club_id

    def get_uid(self, login):
        try:
            return utils.getAuthInfo(login=login).uid
        except exceptions.NotFound:
            _log.warning("Entry body contains incorrect login = %s." % (login,))
            return self.default_blog_id

    def get_feed_id(self, match):
        try: login = match.group('login')
        except: login = None
        try: club = match.group('club')
        except: club = None
        if login and club:
            msg = "Has both 'club' and 'login' id: %s, %s." % (club, login)
            if self.default_club_id is not None:
                # in development, see the error
                raise ValueError(msg)
            else:
                _log.error(msg)
                return None
        if club:
            try:
                return GetClubID(club)
            except exceptions.NotFound:
                return self.default_club_id
        assert login, 'Neither login nor club slug matched %s by %s' \
                % (match.groups(), match.re.pattern)
        return self.get_uid(login)

    def finditer_blog(self, text):
        for regexp in [
                self.blog_regexp,
                self.club_regexp,
                self.yauser_regexp,
                ]:
            for match in regexp.finditer(text):
                if not match: continue
                for group_name in ('login', 'club'):
                    try:
                        uid = self.get_uid(match.group(group_name))
                    except IndexError:
                        uid = None
                    else:
                        break
                if uid:
                    yield (uid, 0, 0)

    def finditer_item(self, text):
        for regexp in [
                self.comment_regexp,
                self.new_comment_regexp,
                self.post_regexp,
                self.new_post_regexp,
                ]:
            for match in regexp.finditer(text):
                if not match: continue
                try:
                    feed_id = self.get_feed_id(match)
                    item_no = int(match.group('item_no'))
                    # to put everything in one cycle:
                    try: comment_id = int(match.group('comment_id'))
                    except IndexError: comment_id = 0
                    yield (feed_id, item_no, comment_id)
                except:
                    _log.warn('', exc_info=1)
                    continue

    def finditer(self, text):
        return itertools.chain(
                self.finditer_item(text),
                self.finditer_blog(text)
                )

    def match_item(self, url):
        for regexp in [
                self.comment_regexp,
                self.new_comment_regexp,
                self.post_regexp,
                self.new_post_regexp,
                ]:
            match = regexp.match(url)
            if not match: continue
            try:
                feed_id = self.get_feed_id(match)
                item_no = int(match.group('item_no'))
                # to put everything in one cycle:
                try: comment_id = int(match.group('comment_id'))
                except IndexError: comment_id = 0
                return (feed_id, item_no, comment_id)
            except:
                _log.warn('', exc_info=1)
                continue

    def match_blog(self, url):
        for regexp in [
                self.blog_regexp,
                self.club_regexp,
                self.yauser_regexp,
                ]:
            match = regexp.match(url)
            if not match: continue
            try:
                return (self.get_feed_id(match), 0, 0)
            except:
                _log.warn('', exc_info=1)
                continue

    def match(self, url):
        return self.match_item(url) or self.match_blog(url)


def parse_url(url):
    """ Returns feed_id, item_no | None, comment_id | None) """
    return MyUrlParser.getInstance().match(url)

def newsfeed():
    return "%s://my.%s" % (SCHEME, DOMAIN)

def make_url(feed_id=None, item_no=None, comment_id=None, escape=False, source=None):
        assert feed_id or source, "Neither feed_id nor source provided to make_url"
        if source:
            model = source
        else:
            model = Person.get_person(feed_id)
        
        if model.community_type == 'NONE':
            url_pattern = '%(scheme)s://%(feed)s.%(base)s'
        else:
            url_pattern = '%(scheme)s://clubs.%(base)s/%(feed)s'

        if item_no:
            url_pattern += '/%(item)s'
        
        if comment_id:
            url_pattern += '/%(comment)s'
        result = url_pattern % {
            'base': DOMAIN,
            'scheme': SCHEME,
            'feed': model.login,
            'item': item_no,
            'comment': comment_id
        }

        if escape:
            return cgi.escape(result)
        return result
