from typing import Optional

import requests

from django.utils.functional import SimpleLazyObject
from django.utils.translation import gettext_lazy as _

from django.conf import settings


class SenderClient:
    TRANSACTION_URL_TEMPLATE = '/api/0/{account_slug}/transactional/{campaign_slug}/send'
    STATUS_URL_TEMPLATE = '/api/0/{account_slug}/transactional/{campaign_slug}/status?message_id={message_id}'

    def __init__(
        self,
        host: str,
        authorization_key: str,
        account_slug: str,
        verify_host: bool = True,
        timeout: int = 10,
        retries: int = 3,
        session: Optional[requests.Session] = None
    ):
        self.host = host
        self.authorization_key = authorization_key
        self.account_slug = account_slug
        self.verify_host = verify_host

        self.timeout = timeout
        self.retries = retries
        self.session = session or requests.Session()

    def send_message(
        self,
        subject: str,
        sender: str,
        receiver: str,
        headers: dict,
        campaign_slug: str,
        template_args: Optional[dict] = None,
        countdown: Optional[int] = None,
        expires: Optional[int] = None,
        ignore_empty_email: bool = True,
        sync: bool = False,
    ):
        if not settings.SENDER_ENABLED:
            return {'result': {'status': 'cancelled', 'message': _("Отправка писем выключена (SENDER_ENABLED=false)")}}

        headers['subject'] = subject

        data = {
            'from_name': sender,
            'from_email': settings.SENDER_ROBOT_EMAIL,
            'to_email': receiver,
            'headers': headers,
            'host': self.host,
            'async': not sync,
            'ignore_empty_email': ignore_empty_email,
        }
        if template_args:
            data['args'] = template_args
        if countdown:
            data['countdown'] = countdown
        if expires:
            data['expires'] = expires

        url_kwargs = {
            'account_slug': self.account_slug,
            'campaign_slug': campaign_slug,
        }
        url = f'{self.host}{self.TRANSACTION_URL_TEMPLATE.format(**url_kwargs)}'

        response = self.session.request(
            'post',
            url=url,
            json=data,
            auth=(self.authorization_key, ''),
            verify=self.verify_host,
            timeout=self.timeout,
        )
        response.raise_for_status()

        return response.json()

    def status(self, campaign_slug: str, message_id: str):
        url_kwargs = {
            'account_slug': self.account_slug,
            'campaign_slug': campaign_slug,
            'message_id': message_id,
        }

        url = f'{self.host}{self.STATUS_URL_TEMPLATE.format(**url_kwargs)}'
        response = self.session.request(
            'get',
            url=url,
            auth=(self.authorization_key, ''),
            verify=self.verify_host,
            timeout=self.timeout,
        )
        response.raise_for_status()

        return response.json()


def get_client() -> SenderClient:
    return SenderClient(
        host=settings.SENDER_HOST,
        authorization_key=settings.SENDER_AUTHORIZATION_KEY,
        verify_host=settings.VERIFY_SENDER_HOST,
        account_slug=settings.SENDER_ACCOUNT_SLUG,
    )


sender_client = SimpleLazyObject(lambda: get_client())
