# coding: utf-8

import json
from ids.utils import https

from django.conf import settings
from at.common import oauth
from at.common.decorators import retry_x_times_at
from at.common.tvm2_client import get_service_ticket


DEFAULT_TIMEOUT = 3


class InvalidAnswer(Exception):
    pass


class BadRequest(Exception):
    pass


class Resource(object):
    def __init__(self, name, session, timeout, base, args=None):
        self.url = name
        if args:
            self.args = args
        else:
            self.args = {}
        self._data = None
        self.session = session
        self.base = base
        self.timeout = timeout

    def _set_arg(self, key, value, join=True):
        self._data = None
        if join and key in self.args:
            self.args[key] = self.args[key] + ',' + value
        else:
            self.args[key] = value

    def filter(self, **kwargs):
        for key, value in list(kwargs.items()):
            key = key.replace('__', '.')
            self._set_arg(key, value)
        return self

    def _set_metafield(self, meta, fields=None, join=False):
        if not fields: return self
        if isinstance(fields, int):
            fields = str(fields)
        if not isinstance(fields, str):
            fields = ','.join(fields)
        fields = fields.replace('__', '.')
        self._set_arg(meta, fields, join)
        return self

    def sort(self, fields=None):
        return self._set_metafield('_sort', fields)

    def query(self, query=None):
                return self._set_metafield('_query', query)

    def fields(self, fields=None):
        return self._set_metafield('_fields', fields)

    def limit(self, limit=None):
        return self._set_metafield('_limit', limit, False)

    def page(self, page=1):
        return self._set_metafield('_page', page)

    def all(self, per_page=50):
        data = self.limit(per_page)
        total_pages = data['pages']
        for x in range(2, total_pages+2):
            yield data
            data = self.limit(per_page).page(x)

    def _get_url(self):
        return (self.base + self.url, self.args)

    def _do_request(self):
        self._data = self.get(*self._get_url())

    def get_data(self):
        if self._data is None:
            self._do_request()
        return self._data
    data = property(get_data)

    def __iter__(self):
        return iter(self.data['result'])

    def __getitem__(self, key):
        if isinstance(key, int):
            try:
                return self.data['result'][key]
            except (KeyError, IndexError):
                raise IndexError(key)
        else:
            return self.data[key]

    @retry_x_times_at(InvalidAnswer, 5)
    def get(self, path, args):
        resp = self.session.get(
            path,
            timeout=self.timeout,
            params=args,
        )
        if resp.status_code / 100 == 5:
            raise InvalidAnswer("500 code error from staff", resp)
        if resp.status_code / 100 == 4:
            raise BadRequest("Bad request", path, args, resp)
        try:
            return json.loads(resp.content.decode('utf-8'))
        except ValueError:
            raise InvalidAnswer("Invalid answer from staff!", resp)


class Api(object):
    base = settings.STAFF_API_URL

    def __init__(self, cookie=None, user_ticket=None, version='v3', timeout=DEFAULT_TIMEOUT):
        self.base = self.base + '/' + version + '/'
        self.session = https.get_secure_session()
        self.timeout = timeout
        cookies = {} if not cookie else {'Session_id': cookie}
        self.session.headers['X-Ya-Service-Ticket'] = get_service_ticket('staff')
        if user_ticket:
            self.session.headers['X-Ya-User-Ticket'] = user_ticket
        self.session.cookies.update(cookies)
        self.session.headers['Content-Type'] = 'application/json'

    @classmethod
    def with_ticket_for_user(cls, auth_info, timeout=DEFAULT_TIMEOUT):
        return cls(user_ticket=auth_info.user_ticket, timeout=timeout)

    def __getattr__(self, name):
        if name.startswith('__'):
            try:
                return self.__dict__[name]
            except KeyError:
                raise AttributeError(name)
        else:
            return Resource(name, self.session, self.timeout, self.base)
