# coding: utf-8


import json
from .uriutils import expand
from .uriutils import Matcher


matcher = Matcher()


class Resource(object):

    def __init__(self, client):
        self._client = client
        self._session = client.session
        self._url = client.path + '/v2/' + self._resource_
        self._value = None
        self._next = None

    @classmethod
    def register(cls, map):
        matcher.add('v2/' + cls._resource_, cls)

    def get(self, id):
        self._get(expand(self._url, {'id': id}))

    def _get(self, uri):
        r = self._session.get(uri)
        self._check_status(r)
        self._build(json.loads(r.content.decode('utf-8')))

    def get_all(self):
        return self._get_all(expand(self._url, {}))

    def _get_all(self, uri):
        r = self._session.get(uri)
        self._check_status(r)

        for element in map(self._init_build, json.loads(r.content)):
            yield element

        while self._next is not None:
            r = self._session.get(self._next['url'])
            self._check_status(r)

            for element in map(self._init_build, json.loads(r.content)):
                yield element

    def _init_build(self, object):
        value = self.__class__(self._client)
        value._build(object)
        return value

    def create(self, data):
        self._create(expand(self._url, {}), data)

    def _create(self, uri, data):
        r = self.post(uri, data)
        self._build(json.loads(r.content.decode('utf-8')))

    def _patch(self, uri, data):
        r = self._session.patch(uri, data=json.dumps(data))
        self._build(json.loads(r.content.decode('utf-8')))

    # TODO: remove?
    def post(self, url, data):
        r = self._session.post(url, data=json.dumps(data))
        self._check_status(r)
        return r

    def _check_status(self, r):
        if r.status_code == 404:
            raise NotFound(r)
        elif r.status_code >= 400:
            raise StartrekError(r)
        self._next = r.links.get('next')

    def _build(self, raw_object):
        obj = {}
        for field, value in raw_object.items():
            obj[field] = self._build_value(field, value)
        self._value = obj
        self.raw_value = raw_object
        return self

    def _build_value(self, field, value):
        if field == 'customFields':
            return value
        elif isinstance(value, list):
            return [self._build_value(field, element) for element in value]
        elif isinstance(value, dict) and 'self' in value:
            return Reference(field, value, self._client)
        else:
            return value

    def __repr__(self):
        return self.__str__()

    def __getattr__(self, name):

        if name.startswith('_'):
            return getattr(super(Resource, self), name)

        if name in self._value:
            return self._value[name]

        raise AttributeError('Object "%s" has no attribute %s' % (type(self), name))


class Reference(object):

    def __init__(self, field, value, client):
        self._client = client
        self.id = value['id']
        self.self = value['self']
        self.type = field.title()
        self.outward = value.get('outward')
        self.inward = value.get('inward')
        if 'display' in value:
            self.name = value['display']
        else:
            self.name = value['id']
        self._value = None

    def __repr__(self):
        return self.__str__()

    def __str__(self):
        if self._value is None:
            return '[%s:%s]' % (self.type.encode('utf-8'), self.name.encode('utf-8'))
        else:
            return self._value.__str__()

    def __getattr__(self, name):
        if self._value is None:
            resource_descriptor = matcher.match(
                self.self.replace(
                    self._client.path,
                    ''))
            if resource_descriptor is None:
                raise Exception('Unknown resource: ' + self.self)
            resource = resource_descriptor(self._client)
            resource._get(self.self)
            self._value = resource

        return getattr(self._value, name)


class IssueType(Resource):
    _resource_ = 'issuetypes/{id}'

    def __str__(self):
        return 'IssueType{{%s:%s}}' % (self.id, self.type.key)


class Priority(Resource):
    _resource_ = 'priorities/{id}'

    def __str__(self):
        return 'Priority{{%s:%s}}' % (self.id, self.type.key)


class User(Resource):
    _resource_ = 'users/{id}'

    def __str__(self):
        return 'User{{%s:%s}}' % (self.id, self.type.key)


class Group(Resource):
    _resource_ = 'groups/{id}'

    def __str__(self):
        return 'Group{{%s:%s}}' % (self.id, self.type.key)


class Status(Resource):
    _resource_ = 'statuses/{id}'

    def __str__(self):
        return 'Status{{%s:%s}}' % (self.id, self.type.key)


class Resolution(Resource):
    _resource_ = 'resolutions/{id}'

    def __str__(self):
        return 'Resolution{{%s:%s}}' % (self.id, self.type.key)


class Version(Resource):
    _resource_ = 'versions/{id}'

    def __str__(self):
        return 'Version{{%s:%s}}' % (self.id, self.type.key)


class Component(Resource):
    _resource_ = 'components/{id}'

    def __str__(self):
        return 'Component{{%s:%s}}' % (self.id, self.type.key)


class Issue(Resource):
    _resource_ = 'issues/{id}'

    def __str__(self):
        return (
            'Issue{{%s:%s:%s}}' % (
                self.key.encode('utf-8'),
                self.type.name.encode('utf-8'),
                self.summary.encode('utf-8'))
        )

    def get_all(self, query):
        self._query = query
        r = self._session.post(
            expand(self._url, {'id': '_search'}),
            data=json.dumps({'query': query}))
        self._check_status(r)

        for element in map(self._init_build, json.loads(r.content)):
            yield element

        while self._next is not None:
            r = self._session.post(
                self._next['url'],
                data=json.dumps({'query': self._query}))
            self._check_status(r)

            for element in map(self._init_build, json.loads(r.content)):
                yield element

    def __getattr__(self, name):
        if name == 'transitions':
            if 'transitions' in self._value:
                return self._value['transitions']
            else:
                return self._client.get_transitions(self.key)
        elif name == 'comments':
            if 'comments' in self._value:
                return self._value['comments']
            else:
                return self._client.get_comments(self.key)
        elif name == 'links':
            if 'links' in self._value:
                return self._value['links']
            else:
                return self._client.get_links(self.key)
        elif name == 'attachments':
            if 'attachments' in self._value:
                return self._value['attachments']
            else:
                return self._client.get_attachments(self.key)
        return super(Issue, self).__getattr__(name)

    def transition_to(self, transition, **kwargs):
        transition_data = {}
        if 'comment' in kwargs:
            transition_data['comment'] = kwargs['comment']
            kwargs.pop('comment', None)
        if kwargs:
            transition_data['update'] = kwargs
        self.post(expand(self._url, {"id":self.key}) +
            expand('/transitions/{transition}/_execute', {'transition': transition}),
            data=transition_data)

    def add_comment(self, text):
        return self._client.add_comment(self.key, text)

    def add_link(self, relationship, to):
        return self._client.add_link(self.key, relationship, to)


class Queue(Resource):
    _resource_ = 'queues/{id}?expand=team'

    def __str__(self):
        return 'Queue{{%s:%s}}' % (self.key.encode('utf-8'), self.name.encode('utf-8'))

    def get_team(self):
        return self._value.get('teamUsers', [])

    def get_lead(self):
        return self._value['lead'].id


class Transition(Resource):
    _resource_ = 'issues/{issue}/transitions/{id}'

    def __str__(self):
        return 'Transition{{%s:%s}}' % (self.id, self.to)

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue):
        return self._get_all(expand(self._url, {'issue': issue}))


class Comment(Resource):
    _resource_ = 'issues/{issue}/comments/{id}'

    def __str__(self):
        return 'Comment{{%s:%s}}' % (self.id, self.text.encode('utf-8'))

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue):
        return self._get_all(expand(self._url, {'issue': issue}))

    def create(self, issue, text, files=None):
        if files is None:
            self._create(
                expand(
                    self._url, {
                        'issue': issue}), data={
                    'text': text})
        else:
            files_dict = dict([(file, open(file, 'rb')) for file in files])
            r = self._session.post(expand(self._url, {'issue': issue}),
                                   headers={
                                       'Content-Type': None},
                                   files=files_dict,
                                   data=json.dumps({'text': text}))
            self._check_status(r)

    def update(self, text):
        if self._value is None:
            raise Exception('Comment is not loaded')
        self._patch(self.self, {'text': text})

    def update_one(self, issue, id, text):
        self._patch(
            expand(
                self._url, {
                    'issue': issue, 'id': id}), {
                'text': text})

    def delete(self):
        if self._value is None:
            raise Exception('Comment is not loaded')
        r = self._session.delete(self.self)
        self._check_status(r)

    def delete_one(self, issue, id):
        r = self._session.delete(expand(self._url, {'issue': issue, 'id': id}))
        self._check_status(r)


class IssueLink(Resource):
    _resource_ = 'issues/{issue}/links/{id}'

    def __str__(self):
        return (
            'IssueLink{{%s:%s}}' % (
                self.object.key,
                self.type[self.direction])
        )

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue):
        return self._get_all(expand(self._url, {'issue': issue}))

    def create(self, issue, relationship, to):
        self._create(
            expand(
                self._url, {
                    'issue': issue}), data={
                'relationship': relationship, 'issue': to})

    def delete(self):
        if self._value is None:
            raise Exception('IssueLink is not loaded')
        r = self._session.delete(self.self)
        self._check_status(r)

    def delete_one(self, issue, id):
        r = self._session.delete(expand(self._url, {'issue': issue, 'id': id}))
        self._check_status(r)


class Attachment(Resource):
    _resource_ = 'issues/{issue}/attachments/{id}'

    def __str__(self):
        return (
            'Attachment{{%s}}' % (
                self.name)
        )

    def get(self, issue, id):
        self._get(expand(self._url, {'issue': issue, 'id': id}))

    def get_all(self, issue):
        return self._get_all(expand(self._url, {'issue': issue}))

    def upload(self, issue, file):
        files = {'file': open(file, 'rb')}
        r = self._session.post(expand(self._url, {'issue': issue}),
                               headers={
                                   'Content-Type': None},
                               files=files)
        self._check_status(r)

    def download_to(self, path):
        r = self._session.get(self.content)
        self._check_status(r)
        with open(path + '/' + self.name, 'wb') as file:
            file.write(r.content)

    def delete(self):
        if self._value is None:
            raise Exception('Attachment is not loaded')
        r = self._session.delete(self.self)
        self._check_status(r)

    def delete_one(self, issue, id):
        r = self._session.delete(expand(self._url, {'issue': issue, 'id': id}))
        self._check_status(r)


class StartrekError(Exception):

    def __init__(self, r):
        self.code = r.status_code
        if self.code < 500:
            error = json.loads(r.content)
            if 'errors' in error:
                self.messages = error['errors']
            else:
                self.messages = error['errorMessages']
            if 'invocation-info' in error:
                self._host = error['invocation-info']['hostname']

    def __str__(self):
        return (
            'Error[%s:%s]' % (
                self.code,
                ''.join([message.encode('utf-8') for message in self.messages]))
        )


class NotFound(StartrekError):

    def __str__(self):
        return (
            'NotFound[%s]' % (
                ''.join([message.encode('utf-8') for message in self.messages]))
        )

Issue.register(map)
Queue.register(map)
IssueType.register(map)
Priority.register(map)
User.register(map)
Group.register(map)
Status.register(map)
Resolution.register(map)
Version.register(map)
Component.register(map)
Transition.register(map)
Attachment.register(map)
Comment.register(map)
IssueLink.register(map)
