# -*- coding: utf-8 -*-
import io as StringIO
from contextlib import closing
import sys
import os
import re
import traceback
import shutil
import functools
import inspect
from time import time, clock, mktime
import threading
from urllib.parse import urlencode
import logging
import random
import datetime
import warnings
import resource


import yenv
import blackbox
import lxml.builder
import lxml.etree as ET
from django.conf import settings
from django_yauth.util import get_real_ip

from at.common import assertions, dbswitch
from at.common import exceptions
from django.conf import settings
from at.common.objects import AuthInfo
from at.common.assertions import assert2str

_log = logging.getLogger(__name__)
trace_logger = logging.getLogger('at.trace')
utils_blackbox = None


def get_utils_blackbox():
    global utils_blackbox
    if utils_blackbox is None:
        utils_blackbox = blackbox.Blackbox(blackbox.BLACKBOX_URL, 0.3, retry_count=3)
    return utils_blackbox


def getNoneAuthInfo():
    return AuthInfo(uid=0, login="")


def getGodAuthInfo():
    return AuthInfo(uid=settings.GODMODE_UID, login='SYSTEM')


def isGodAuthInfo(ai):
    return ai.uid == settings.GODMODE_UID


def getAuthInfo(uid=None, login=None, any=None, request=None):
    from at.aux_ import models

    def inner(uid, login, any, request):
        if login is not None:
            login = force_unicode(login)
        if any is not None:
            try:
                uid = int(any)
            except:
                login = any
        assert uid or login is not None, 'user not specified'
        if login and uid: # иногда мы передаем оба параметра и не хотим никуда ходить
            return AuthInfo(uid=uid, login=login)
        if login is None:
            # by uid
            args = {'person_id': uid}
        else:
            # by login
            args = {'login': login}
        args['community_type'] = 'NONE'
        model = models.Person.get_by(**args)
        if model is not None:
            return AuthInfo(uid=model.person_id, login=force_str(model.login))
        else:
            try:
                user_ip = get_real_ip(request) if request else '127.0.0.1'
                _log.info('uid: %s, login: %s, user_ip: %s' % (uid, login, user_ip))
                if login is None:
                    # if not found, fields = {} => raises KeyError
                    login = get_utils_blackbox().userinfo(uid, user_ip, [blackbox.FIELD_LOGIN])['fields']['login']
                elif not uid:
                    # if not found, uid = '' => raises ValueError
                    uid = int(get_utils_blackbox().userinfo(login, user_ip, [], by_login = True)['uid'])
                return AuthInfo(uid=uid, login=login)
            except (KeyError, ValueError) as err:
                raise exceptions.NotFound('User not found (%s, %s)' % (uid, login))
    return inner(uid, login, any, request)


def uid2login(uid, lower=False):
    r = getAuthInfo(uid=uid).login
    return r.lower() if lower else r


get_login = uid2login


BASE_HOST = 'at.yandex-team.ru'
def ya_user_link(feed_info):
    if is_community_id(feed_info.feed_id):
        return 'http://clubs.%s/%s/' % (BASE_HOST, feed_info.master_url)
    else:
        return 'http://%s.%s/' % (feed_info.master_url, BASE_HOST)


#===========================

def remove_nonxml_chars(s):
    is_uni = type(s) == str
    if not is_uni:
        s = str(s, 'utf8')
    res = re.sub('[\u0000-\u0008\u000B\u000C\u000E-\u001F\uD800-\uDFFF\uFFFE\uFFFF]+', ' ', s)
    if not is_uni:
        res = res.encode('utf8')
    return res


def redirect(uri, args=None):
    root = aux()
    if uri:
        retpath = ET.SubElement(root, "retpath")
        retpath.attrib['retpage'] = uri
        if args:
            retpath.attrib['retparams'] = urlencode(args)
    return root


def Status(code, msg="", reason=None, extra_attribs=None):
    global _hostname
    attribs = extra_attribs if extra_attribs else {}
    attribs.update({'code': str(code), 'hostname': _hostname})
    status = ET.Element('status', attribs);
    if reason is not None:
        status.attrib['reason'] = reason
    status.text = force_unicode(msg)
    return ET.tostring(status)


def req2dict(req):
    if isinstance(req, dict):
        return req
    res = dict([(pv.Param, pv.Value) for pv in req.queryArgs])
    return res


def req2cookiedict(req):
    return dict([(pv.Param, pv.Value) for pv in req.cookies])


def req2xml(req):
    root = ET.Element("request")
    for x in req.items():
        elem = ET.SubElement(root, x[0])
        elem.text = x[1]

    return root


def aux(*internals):
    root = ET.Element("aux")
    for extra in internals:
        if isinstance(extra, str):
            element = ET.XML(extra)
        else:
            element = extra

        root.append(element)
    return root


def params2dict(params):
    return dict([(hasattr(p, 'Param') and (p.Param, p.Value) or p) for p in params])


# FIXME переехать на yutil.log_exception_to
def log_exception(func):
    @functools.wraps(func)
    def protected(*args, **kw):
        before = datetime.datetime.now()

        try:
            return func(*args, **kw)
        except (
            AssertionError,
            exceptions.NotFound,
            exceptions.AccessDenied,
            assertions.AssertionFailure
        ) as exc:
            params = ', '.join([str(i) for i in (list(args) + ['%s=%s' % (k, v) for k, v in list(kw.items())])])
            _log.warning("%s(%s): %s" % (func.__name__, params, exc))
            raise
        except Exception:
            ex_type, ex_inst, ex_tb = sys.exc_info()
            ex_tb = ex_tb.tb_next  # Для настоящих lineno и filename
            _log.exception(
                'Error %s:%s:%s\n %%s' %
                (ex_type, ex_tb.tb_frame.f_code.co_filename, ex_tb.tb_frame.f_lineno),
                ''.join(traceback.format_exception(ex_type, ex_inst, ex_tb))
            )
            raise
        finally:
            ex_type, ex_inst, ex_tb = sys.exc_info()
            passed = datetime.datetime.now() - before
            msg = '%s executed in %d.%06d'
            if ex_type:
                msg += ' with_exception %s' % ex_type
            trace_logger.debug(msg, func.__name__, passed.seconds, passed.microseconds)

    protected.__doc__ = func.__doc__
    protected.__name__ = func.__name__
    return protected


def str_value_of(obj):
    '''
    Prints internals of any object.
    '''
    if hasattr(obj, '__dict__'):
        return '%s(%s)' % (
        obj.__class__, ', '.join(['%s=%s' % (i, str_value_of(getattr(obj, i))) for i in obj.__dict__]))
    elif isinstance(obj, (list, tuple)):
        return '[%s]' % (', '.join([str_value_of(i) for i in obj]))
    else:
        return str(obj)

# FIXME переделать на yutil.call_log


def print_call(func):
    def decorator(*args, **kw):
        _log.debug(func.__name__)
        return func(*args, **kw)

    decorator.__doc__ = func.__doc__
    decorator.__name__ = func.__name__
    return decorator


log_call = print_call


def check_auth(ai_arg_num=1):
    def decorator_caller(func):
        @functools.wraps(func)
        def decorator(*args, **kw):
            if len(args) > 1 and args[ai_arg_num].uid == 0:
                return ET.ElementTree(ET.XML(Status("NotAuthorized")))
            return func(*args, **kw)

        return decorator

    return decorator_caller


def stopwatch(fun):
    from datetime import datetime, timedelta

    time_now = datetime.now

    @functools.wraps(fun)
    def new_fun(*args, **kw):
        thread_name = 'undefinedThreadName'
        job_id = 'undefinedUUID'
        start_time = time_now()
        start_clock = clock()
        start_resources = resource.getrusage(resource.RUSAGE_SELF)
        try:
            try:
                thread_name = threading.currentThread().getName()
                # job_id = uuid.uuid1()
                _log.debug('[%s.%s] %s started at %s' % (thread_name, job_id, fun.__name__, start_time))
            except Exception as msg:
                _log.error('Error in decorator: %s (with "[%s.%s] %s started at %s")' \
                                  % (msg, thread_name, job_id, fun.__name__, start_time))

            return fun(*args, **kw)
        finally:
            end_resources = resource.getrusage(resource.RUSAGE_SELF)
            end_clock = clock()
            end_time = time_now()
            cpu_time = end_clock - start_clock
            system_time_delta = end_resources.ru_stime - start_resources.ru_stime
            user_time_delta = end_resources.ru_utime - start_resources.ru_utime
            execution_time = end_time - start_time
            _log.debug('[%s.%s] %s ended at %s' % (thread_name, job_id, fun.__name__, end_time))
            logging_handler = _log.debug
            if execution_time > timedelta(0, 1, 0):
                logging_handler = _log.warning
            elif execution_time > timedelta(0, 0, 500000):
                logging_handler = _log.info
            logging_handler('[%s.%s] %s execution time=%s (cpu=%f, system=%f, user=%f)' %
                            (thread_name, job_id, fun.__name__, execution_time, cpu_time, system_time_delta,
                             user_time_delta))

    return new_fun


def format_id_list(list):
    if list:
        return "null, " + ", ".join(["%d" % x for x in list])
    return "null"


# FIXME унести в yutil.deco
def decorate(cls, wrap_fun, recursive=False):
    from copy import copy

    res_cls = copy(cls)
    if recursive:
        names = dir(res_cls)
    else:
        names = list(res_cls.__dict__.keys())
    for name in names:
        fun = getattr(res_cls, name)  # added
        if callable(fun) and name[0].isupper():
            new_fun = wrap_fun(fun)
            setattr(res_cls, name, new_fun)

    return res_cls


def error2xml(e, func):
    global _hostname
    err = ET.Element('error', {'hostname': _hostname})
    ET.SubElement(err, "traceback").text = traceback.format_exc()
    ET.SubElement(err, "dsc").text = str(e)
    ET.SubElement(err, "method").text = func.__name__
    ET.SubElement(err, "type").text = e.__class__.__name__
    try:
        assert2str(e, root=ET.SubElement(err, "exception"))
    except:
        _log.error(traceback.format_exc())

    return ET.ElementTree(err)


# FIXME объединить с yutil xml_wrap
def et2xml(func):
    global _hostname

    def method(*args, **kw):
        try:
            ret = func(*args, **kw)
            if hasattr(ret, 'write'):
                # assume it's ET element
                ret.getroot().attrib["hostname"] = _hostname
            return et_to_string(ret)
        except assertions.AssertionFailure as e:
            try:
                assert_str = assert2str(e)
                _log.debug('ASSERTION: %s', assert_str)
                return assert_str
            except Exception:
                _log.exception('Wrapping exception in xml')
                if settings.WRAP_EXCEPTIONS_IN_XML:
                    return et_to_string(error2xml(e, func))
                else:
                    raise
        except Exception as e:
            _log.exception('Wrapping exception in xml')
            if settings.WRAP_EXCEPTIONS_IN_XML:
                return et_to_string(error2xml(e, func))
            else:
                raise

    method.__doc__ = func.__doc__
    method.__name__ = func.__name__
    return method


def simple_serialize(val, root):
    if isinstance(val, dict):
        for k, v in val.items():
            elem = ET.SubElement(root, k)
            simple_serialize(v, elem)
    elif isinstance(val, (list, tuple)):
        for v in val:
            elem = ET.SubElement(root, 'item')
            simple_serialize(v, elem)
    elif isinstance(val, str):
        root.text = val
    elif isinstance(val, (int, float)):
        root.text = str(val)
    elif val is None:
        pass
    else:
        raise ValueError("can't serialize %s: '%s'" % (type(val), val))


def dict2etree(root_name='result'):
    def deco(func):
        def method(*args, **kw):
            root = ET.Element(root_name)
            simple_serialize(func(*args, **kw), root)
            return ET.ElementTree(root)

        method.__name__ = func.__name__
        method.__doc__ = func.__doc__
        return method

    return deco


def deprecated_ext(msg=''):
    def deprecated(func):
        '''This is a decorator which can be used to mark functions
        as deprecated. It will result in a warning being emitted
        when the function is used.'''

        @functools.wraps(func)
        def new_func(*args, **kwargs):
            warnings.warn_explicit(
                "Call to deprecated function %s, %s." % (func.__name__, msg),
                category=DeprecationWarning,
                filename=func.__code__.co_filename,
                lineno=func.__code__.co_firstlineno + 1
            )
            return func(*args, **kwargs)

        return new_func

    return deprecated


deprecated = deprecated_ext()


def is_community_id(feed_id):
    if feed_id:
        return bool(feed_id >= settings.COMMUNITY_START_ID)
    return False


def hostname():
    fp = None
    try:
        fp = os.popen('hostname')
        return fp.read().strip()
    finally:
        if fp:
            fp.close()
            del fp


_hostname = hostname()


class CursorProxy:
    def __init__(self, instance):
        self._instance = instance

    def execute(self, sql, params=None):
        if params is None:
            params = ()
        elif isinstance(params, dict):
            for key in sorted(list(params.keys()), key=lambda x: len(x), reverse=True):
                sql = sql.replace(':{}'.format(key), '%({})s '.format(key))
        elif not isinstance(params, (list, tuple)):
            params = (params,)
        if hasattr(settings, 'USE_SQLITE_IN_PY_TEST') and settings.USE_SQLITE_IN_PY_TEST:
            sql = sql % params
            params = ()
        self._instance.execute(sql, params)
        return self

    def keys(self):
        column_names = [col[0] for col in self._instance.description]
        return column_names

    def scalar(self, sql):
        result = self.execute(sql, ()).fetchone()
        return result[0] if result else None

    def __enter__(self):
        self._instance = self._instance.__enter__()
        return self

    def __exit__(self, type, value, tb):
        return self._instance.__exit__(type, value, tb)

    def __getattr__(self, item):
        return getattr(self._instance, item)

    def __iter__(self):
        return iter(self._instance)


def get_connection():
    from django.db import connections
    from django_replicated.utils import routers
    state = routers.state()
    if state == 'master':
        alias = routers.db_for_write()
    else:
        alias = routers.db_for_read()
    connection = connections[alias]
    return CursorProxy(connection.cursor())


class XPath:
    @staticmethod
    def evaluate(node, xpath):
        return node.xpath(xpath)


def dt2timestamp(dt):
    return int(mktime(dt.timetuple()))


def dt2isoformat(dt):
    return dt.isoformat()[:len('YYYY-MM-DD hh:mm:ss')]


def log_result(func):
    def deco(*args, **kwargs):
        result = func(*args, **kwargs)
        _log.debug(" func: %r return: %r" % (func.__name__, result))
        return result

    deco.__name__ = func.__name__
    deco.__doc__ = func.__doc__
    return deco


def force_unicode(s):
    if isinstance(s, bytes):
        return s.decode('utf-8')
    return str(s)


def force_str(s):
    if isinstance(s, str):
        return s.encode('utf-8')
    return bytes(s)


def bool2str(b):
    return "yes" if b else "no"


_email_re = re.compile(r'[\w-]+(\.[\w-]+)*@([\w-]+\.)+[a-z]{2,7}')


def validate_email_address(addr):
    return _email_re.match(addr)


class ParsedStatus(object):
    def __init__(self, text):
        self.raw = text
        status = ET.XML(text)

        if status.tag != 'status':
            raise ValueError('not a valid status xml: %s' % (text,))

        self.code = status.get('code')
        self.message = status.text
        self.success = self.code in ('OK', 'Success', 'success')

    def __str__(self):
        return '%s (%s)' % (self.code, self.message)

# Syntax sugar, verb looks better in the code.
parse_status = ParsedStatus


def atomical_file_write(path, content):
    tmp_path = path + '.tmp'
    with open(tmp_path, 'w') as out:
        out.write(content)
    shutil.move(tmp_path, path)


def build_xml(tree):
    """Build xml from description"""
    E = lxml.builder.ElementMaker()

    def inner_build_xml(tree):
        if len(tree) == 2:
            attrs = {}
            tag_name, content = tree
        else:
            tag_name, attrs, content = tree
        if isinstance(content, tuple):
            escape = False
            content = [inner_build_xml(content)]
        elif isinstance(content, list):
            escape = False
            content = [inner_build_xml(node) for node in content]
        else:
            escape = True
            content = force_unicode(content or '')
        if escape:
            elem = E(tag_name,
                     content,
                     **attrs)
        else:
            elem = E(tag_name, **attrs)
            for node in content:
                child = ET.fromstring(node)
                elem.append(child)
        return ET.tostring(elem, encoding='utf-8')

    return inner_build_xml(tree)


def batch(iterable, n = 1):
   l = len(iterable)
   for ndx in range(0, l, n):
       yield iterable[ndx:min(ndx+n, l)]


def usec(datetime_object=None):
    if datetime_object is None:
        return int(time() * 1000000)
    random_part = random.randint(0, 999999)
    return int(mktime(datetime_object.timetuple()) * 1000000) + random_part


def parse_date(value):
    try:
        dt = datetime.datetime.strptime(value, '%Y-%m-%d')
    except ValueError:
        dt = datetime.datetime.strptime(value, '%d.%m.%Y')
    return dt.date()


def sanitize_url(url):
    url = url.strip()
    assert url, "Can't sanitize empty url"
    allowed_schemes = ['ftp',
                       'gopher',
                       'http',
                       'https',
                       'mailto',
                       'news',
                       'xmpp',
                       'callto',
                       'cvs',
                       'irc',
                       'sftp',
                       'svn']
    if url.startswith("//"):
        assert url != "//", "Can't sanitize empty url"
        return url
    for schema in allowed_schemes:
        assert url != schema, "Can't sanitize empty url"
        if url.startswith("%s:" % schema):
            return url
    return "http://%s" % url


def wrap_in(tags_str):
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            tags = tags_str.split('/')
            prefix = ''.join(['<%s>' % tag for tag in tags])
            suffix = ''.join(['</%s>' % tag for tag in reversed(tags)])
            result = func(*args, **kwargs)
            return prefix + result + suffix
        return wrapper
    return decorator


def for_all_methods(decorator):
    def decorate(cls):
        for name, fn in inspect.getmembers(cls, inspect.ismethod):
            setattr(cls, name, decorator(fn))
        return cls
    return decorate


def timeit(func):
    @functools.wraps(func)
    def decorator(*args, **kwargs):
        before = datetime.datetime.now()
        result = func(*args, **kwargs)
        passed = datetime.datetime.now() - before
        msg = '%s executed in %d.%06d'
        trace_logger.debug(msg, func.__name__, passed.seconds, passed.microseconds)
        return result
    return decorator


def et_to_string(t):
    if hasattr(t, 'write'):
        buf = StringIO.BytesIO()
        t.write(buf, encoding='utf-8')

        return buf.getvalue()
    else:
        return t


def delta_in_ms(dt, as_str=True):
    seconds, microseconds = dt.seconds, dt.microseconds

    in_ms = (seconds * 10 ** 6 + microseconds) / 1000.0
    if as_str:
        return '%.3fms' % in_ms


def parse_isoformat(dt_str):
    import dateutil
    return dateutil.parser.parse(dt_str)
