# -*- coding: utf-8 -*-

import logging

from at.common import utils
from at.common import groups
from at.common import dbswitch
from at.common.utils import get_connection
from at.aux_ import entries
from at.aux_.models import Person
from at.aux_.Profile import get_status_change_job
from at.pump.Query import Handler, Event
from at.pump import HandlerRegistry


_log = logging.getLogger(__name__)


class ChangePostsAccessesHandler(Handler):
    accepts_new_style = True
    sleep_time = 3

    class Event(Event):
        accepts_data = True
        field_names = ['feed_id', 'old_accesses', 'new_access_type']

    def get_posts(self, person_id, accesses):
        with utils.get_connection() as connection:
            return connection.execute(
                """
                SELECT person_id, post_no
                FROM Posts
                WHERE person_id = %%s
                      AND access_group in (%s)
                """ % ",".join(accesses), (person_id, )).fetchall()

    def action(self, event):
        feed_id, old_accesses_str, new_access_str = event
        old_accesses_list = [str(groups.get_int_access(access)) for access in old_accesses_str.split(',')]

        if not set(old_accesses_list) - {new_access_str}:
            msg = "can't change access! invalid old_accesses = %r new_access=%s"
            _log.warning(msg, old_accesses_str, new_access_str)
            return event.get_human_repr() + ' invalid_access'

        for feed_id, post_no in self.get_posts(feed_id, old_accesses_list):

            HandlerRegistry.put_event(
                'ChangePostAccess',
                feed_id=feed_id,
                post_no=post_no,
                new_access_type=groups.get_int_access(new_access_str)
            )

        return True


class FeedStatusHandler(Handler):
    accepts_new_style = True


    class Event(Event):
        field_names = ['uid']

    def action(self, event):
        uid = event.fields['uid']
        job = get_status_change_job(uid)
        fi = Person.get_person(uid)
        if fi.status != self.valid_status:
            msg = "Invalid person(%d) status='%s' %s require '%s' status"
            _log.warning(
                msg,
                uid,
                fi.status,
                self.__class__.__name__,
                self.valid_status
            )
            return uid + ' invalid status'

        ret = self.action_impl(uid)
        if not job.is_active():
            _log.error("change_blog_status job is not active when job has actually been completed. person_id = %d" % (uid,))
        else:
            _log.warn("Stopping change_blog_status job. person_id = %d" % (uid,))
        job.stop()
        return ret


class DeleteFeedHandler(FeedStatusHandler):
    valid_status = 'deleted'

    def action_impl(self, uid):
        sql = """
        SELECT
            post_no
        FROM
            Posts
        WHERE
            person_id = %s AND deleted = 0
        """
        with get_connection() as conn:
            for row in conn.execute(sql, (uid, )):
                post = entries.models.load_entry(
                    feed_id=uid, item_no=row[0])
                post.mark_deleted(manually=False)
                post.save()
        return True


class UndeleteFeedHandler(FeedStatusHandler):
    # FIXME AT-1947
    valid_status = 'normal'
    def action_impl(self, uid):
        list_posts_sql = 'SELECT post_no FROM Posts WHERE person_id=%s AND deleted=2'
        queries = [
                'UPDATE %s.Posts SET deleted=0 '\
                    'WHERE person_id=%s AND post_no IN (%s)',
                'UPDATE %s.Comments SET deleted=0 '\
                    'WHERE person_id=%s AND post_no IN (%s) AND comment_id=0',
                'UPDATE %s.PostCategories SET deleted=0 '\
                    'WHERE feed_id=%s AND post_no IN (%s)'
                ]
        with get_connection() as conn:
            post_nos = ','.join(str(row[0]) for row in conn.execute(list_posts_sql, (uid, )))
            for db in ['f7', 'tmp']:
                for query in queries:
                    conn.execute(query % (db, uid, post_nos)).fetchone()
        return True


class DismissedFeedHandler (FeedStatusHandler):
    valid_status = 'dismissed'
    def action_impl(self, uid):
        pass


class ReadmittedFeedHandler (FeedStatusHandler):
    valid_status = 'normal'
    def action_impl(self, uid):
        pass


class ChangePostAccess(Handler):
    """Используется для изменения уровня доступа публичных постов клуба,
    если клуб переведен в закрытый режим"""
    accepts_new_style = True

    class Event(Event):
        field_names = ['feed_id', 'post_no', 'new_access_type']

    def action(self, event):
        feed_id = event.fields['feed_id']
        post_no = event.fields['post_no']
        access_type = event.fields['new_access_type']
        post = entries.load_entry(feed_id, post_no)
        post.access_group = access_type
        post.save()



import types
names = list(locals().keys())
__all__ = [name for name, obj in locals().items() if isinstance(obj, type) and issubclass(obj,
                                                 (Event, Handler))]
