# coding: utf-8

from collections import defaultdict
import datetime
import logging

from at.common import exceptions
from at.common import utils
from at.common.Context import ContextBinder
from at.aux_ import Mailing
from at.aux_.feeds.loaders import digest as digest_loaders
from at.aux_ import Friends
from at.aux_ import entries
from at.pump import Mailer
from at.pump import HandlerRegistry


_log = logging.getLogger(__name__)


def create_digest_events():
    digest_users_with_modes = Mailing.get_digest_users()
    counter = 0
    for counter, (uid, mode) in enumerate(digest_users_with_modes, start=1):
        HandlerRegistry.put_event(
            'MailDigestHandler',
            target=uid,
            mode=mode,
        )
    return str(counter)


class DigestSet(object):
    @classmethod
    def get_digest_set(cls, target, popular):
        if popular:
            return PopularDigestSet(target)
        else:
            return FriendsFeedDigestSet(target)

    def __init__(self, target, size=None):
        self.target = target
        self.ai = utils.getAuthInfo(uid=target, login='_')
        self.size = size
        self.loaded_posts = []
        self.lower_bound = utils.usec(
            datetime.datetime.now() - datetime.timedelta(hours=24)
        )
        self.load_posts()

    def load_posts(self):
        for post in self.loader().posts_data:
            feed_id, item_no, store_time, _ = post
            try:
                # проверка доступа, т.к. PostWrapper доступов не проверяет
                # TODO сделать PostWrapper сабклассом entries.model
                entries.load_entry(feed_id, item_no)

                # Each post should has feed_id, item_no, comment_id
                # in context for correct xsl rendering
                with ContextBinder(feed_id=feed_id,
                                   item_no=item_no,
                                   comment_id=0):
                    self.loaded_posts.append(DigestPostWrapper(feed_id,
                                                               item_no))
            except (exceptions.AccessDenied, exceptions.NotFound):
                continue
            except Exception:
                _log.error("Can't render post %s.%s for digest",
                           feed_id,
                           item_no,
                           exc_info=1)


class FriendsFeedDigestSet(DigestSet):
    mode = "old"

    def loader(self):
        return digest_loaders.DigestFriendsLoader(viewer_id=self.target, host_id=self.target, tb=self.lower_bound)

    def __init__(self, target, size=None):
        size = size or 200
        super(FriendsFeedDigestSet, self).__init__(target, size)

    def __iter__(self):
        return iter(self.loaded_posts)

    def __bool__(self):
        return bool(self.loaded_posts)


class PopularDigestSet(DigestSet):
    mode = "new"
    special_types = ['links']

    def loader(self):
        return digest_loaders.DigestPopularLoader(viewer_id=self.target, tb=self.lower_bound)

    def __init__(self, target, size=None):
        size = size or 20
        super(PopularDigestSet, self).__init__(target, size)

        for post_type in self.special_types:
            getattr(self, 'handle_%s' % post_type)()

    def handle_links(self):
        """Обработать все посты типа link в текущем сете"""
        posts = self.loaded_posts
        linked_posts = defaultdict(set)
        who_share = defaultdict(set)
        all_sharers = set()

        def only_post_links(posts):
            return (post for post in posts
                    if post.post_type == 'link' and post.link_type == 'post')

        for post in only_post_links(posts):
            linked_posts[post.shared_post].add(post)
            who_share[post.shared_post].add(post.post_author)
            all_sharers.add(post.post_author.uid)
        # Одним запросом отсортировать всех, кто делился постами в этом сете,
        # по дружбе к получателю
        all_sharers_sorted_by_friendship = [user for (user, _) in
                                            Friends.friendship_sort_uid_list(self.target,
                                                                             list(all_sharers))]
        posts = [post for post in posts
                 if post not in only_post_links(posts)]
        for post, links in list(linked_posts.items()):
            if post.post_author.uid == self.target:
                # Пропускать линки на свои посты
                continue
            if post.for_today:
                # Если пост сегодняшний, добавляю сам пост в дайджест
                if post not in posts:
                    posts.append(post)
            else:
                # Иначе самую популярную ссылку
                link_for_digest = sorted(links, key=lambda p: p.score, reverse=True)[0]
                # Чтобы у трекбеков был список поделившихся от оригинального поста
                who_share[link_for_digest] = who_share[post]
                if post not in posts:
                    posts.append(link_for_digest)
        for post in posts:
            # Добавляем к каждому посту список поделившихся пользователей,
            # отсортированный по дружественности к получателю
            post.shared_by = [user for user, _ in
                              sorted(((user, all_sharers_sorted_by_friendship.index(user.uid))
                                      for user in who_share[post]),
                                     key=lambda tup: tup[1])]
        self.loaded_posts = posts

    def __iter__(self):
        return iter(sorted(
            self.loaded_posts[:self.size],
            key=lambda post: post.store_time))

    def __bool__(self):
        return bool(self.loaded_posts[:self.size])


class DigestPostWrapper(Mailer.PostWrapper):
    def __init__(self, feed_id, item_no, comment_id=0):
        # Чтобы не перегружать PostWrapper вынес некоторые
        # дополнительные поля в отдельный класс
        super(DigestPostWrapper, self).__init__(feed_id, item_no, comment_id)
        self.shared_by = None

        if self.post.type == 'link':
            self.shared_post = DigestPostWrapper(self.post.shared_feed_id,
                                                 self.post.shared_item_no,
                                                 self.post.shared_comment_id)

        if self.post_type == 'link' and \
                self.shared_post and \
                        self.shared_post.comment_id == 0:
            self.link_type = 'post'
        elif self.post_type == 'link' and self.shared_post:
            self.link_type = 'comment'
        elif self.post_type == 'link':
            self.link_type = 'external'
        else:
            self.link_type = None

    @property
    def for_today(self):
        return self.post.store_time.date() == datetime.date.today()

    @property
    def key(self):
        return self.feed_id, self.item_no, self.comment_id

    def __hash__(self):
        return hash(self.key)

    def __eq__(self, other):
        return self.key == other.key
