# -*- coding: utf-8 -*-

import logging
import types

from at.common import exceptions
from at.aux_ import Friends
from at.common import dbswitch
from at.common import utils
from at.common import staff
from at.common import Recommendations
from at.pump.Query import Handler, Event
from at.aux_.models import Person


_log = logging.getLogger(__name__)


class AutofriendshipManager(Handler):
    """ Собирает информацию, с кем надо задружить новичка. """
    class Event(Event):
        field_names = ['uid']

    def action(self, event):
        uid, = event
        person = Person.objects.get(person_id=uid)
        if not person.has_access:
            return True
        api = staff.Api()
        person = api.persons.filter(uid=uid, official__is_dismissed=False)\
            .fields(['location.office.code', 'location.office.city.country.code', 'language.ui', 'official.is_robot'])
        if not person:
            return True
        person = person[0]
        if person['official']['is_robot']:
            return True # Do not autofriend robots
        friends = Recommendations.get_colleagues(uid, api=api)

        if Recommendations.speaks_russian(person, api=api):
            clubs = Recommendations.RECOMMENDED_CLUBS_RU + \
                    Recommendations.get_office_clubs(person['location']['office']['code']) \
                    + [info[0] for info in Recommendations.get_recommended_clubs(friends)]
        else:
            # Don't use dynamic recommendations for turkey,
            # because there's no english-writing clubs anyway
            clubs = Recommendations.RECOMMENDED_CLUBS_EN
        for feed_id in friends + clubs:
            from at.pump import HandlerRegistry
            HandlerRegistry.put_event(
                'AutofriendingHandler',
                feed_id=feed_id,
                uid=uid,
            )
        # Explicitly creating targets before using them
        # so that they get registered in SourceRegistered and retrieved in bulk
        friends_t = list(Person.get_persons(friends).values())
        clubs_t = list(Person.get_persons(clubs).values())
        login = utils.getAuthInfo(uid)
        _log.warn('New user %s got friends: %s' % \
                (login, ', '.join(f.login for f in friends_t)))
        _log.warn('New user %s got clubs: %s' % \
                (login, ', '.join(c.login for c in clubs_t)))



class AutofriendingHandler(Handler):
    """ Асинхронное установление связей при заведении дневника """
    class Event(Event):
        field_names = ['feed_id', 'uid']

    def action(self, event):
        feed_id, uid = event
        ai = utils.getAuthInfo(uid=uid, login='blah')
        with dbswitch.root_rw_session():
            try:
                if utils.is_community_id(feed_id):
                    Friends.joinCommunity(ai, ai.uid, feed_id)
                else:
                    Friends.friendAndFollow(ai, feed_id, ai.uid)
            except exceptions.NotFound:
                pass
            except Exception:
                _log.warning('Unexpected error in AutofriendingHandler', exc_info=1)



class UnfriendHandler(Handler):
    class Event(Event):
        field_names = ['feed_id', 'person_id']
    def action(self, event):
        feed_id, person_id = event
        ai = utils.getAuthInfo(person_id)
        with dbswitch.root_rw_session():
            if utils.is_community_id(feed_id):
                Friends.unjoinFromCommunity(ai, uid=person_id, feed_id=feed_id)
            else:
                Friends.unfriendAndUnfollow(ai, whom=feed_id, where=person_id)


names = list(locals().keys())
__all__ = [name for name, obj in locals().items() if isinstance(obj, type) and issubclass(obj,
                                                 (Event, Handler))]
