# coding: utf-8

import logging

import yenv

from at import celery_app
from django.conf import settings

logger = logging.getLogger(__name__)

registry = {}


class DuplicateRegistrationError(Exception):
    pass


def register(cls):
    if cls.__name__ in registry:
        raise DuplicateRegistrationError(cls.__name__)
    registry[cls.__name__] = cls


def put_event(name, *args, **kwargs):
    """
    Эта функция временная, чтобы не исправлять код,
    ставящий отложенные задачи
    """
    task_meta = PUMP_HANDLERS.get(name)
    if not task_meta:
        return
    celery_task_name = task_meta['celery_task']
    delay = task_meta.get('delay', 0)
    lifetime = task_meta.get('lifetime')

    task = get_celery_task(celery_task_name)
    task.apply_async(
        args=args,
        kwargs=kwargs,
        countdown=delay,
        expires=lifetime and lifetime * 60 * 60,
    )


def get_celery_task(short_name):
    full_name = 'at.pump.tasks.' + short_name
    if full_name not in celery_app.tasks:
        # это костыль, потому что я не смог сконфигурить celery по инструкции
        # и таски не автоимпортятся.
        celery_app.autodiscover_tasks(
            lambda: settings.INSTALLED_APPS, force=True)

    return celery_app.tasks[full_name]


PUMP_HANDLERS = {
    'AutofriendingHandler': {
        'lifetime': 1,
        'celery_task': 'autofriend',
    },
    'ChangePostsAccessesHandler': {
        'celery_task': 'change_post_access',
    },
    'ChangePostAccess': {
        'celery_task': 'change_post'
    },
    'CleanupCommentHistoryHandler': {
        'celery_task': 'cleanup_comment_history'
    },
    'CleanupMentionsHandler': {
        'celery_task': 'cleanup_mentions'
    },
    'ClubCheckHandler': {
        'celery_task': 'club_check'
    },
    'ClubNotifyHandler': {
        'celery_task': 'club_notify'
    },
    'CommunityStatHandler': {
        'celery_task': 'community_stat'
    },
    'DeletedCommentHandler': {
        'celery_task': 'handle_deleted_comment',
    },
    'DeleteFeedHandler': {
        'celery_task': 'handle_delete_feed',
    },
    'DeliverMentionsHandler': {
        'celery_task': 'deliver_mentions',
    },
    'DeliverCongratulationsHandler': {
        'celery_task': 'deliver_congratulations'
    },
    'DismissedFeedHandler': {
        'celery_task': 'handle_dismissed_feed'
    },
    'InsertRecentHandler': {
        'lifetime': 12,
        'celery_task': 'insert_recent',
    },
    'SummonFinalizeHandler': {
        'celery_task': 'handle_summon',
    },
    'EditHandler': {
        'celery_task': 'handle_edit',
    },
    'PublishHandler': {
        'celery_task': 'handle_publish',
    },
    'DeleteHandler': {
        'celery_task': 'handle_delete',
    },
    'ModerateHandler': {
        'celery_task': 'handle_moderate',
    },
    'PremoderateHandler': {
        'celery_task': 'handle_premoderate',
    },
    'CCFinalizeHandler': {
        'celery_task': 'handle_cc',
    },
    'ISearchHandler': {
        'celery_task': 'push_isearch',
    },
    'MailMentionHandler': {
        'celery_task': 'mail_mention',
        'delay': 300,
    },
    'MailDigestHandler': {
        'celery_task': 'mail_digest',
    },
    'ModifyCommentsCounterHandler': {
        'celery_task': 'modify_comments_counter',
    },
    'MailNewCommentHandler': {
        'delay': 60,
        'lifetime': 12,
        'celery_task': 'mail_new_comment',
    },
    'MailNewPostHandler': {
        'delay': 600,
        'lifetime': 12,
        'celery_task': 'mail_new_post',
    },
    'MailNewRelationHandler': {
        'lifetime': 12,
        'celery_task': 'mail_new_relation',
    },
    'MailPremoderatedPostHandler': {
        'delay': 300,
        'celery_task': 'mail_premoderated_post',
    },
    'ScorePostHandler': {
        'celery_task': 'score_post',
    },
    'ReadmittedFeedHandler': {
        'celery_task': 'handle_readmitted_feed',
    },
    'RecentCommentsHandler': {
        'lifetime': 12,
        'celery_task': 'handle_recent_comments',
    },
    'RemoveMentionNotificationsHandler': {
        'celery_task': 'remove_mention_notification',
    },
    'SynchronizeMongoFrienderHandler': {
        'celery_task': 'sync_mongo_friender'
    },
    'CleanupLikesHandler': {
        'celery_task': 'cleanup_likes',
    },
    'UndeleteFeedHandler': {
        'celery_task': 'undelete_feed',
    },
    'UnfriendHandler': {
        'lifetime': 1,
        'celery_task': 'handle_unfriend',
    },
    'UpdateUserEventsAfterBlogStatusChange': {
        'celery_task': 'handle_blog_status_change',
    },
    'LikeNotificationHandler': {
        'delay': 180,
        'celery_task': 'handle_like_notification',
    },
}

if settings.ENABLE_AUTOFRIEND:
    PUMP_HANDLERS.update({
        'AutofriendshipManager': {
            'celery_task': 'autofriendship_manage'
        }
    })

if yenv.type != 'production':
    for handler_opts in list(PUMP_HANDLERS.values()):
        if 'delay' in handler_opts:
            del handler_opts['delay']
